/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * LocalBinaryReader.java
 *
 * Created on August 21, 2007, 8:31 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package org.broad.igv.util;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

/**
 *
 * @author jrobinso
 */
public class LocalBinaryReader extends BinaryReader {
    
    /** Creates a new instance of LocalBinaryReader */
    public LocalBinaryReader() {
    }
    
    protected byte [] readBytes(File file) {
        try {
            int fileSize = (int) file.length();
            BufferedInputStream inStream = new BufferedInputStream(new FileInputStream(file), fileSize);
            
            byte [] bytes = super.readBytes(inStream, fileSize);
            inStream.close();
            return bytes;
        } catch (IOException ex) {
            ex.printStackTrace();
            throw new RuntimeException(ex);
        }
        
    }
    
    
    /**
     *  Read a selection of sampledBytes from the given file.
     *     start -  logical position in file to start reading.  The logical
     *              position is related to the file position as follows
     *                   file position = logical position X  readSize.
     *
     *     stop  -  logical position in file to stop (should be <= file length)
     *
     *     readSize - number of sampledBytes for a single read (4 sampledBytes for floats)
     *
     *     freq - how often to read.
     *
     *   Example -- to read every 5th float enter a readSize=4,  freq=5
     */
    public   byte[]  sampleBytes(File inputFile, int start, int stop, int readSize, int freq) throws IOException {
        
        
        // Read the entire region into a byte array.  In most cases this is faster than
        // reading individual floats and skipping on the file stream.
        InputStream  is = new BufferedInputStream( new FileInputStream(inputFile));
        byte [] allBytes = new byte[(stop - start) * readSize];
        is.skip(start * readSize);
        is.read(allBytes);
        is.close();
        
        // Compute the number of chunks to read, and the total number of bytes
        // required.  Create an array to hold the results.
        int numberOfReads = (int) ((stop - start) / freq);
        int numberOfBytes = numberOfReads * readSize;
        byte[] sampledBytes   = new byte[numberOfBytes];
        
        // Stream over the file contents buffer (allBytes) reading every "freq"
        // chunk into the results buffer (sampledBytes)
        ByteArrayInputStream dis = new ByteArrayInputStream(allBytes);
        int numRead = 0;
        int nSkip = (freq - 1) * readSize;
        while (numRead < sampledBytes.length) {
            numRead += dis.read(sampledBytes, numRead, readSize);
            dis.skip(nSkip);
        }
        return sampledBytes;
    }
    
    
    /**
     * Return all data files in the data directory. This includes location,
     * marker (snp) id, and copy number files for all chromosomes.
     *
     */
    public  List<File> getDataFiles(String dataFileDirectory) {
        File[] files = new File(dataFileDirectory).listFiles(new FilenameFilter() {
            public boolean accept(File dir, String name) {
                return name.endsWith("bin");
            }
        });
        return (files == null ? new ArrayList<File>() : java.util.Arrays.asList(files));
    }
    
    
    
    public static void main(String [] args) {
        String file = "data/solexa/chr6.MEF.K4.bin";
        long t0 = System.currentTimeMillis();
        BinaryReader reader = new LocalBinaryReader();
        reader.setLittleEndian(true);
        float [] floats = reader.readFloats(new File(file));
        System.out.println(floats.length + " floats read in " + (System.currentTimeMillis() - t0) + " ms");
    }
    
    
    
}
