/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.util;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.data.*;
import org.apache.log4j.Logger;


import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;

import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.broad.igv.feature.FeatureUtils;

/**
 *
 * @author jrobinso
 */
public class FeatureCountParser {

    static Logger log = Logger.getLogger(FeatureCountParser.class);
    
    
    
    public void convertFile(File inputFile,  File outputFile) {
        Collection<FeatureCount> counts = countFeatures(inputFile);
        outputCounts(this.segregateAndSortCounts(counts), outputFile);
    }

    /**
     * This method counts the number of occurrences of each distinct feature in the
     * file.   Two featuers are considered equal if they have the same loci (chr, start, stop).
     *
     * @param file
     *
     * @return  A collection of FeatureCount objects, one per each distinct feature
     */
     Collection<FeatureCount> countFeatures(File file) {

        Map<Integer, FeatureCount> counts = new HashMap();

        BufferedReader br = null;

        try
        {
            br = new BufferedReader(new FileReader(file));
            String nextLine = null;

            while ((nextLine = br.readLine()) != null)
            {
                String[] tokens = nextLine.split("\t");
                if (tokens.length > 2)
                {
                    FeatureCount c = new FeatureCount(tokens[0].trim(),
                        Integer.parseInt(tokens[1].trim()),
                        Integer.parseInt(tokens[2].trim()));

                    FeatureCount count = counts.get(c.getKey());
                    if (count == null)
                    {
                        counts.put(c.getKey(), c);
                    } else
                    {
                        count.incrementCount();
                    }
                }

            }
        } catch (IOException e)
        {
            log.error("Error parsing file: " + file, e);
        } finally
        {
            if (br != null)
            {
                try
                {
                    br.close();
                } catch (IOException e)
                {
                }
            }
        }

        return counts.values();
    }

    Map<String, List<FeatureCount>> segregateAndSortCounts(Collection<FeatureCount> counts) {
        Map<String, List<FeatureCount>> countMap = new LinkedHashMap();

        for (FeatureCount count : counts)
        {
            List<FeatureCount> countList = countMap.get(count.getChr());
            if (countList == null)
            {
                countList = new ArrayList();
                countMap.put(count.getChr(), countList);
            }
            countList.add(count);
        }

        for (List<FeatureCount> countList : countMap.values())
        {
            FeatureUtils.sortFeatureList(countList);
        }
        return countMap;
    }
    
    void outputCounts(Map<String, List<FeatureCount>> countMap, File outputFile) {
        
        PrintWriter pw = null;
        
        try {
            if(!outputFile.getName().endsWith(".igv")) {
                outputFile = new File(outputFile.getAbsolutePath() + ".igv");
            }
            pw = new PrintWriter(new FileWriter(outputFile));
            pw.println("chr\tstart\tend\tfeature\tcount");
            for(List<FeatureCount> counts : countMap.values()) {
                for(FeatureCount count : counts) {
                    pw.println(
                        count.chr + "\t" + count.getStart() + "\t" + count.getEnd() + "\t\t" + count.getScore());
                }
            }
        }
        catch(IOException e) {
            log.error("Error outputing counts", e);
        }
        finally {
            if(pw != null) {
                pw.close();
            }
        }
    }
    
    
}
