/*
 The Broad Institute
 SOFTWARE COPYRIGHT NOTICE AGREEMENT
 This software and its documentation are copyright (2003-2006) by the
 Broad Institute/Massachusetts Institute of Technology. All rights are
 reserved.

 This software is supplied without any warranty or guaranteed support
 whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 use, misuse, or functionality.
 */

package org.broad.igv.util;

import java.io.File;
import java.io.IOException;

/**
 * 
 * @author Joshua Gould
 * 
 */
public class BrowserLauncher {

    /**
     * The Java virtual machine that we are running on. Actually, in most cases we only care about the operating system,
     * but some operating systems require us to switch on the VM.
     */
    private static int jvm;

    /** JVM constant for any Windows JVM */
    private static final int WINDOWS = 1;

    /** JVM constant for any Mac JVM */
    private static final int MAC = 2;

    /** JVM constant for any other platform */
    private static final int OTHER = 3;

    /** specified the path to the browser */
    private static final int SPECIFIED = 4;

    /** The browser for the system */
    private static String specifiedBrowser;

    /**
     * An initialization block that determines the operating system.
     */
    static {
	String osName = System.getProperty("os.name");
	if (osName.startsWith("Mac OS")) {
	    jvm = MAC;
	} else if (osName.startsWith("Windows")) {
	    jvm = WINDOWS;
	} else {
	    jvm = OTHER;
	}
    }

    /**
     * This class should be never be instantiated; this just ensures so.
     */
    private BrowserLauncher() {
    }

    /**
     * Attempts to open the default web browser to the given URL.
     * 
     * @param url
     *                The URL to open
     * @throws IOException
     *                 If the web browser could not be located or does not run
     */
    public static void openURL(String url) throws IOException {
	switch (jvm) {
	case MAC:
	    Runtime.getRuntime().exec(new String[] { "/usr/bin/open", url });
	    break;

	case WINDOWS:
	    Process process = Runtime.getRuntime().exec("rundll32 url.dll,FileProtocolHandler " + url);
	    // This avoids a memory leak on some versions of Java on Windows.
	    // That's hinted at in
	    // <http://developer.java.sun.com/developer/qow/archive/68/>.
	    try {
		process.waitFor();
		process.exitValue();
	    } catch (InterruptedException ie) {
		throw new IOException("InterruptedException while launching browser: " + ie.getMessage());
	    }
	    break;
	case OTHER:
	    if (new File("/usr/bin/gnome-open").exists()) {
		Runtime.getRuntime().exec(new String[] { "/usr/bin/gnome-open", url });
	    } else if (new File("/usr/bin/kde-open").exists()) {
		Runtime.getRuntime().exec(new String[] { "/usr/bin/kde-open", url });
	    }

	    break;
	case SPECIFIED:
	    process = Runtime.getRuntime().exec(new String[] { specifiedBrowser, url });
	    try {
		process.waitFor();
		process.exitValue();
	    } catch (InterruptedException ie) {
		throw new IOException("InterruptedException while launching browser: " + ie.getMessage());
	    }
	    break;
	}
    }

    public static void setSpecifiedBrowser(String s) {
	specifiedBrowser = s;
	jvm = SPECIFIED;
    }

}