/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * BinaryParser.java
 *
 * Created on July 26, 2007, 1:01 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package org.broad.igv.util;


import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import org.j3d.io.EndianConverter;

/**
 *
 * @author jrobinso
 */
public abstract class BinaryReader {
    
    private boolean littleEndian = false;
    
    /** Creates a new instance of BinaryParser */
    public BinaryReader() {
    }
    
    public BinaryReader(boolean littleEndian) {
        this.setLittleEndian(littleEndian);
    }
    
    /**
     * Read an array of floats from a  binary file.   It is assumed that the
     * file contains floats written with DataOutput.writeFloat();
     */
    
    public float[] readFloats(File file) {
        return convertBytesToFloats(readBytes(file));
    }
    
    /**
     * Read an array of ints from the binary file.   It is assumed that the
     * file contains ints written with DataOutput.writeInt();
     */
    /**
     * Read an array of ints from the binary file.   It is assumed that the
     * file contains ints written with DataOutput.writeInt();
     */
    public int[] readInts(File file) {
        return ByteConverter.convertBytesToInts(readBytes(file));
    }
    
    
    /**
     * Read an array of strings from the binary file.   It is assumed that the
     * file contains fixed length ascii arrays of nChars characters each.
     */
    public String[] readStrings(File file, int nChars) {
        return ByteConverter.convertBytesToStrings(readBytes(file), nChars);
    }
    /**
     * Read an array of longs from the binary file.   It is assumed that the
     * file contains longs written with DataOutput.writeInt();
     */
    public long[] readLongs(File file) {
        return ByteConverter.convertBytesToLongs(readBytes(file));
    }
    
    
    /**
     *  Read a selection of floats from the given file.
     *     start -  logical position in file to start reading.  The logical
     *              position is related to the file position as follows
     *                   file position = logical position X  4.
     *
     *     stop  -  logical position in file to stop (should be <= file length)
     *
     *     freq - how often to read.
     *
     *   Example -- to read every 5th float enter a readSize=4,  freq=5
     */
    public   float[]  sampleFloats(File inputFile, int start, int stop, int freq) throws IOException {
        byte[] bytes = sampleBytes(inputFile, start, stop, (Float.SIZE / 8), freq);
        return convertBytesToFloats(bytes);
    }
    
    /**
     *  Read a selection of longs from the given file.
     *     start -  logical position in file to start reading.  The logical
     *              position is related to the file position as follows
     *                   file position = logical position X  4.
     *
     *     stop  -  logical position in file to stop (should be <= file length)
     *
     *     freq - how often to read.
     *
     *   Example -- to read every 5th float enter a readSize=4,  freq=5
     */
    public   long[]  sampleLongs(File inputFile, int start, int stop, int freq) throws IOException {
        byte[] bytes = sampleBytes(inputFile, start, stop, (Long.SIZE / 8), freq);
        return ByteConverter.convertBytesToLongs(bytes);
    }
    
    
    /**
     *  Read a selection of longs from the given file.
     *     start -  logical position in file to start reading.  The logical
     *              position is related to the file position as follows
     *                   file position = logical position X  4.
     *
     *     stop  -  logical position in file to stop (should be <= file length)
     *
     *     freq - how often to read.
     *
     *   Example -- to read every 5th float enter a readSize=4,  freq=5
     */
    public   String[]  sampleStrings(File inputFile, int start, int stop, int freq, int nChars) throws IOException {
        
        int stringSizeInBytes = nChars * (Character.SIZE / 8);
        byte[] bytes = sampleBytes(inputFile, start, stop, stringSizeInBytes, freq);
        return ByteConverter.convertBytesToStrings(bytes, nChars);
    }
    
    
    /**
     *  Read a selection of sampledBytes from the given file.
     *     start -  logical position in file to start reading.  The logical
     *              position is related to the file position as follows
     *                   file position = logical position X  readSize.
     *
     *     stop  -  logical position in file to stop (should be <= file length)
     *
     *     readSize - number of sampledBytes for a single read (4 sampledBytes for floats)
     *
     *     freq - how often to read.
     *
     *   Example -- to read every 5th float enter a readSize=4,  freq=5
     */
    protected abstract byte[]  sampleBytes(File inputFile, int start, int stop, int readSize, int freq) throws IOException;
    
    /**
     * Read all bytes in the specified file.  Implementation is the responsibility
     * of subclasses.
     */
    protected abstract byte [] readBytes(File file);
    
    public abstract List<File> getDataFiles(String dataDirectory);
    
    /**
     * Read the number of bytes specificed from the input stream
     */
    protected  byte[] readBytes(InputStream inStream, int nBytes) throws IOException {
        byte [] bytes = new byte[nBytes];
        int bytesRead = 0;
        while (bytesRead < nBytes) {
            bytesRead+=inStream.read(bytes, bytesRead, inStream.available());
        }
        return bytes;
    }
    
    /**
     * Convert bytes to floats.  Handles endian conversion
     */
    private float [] convertBytesToFloats(byte [] bytes) {
        if(littleEndian) {
            float [] floats = new float[bytes.length / (Float.SIZE / 8)];
            EndianConverter.convertLittleEndianToFloat(bytes, floats, bytes.length, 0, floats.length);
            return floats;
        } else {
            return ByteConverter.convertBytesToFloats(bytes);
        }
    }

    public void setLittleEndian(boolean littleEndian) {
        this.littleEndian = littleEndian;
    }
    
}
