/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/**
 * GenericUtilities.java
 *
 * Created on November 29, 2007, 2:14 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.broad.igv.ui.util;

import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.*;
import java.util.LinkedHashMap;
import javax.imageio.ImageIO;
import javax.swing.filechooser.FileFilter;

import org.apache.log4j.Logger;
import org.broad.igv.IGVConstants;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.jibble.epsgraphics.EpsGraphics2D;
import org.jibble.epsgraphics.EpsDocument;

/**
 *
 * @author eflakes
 */
public class GenericUtilities {

    /**
     * Class logger
     */
    private static Logger logger = Logger.getLogger(GenericUtilities.class);
    final private static LinkedHashMap<SnapshotFileType, SnapshotFileFilter> SNAPSHOT_TYPE_TO_FILTER = new LinkedHashMap();


    static {

        SNAPSHOT_TYPE_TO_FILTER.put(SnapshotFileType.JPEG,
                new SnapshotFileFilter(SnapshotFileType.JPEG));
        //SNAPSHOT_TYPE_TO_FILTER.put(SnapshotFileType.PDF,
        //    new SnapshotFileFilter(SnapshotFileType.PDF));
        SNAPSHOT_TYPE_TO_FILTER.put(SnapshotFileType.EPS,
                new SnapshotFileFilter(SnapshotFileType.EPS));
        //SNAPSHOT_TYPE_TO_FILTER.put(SnapshotFileType.SVG,
        //    new SnapshotFileFilter(SnapshotFileType.SVG));
        SNAPSHOT_TYPE_TO_FILTER.put(SnapshotFileType.PNG,
                new SnapshotFileFilter(SnapshotFileType.PNG));
    }

    public static FileFilter[] getAllSnapshotFileFilters() {
        return SNAPSHOT_TYPE_TO_FILTER.values().toArray(
                new FileFilter[SNAPSHOT_TYPE_TO_FILTER.size()]);
    }

    public static SnapshotFileFilter getSnapshotFileFilterForType(SnapshotFileType type) {
        return SNAPSHOT_TYPE_TO_FILTER.get(type);
    }

    /**
     * Creates a device compatible buffered image.
     *
     * @param width the image width in pixels
     * @param weight the image height in pixels
     */
    public static BufferedImage getDeviceCompatibleImage(int width, int height) {

        GraphicsEnvironment graphicsEnvironment =
                GraphicsEnvironment.getLocalGraphicsEnvironment();
        GraphicsDevice screenDevice =
                graphicsEnvironment.getDefaultScreenDevice();
        GraphicsConfiguration graphicConfiguration =
                screenDevice.getDefaultConfiguration();
        BufferedImage image =
                graphicConfiguration.createCompatibleImage(width, height);

        return image;
    }

    /**
     * Snapshot types
     */
    public static enum SnapshotFileType {

        NULL("", ""),
        EPS(".eps", "Encapsulated Postscript Files (*.eps)"),
        PDF(".pdf", "Portable Document FormatFles (*.pdf)"),
        SVG(".svg", "Scalable Vector Graphics Files (*.svg)"),
        PNG(".png", "Portable Network Graphics Files (*.png)"),
        JPEG(".jpeg", "Joint Photographic Experts Group Files (*.jpeg)");
        private String fileExtension;
        private String fileDescription;

        SnapshotFileType(String extension, String description) {
            fileExtension = extension;
            fileDescription = description;
        }

        public String getExtension() {
            return fileExtension;
        }

        public String getDescription() {
            return fileDescription;
        }
    }

    public static String getFileExtension(String filePath) {

        String extension = null;

        int indexOfExtension = filePath.lastIndexOf(".");
        if (indexOfExtension >= 0) {
            extension = filePath.substring(indexOfExtension, filePath.length());
        }
        return extension;
    }

    public static SnapshotFileType getSnapshotFileType(String fileExtension) {

        String extension = fileExtension.toLowerCase();
        SnapshotFileType type = null;

        if (SnapshotFileType.EPS.getExtension().equals(extension)) {
            type = SnapshotFileType.EPS;
        } else if (SnapshotFileType.PDF.getExtension().equals(extension)) {
            type = SnapshotFileType.PDF;
        } else if (SnapshotFileType.SVG.getExtension().equals(extension)) {
            type = SnapshotFileType.SVG;
        } else if (SnapshotFileType.PNG.getExtension().equals(extension)) {
            type = SnapshotFileType.PNG;
        } else if (SnapshotFileType.JPEG.getExtension().equals(extension)) {
            type = SnapshotFileType.JPEG;
        } else {
            type = SnapshotFileType.NULL;
        }
        return type;
    }

    public static boolean isValidSnapshotFileType(SnapshotFileType type) {

        boolean isValid = false;

        if (SnapshotFileType.EPS.equals(type) ||
                SnapshotFileType.PDF.equals(type) ||
                SnapshotFileType.SVG.equals(type) ||
                SnapshotFileType.PNG.equals(type) ||
                SnapshotFileType.JPEG.equals(type)) {
            isValid = true;
        }
        return isValid;
    }

    /**
     * Snapshot file filter
     */
    public static class SnapshotFileFilter extends FileFilter {

        private SnapshotFileType type = SnapshotFileType.EPS;

        public SnapshotFileFilter(SnapshotFileType type) {
            this.type = type;
        }

        public boolean accept(File file) {

            if (file.isDirectory()) {
                return true;
            }

            return file.getName().toLowerCase().endsWith(type.getExtension());
        }

        public String getDescription() {
            return type.getDescription();
        }

        public String getExtension() {
            return type.getExtension();
        }
    }

    /**
     * Genome Archive file filter
     */
    public static class GenomeArchiveFileFilter extends FileFilter {

        public GenomeArchiveFileFilter() {
        }

        public boolean accept(File file) {

            if (file.isDirectory()) {
                return true;
            }

            return file.getName().toLowerCase().endsWith(
                    IGVConstants.GENOME_FILE_EXTENSION);
        }

        public String getDescription() {
            return "Genome Archive File";
        }

        public String getExtension() {
            return IGVConstants.GENOME_FILE_EXTENSION;
        }
    }

    public static void doComponentSnapshot(Component component, File file) {
        doComponentSnapshot(component, file, SnapshotFileType.PDF);
    }

    public static void doComponentSnapshot(Component component, File file,
            SnapshotFileType type) {


        int width = component.getWidth();
        int height = component.getHeight();

        // Call appropriate converter
        if (type.equals(SnapshotFileType.JPEG)) {
            //GenericUtilities.generateJpegUsingJava2D(component, file, width,
            //                                         height);
            exportScreenShotJPEG(component, file, width, height);
        } else if (type.equals(SnapshotFileType.EPS)) {
            //GenericUtilities.generateEpsUsingJava2D(component, file, width,
            //                                        height);
            exportScreenShotEPS(component, file, width, height);
        } else if (type.equals(SnapshotFileType.PNG)) {
            //GenericUtilities.generatePngUsingJava2D(component, file, width,
            //                                        height);
            exportScreenShotPNG(component, file, width, height);
        }

    }

    private static void exportScreenShotJPEG(Component target, File selectedFile,
            int width, int height) {

        BufferedImage image =
                new BufferedImage(width, height, BufferedImage.TYPE_BYTE_INDEXED);
        Graphics g = image.createGraphics();
        target.paintAll(g);

        if (selectedFile != null) {

            if (!selectedFile.getName().toLowerCase().endsWith(".jpeg")) {
                String correctedFilename = selectedFile.getAbsolutePath() + ".jpeg";
                selectedFile = new File(correctedFilename);
            }
            writeImage(image, selectedFile, "jpeg");
        }
    }

    private static void exportScreenShotEPS(Component target, File selectedFile,
            int pageWidth, int pageHeight) {

        Graphics2D epsG = new EpsGraphics2D();

        // TODO Figure this out. This line is a hack to get the entire
        // image to draw unclipped - without it image is clipped even
        // when clipping is off
        epsG.drawString("", pageWidth, pageHeight);

        // Clip to document size
        epsG.clipRect(0, 0, pageWidth, pageHeight);

        target.paint(epsG);

        PrintWriter writer = null;

        try {
            writer = new PrintWriter(new BufferedWriter(
                    new FileWriter(selectedFile.getAbsolutePath())));
            writer.println(epsG.toString());
        } catch (IOException io) {
            logger.error(("Error creating: " + selectedFile.getAbsolutePath()), io);
        } finally {
            if (writer != null) {
                writer.close();
            }
        }
    }

    private static void exportScreenShotPNG(Component target, File selectedFile,
            int width, int height) {

        BufferedImage image =
                new BufferedImage(width, height, BufferedImage.TYPE_BYTE_INDEXED);
        Graphics g = image.createGraphics();
        target.paintAll(g);

        if (selectedFile != null) {

            if (!selectedFile.getName().toLowerCase().endsWith(".png")) {
                String correctedFilename = selectedFile.getAbsolutePath() + ".png";
                selectedFile = new File(correctedFilename);
            }
            writeImage(image, selectedFile, "png");
        }
    }

    private static void writeImage(BufferedImage image, File f, String type) {
        try {
            ImageIO.write(image, type, f);
        } catch (IOException e) {
            logger.error(("Error creating: " + f.getAbsolutePath()), e);
        }
    }
}
