/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.broad.igv.ui.util;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridLayout;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import org.broad.igv.util.Utilities;

/**
 *
 * @author eflakes
 */
public class CheckList extends JPanel {

    private boolean editing = false;
    private boolean isDirty = false;
    private HashSet<String> itemSet = new HashSet<String>();
    private boolean defaultState = true;
    private JLabel header = null;
    private JPanel itemPanel = new JPanel();
    
    // Backup of state
    private Component[] componentState = new Component[0];
    
    public CheckList(boolean defaultState) {
        this(defaultState, null);
    }
    
    public CheckList(boolean defaultState, String headerText) {
        
        setLayout(new BorderLayout());        
        this.defaultState = defaultState;
        itemPanel.setLayout(new GridLayout(0,1));
        
        if(headerText != null) {
            header = new JLabel(headerText);
            add(header, BorderLayout.NORTH);
            add(itemPanel, BorderLayout.CENTER);
        }
        else {
            add(itemPanel);
        }
    }
    
    public void addItems(List<String> itemStrings) {

        for(String itemText : itemStrings) {
            addItem(itemText, defaultState);
        }
        isDirty = true;
    }
    
    public void addItems(List<String> itemStrings, boolean isChecked) {

        for(String itemText : itemStrings) {
            addItem(itemText, isChecked);
        }
        isDirty = true;
    }
    
    public void addItem(String itemText, Boolean isChecked) {
        
        if(itemSet.contains(itemText.trim())) {
            return;
        }
        
        JCheckBox checkBox = new JCheckBox();
        if(itemText != null) {
            checkBox.setText(itemText);
        }
        if(isChecked == null) {
            checkBox.setSelected(defaultState);
        }
        else {
            checkBox.setSelected(isChecked);
        }
        itemPanel.add(checkBox);
        itemSet.add(itemText.trim());
        isDirty = true;
    }
    
    public void clear() {
        itemPanel.removeAll();
        itemSet.clear();
        isDirty = true;
    }

    /**
     * Make sure the current state of the checkboxes are used. Updating
     * is not allowed while in edit mode.
     */
    public void update() {
        
        // Can't update while in the middle of editing
        if(isEditing()) {
            return;
        }
        
        Component[] components = getItemComponents();
        int length = components.length;
        componentState = new Component[length];
        int i = 0;
        
        for(Component component : components) {
            
            boolean isChecked = ((JCheckBox)component).isSelected();
            String text  = ((JCheckBox)component).getText();
            
            JCheckBox checkBox = new JCheckBox();
            checkBox.setSelected(isChecked);
            checkBox.setText(text);
            componentState[i++] = checkBox;
        }
    }

    public boolean isEditing() {
        return editing;
    }

    public void setEditing(boolean editing) {
        this.editing = editing;
    }
    
    public HashSet<String> getSelectedItems() {
        
        // If there has been a modification we need to update first
        if(isDirty) {
            update();
        }
        
        HashSet<String> set = new LinkedHashSet<String>();
        
        for(Component component : componentState) {
            
            if(component instanceof JCheckBox) {
                JCheckBox checkBox = ((JCheckBox)component);
                
                if(!checkBox.isSelected())
                    continue;
                
                String text = checkBox.getText().trim();
                if(text != null && !text.equals("")) {
                    set.add(text);
                }
            }
        }
        return set;
    }

    public HashSet<String> getUnselectedItems() {

        // If there has been a modification we need to update first
        if(isDirty) {
            update();
        }
        
        HashSet<String> set = new LinkedHashSet<String>();
        
        for(Component component : componentState) {
            
            if(component instanceof JCheckBox) {
                JCheckBox checkBox = ((JCheckBox)component);
                
                if(checkBox.isSelected())
                    continue;
                
                String text = checkBox.getText().trim();
                if(text != null && !text.equals("")) {
                    set.add(text);
                }
            }
        }
        return set;
    }
    
    protected Component[] getItemComponents() {
        return itemPanel.getComponents();
    }   
    
    public void sort() {

        List<String> temp = new ArrayList<String>();
        for(String item : itemSet) {
            temp.add(item.toLowerCase());
        }
        
        // Do the sort
        Collections.sort(temp, Utilities.getNumericStringComparator());
        
        Component[] checkBoxes = itemPanel.getComponents();
        itemPanel.removeAll();
        for (int i = 0; i < temp.size(); i++) {
            for (Component checkBox : checkBoxes) {
                String name = ((JCheckBox) checkBox).getText();
                if (name.equalsIgnoreCase(temp.get(i))) {
                    itemPanel.add(checkBox);
                    break;
                }
            }
        }
        isDirty = true;
    }
    
    public void cancelChanges() {

        // Can't cancel changes if not editing
        if(!isEditing()) {
            return;
        }
        
        // Remove the changes
        itemPanel.removeAll();

        // Put the previous state back
        for(Component item : componentState) {
            itemPanel.add(item);            
        }
    }
}
