/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ui.panel;

//~--- non-JDK imports --------------------------------------------------------
import java.awt.event.MouseEvent;
import org.broad.igv.ui.*;


import org.broad.igv.IGVConstants;
import org.broad.igv.track.*;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Dimension;
import java.awt.Rectangle;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;


import javax.swing.JPanel;
import org.apache.log4j.Logger;

/**
 *
 * @author eflakes
 */
abstract public class TrackSetComponent extends JPanel {

    private static Logger log = Logger.getLogger(TrackSetComponent.class);
    final static public String MULTILEVEL_FEATURE_EVENT = "MULTILEVEL_FEATURE_EVENT";
    private boolean trackSelectionEnabled = true;
    List<MouseableRegion> mouseableRegions;
    boolean isDragging = false;
    int dragX, dragY;
    Track dropTarget = null;
    TrackSetView trackSetView;

    /**
     * Constructs ...
     *
     */
    public TrackSetComponent() {
        initialize();
        setFocusable(true);
        mouseableRegions = new ArrayList();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public TrackSetView getTrackSetView() {
        if (trackSetView == null) {
            trackSetView = (TrackSetView) getParent();
        }
        return trackSetView;
    }

    public String getTrackSetID() {
        return getTrackSetView().getName();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int calculateResizeHeight() {

        // Get the TrackSetView that holds the track name,
        // attribute and data panels
        TrackSetView trackView = getTrackSetView();

        // No size if no View yet
        if (trackView == null) {
            return 0;
        }

        // Get the viewport height
        int viewportHeight = trackView.getViewportHeight();

        // Calculate new height
        if (trackView.hasTracks()) {

            // The sum of all the track heights
            int preferredTrackViewHeight = trackView.getPreferredPanelHeight();

            // Make sure the preferred height is never less than
            // the height of the actual viewport view
            preferredTrackViewHeight = (preferredTrackViewHeight < viewportHeight)
                    ? viewportHeight : preferredTrackViewHeight;

            // Resize all track set view's components
            return preferredTrackViewHeight;

        } else {

            // Resize parent container
            return viewportHeight;
        }
    }

    /**
     * Method description
     *
     *
     * @param x
     * @param y
     * @param width
     * @param height
     */
    @Override
    public void setBounds(int x, int y, int width, int height) {

        int adjustedHeight = calculateResizeHeight();
        super.setBounds(x, y, width, adjustedHeight);
    }

    /**
     * Method description
     *
     *
     * @param r
     */
    @Override
    public void setBounds(Rectangle r) {

        int adjustedHeight = calculateResizeHeight();
        super.setBounds(r.x, r.y, r.width, adjustedHeight);
    }

    /**
     * Method description
     *
     *
     * @param width
     * @param height
     */
    @Override
    public void setSize(int width, int height) {

        int adjustedHeight = calculateResizeHeight();
        super.setSize(width, adjustedHeight);
    }

    /**
     * Method description
     *
     *
     * @param d
     */
    @Override
    public void setSize(Dimension d) {

        int adjustedHeight = calculateResizeHeight();
        super.setSize(new Dimension(d.width, adjustedHeight));
    }

    protected void addMousableRegion(MouseableRegion region) {
        mouseableRegions.add(region);
    }

    protected void removeMousableRegions() {
        mouseableRegions.clear();
    }

    protected List<MouseableRegion> getMouseableRegions() {
        return mouseableRegions;
    }

    public boolean scrollTo(String trackName) {
        Track t = findNextTrackMatching(trackName);
        if (t != null) {
            TrackManager.getInstance().clearTrackSelections();
            t.setSelected(true);
            if (trackSetView.getScrollPane().getVerticalScrollBar().isShowing()) {
                trackSetView.getScrollPane().getVerticalScrollBar().setValue(t.getTop());
            }
            return true;
        }

        return false;
    }
    int searchIdx = 0;

    private synchronized Track findNextTrackMatching(String trackName) {
        List<Track> tracks = getAllTracks();
        searchIdx = Math.min(searchIdx, tracks.size());
        for (int i = searchIdx; i < tracks.size(); i++) {
            Track t = tracks.get(i);
            if (t.getDisplayName().toUpperCase().contains(trackName)) {
                searchIdx = i + 1;
                return t;
            }
        }
        for (int i = 0; i < searchIdx; i++) {
            Track t = tracks.get(i);
            if (t.getDisplayName().toUpperCase().contains(trackName)) {
                searchIdx = i + 1;
                return t;
            }
        }
        return null;
    }

    /**
     * Method description
     *
     *
     * @param x
     * @param y
     *
     * @return
     */
    abstract public String getPopupMenuTitle(int x, int y);

    protected Collection<Track> getSelectedTracks() {
        if (log.isDebugEnabled()) {
            log.debug("Enter getSelectedTracks");
        }
        return TrackManager.getInstance().getSelectedTracks();
    }

    public List<Track> getAllTracks() {
        TrackSetView dataTrackView = (TrackSetView) getParent();
        return dataTrackView.getAllTracks();
    }

    boolean isZoomEnabled() {
        return !getViewContext().getChrName().equals(IGVConstants.CHR_ALL);
    }

    /**
     * Initial Setup for the component is below
     */
    private void initialize() {

        //MouseInputAdapter mouseAdapter = new TrackPanelAdapter(this);

        //addMouseMotionListener(mouseAdapter);
        //addMouseListener(mouseAdapter);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isTrackSelectionEnabled() {
        return trackSelectionEnabled;
    }

    /**
     * Method description
     *
     *
     * @param trackSelectionEnabled
     */
    public void setTrackSelectionEnabled(boolean trackSelectionEnabled) {
        this.trackSelectionEnabled = trackSelectionEnabled;
    }

    /**
     * //TODO reimplement this without referencing IGVMainFrame.getInstance().
     * @return
     */
    public ViewContext getViewContext() {
        return IGVModel.getInstance().getViewContext();
    }

    protected void openPopupMenu(final MouseEvent e) {

        if (log.isDebugEnabled()) {
            log.debug("Enter openPopupMenu");
        }

        // If there is a single selected track give it a chance to handle the click
        Collection<Track> selectedTracs = getSelectedTracks();
        if (selectedTracs.size() == 0) {
            return;
        }

        if (log.isDebugEnabled()) {
            log.debug("calling handeClick");
        }

        if (selectedTracs.size() == 1) {
            if (selectedTracs.iterator().next().handleClick(e)) {
                return;
            }
        }
        String title = getPopupMenuTitle(e.getX(), e.getY());

        TrackMenuUtils.getPopupMenu(selectedTracs, e.getX(), e.getY(), title).show(e.getComponent(),
                e.getX(), e.getY());
    }

    protected void toggleTrackSelections(MouseEvent e) {
        for (MouseableRegion mouseRegion : mouseableRegions) {
            if (mouseRegion.containsPoint(e.getX(), e.getY())) {
                TrackManager.getInstance().toggleTrackSelections(mouseRegion.getTracks());
                return;
            }
        }
    }

    protected void clearTrackSelections() {
        TrackManager.getInstance().clearTrackSelections();
    }

    protected void selectTracks(MouseEvent e) {

        if (log.isDebugEnabled()) {
            log.debug("Enter selectTracks");
        }

        for (MouseableRegion mouseRegion : mouseableRegions) {
            if (mouseRegion.containsPoint(e.getX(), e.getY())) {
                TrackManager.getInstance().setTrackSelections(mouseRegion.getTracks());
                return;
            }
        }
    }

    protected boolean isTrackSelected(MouseEvent e) {
        for (MouseableRegion mouseRegion : mouseableRegions) {
            if (mouseRegion.containsPoint(e.getX(), e.getY())) {
                for (Track t : mouseRegion.getTracks()) {
                    if (t.isSelected()) {
                        return true;
                    }
                }
            }
        }
        return false;

    }
}
