/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * TrackPanel.java
 *
 * Created on Sep 5, 2007, 4:09:39 PM
 *
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ui.panel;

//~--- non-JDK imports --------------------------------------------------------
import java.awt.event.AdjustmentEvent;
import org.broad.igv.ui.*;
import org.broad.igv.IGVConstants;
import org.broad.igv.track.Track;
import org.broad.igv.track.TrackGroup;

//~--- JDK imports ------------------------------------------------------------

import java.awt.*;
import java.awt.event.AdjustmentListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;

import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.swing.SwingUtilities;
import javax.swing.event.MouseInputAdapter;
import org.apache.log4j.Logger;
import org.broad.igv.track.TrackManager;
import org.broad.igv.ui.dnd.AbstractGhostDropManager;
import org.broad.igv.ui.dnd.GhostDropEvent;
import org.broad.igv.ui.dnd.GhostDropListener;
import org.broad.igv.ui.dnd.GhostGlassPane;

/**
 *
 * @author jrobinso
 */
public class TrackNamePanel extends TrackSetComponent implements AdjustmentListener {

    private static Logger log = Logger.getLogger(TrackNamePanel.class);
    

    // private List<? extends Track> tracks = TrackManager.getInstance().getDataTracks();
    /**
     * Constructs ...
     *
     */
    static List<DropListener> dropListeners = new ArrayList();

    public TrackNamePanel() {
        init();
    }

    @Override
    protected void paintComponent(Graphics g) {

        super.paintComponent(g);

        removeMousableRegions();

        // Get the current tracks
        TrackSetView dataTrackView = (TrackSetView) getParent();

        // Get available tracks
        Collection<TrackGroup> groups = dataTrackView.getGroups();

        Rectangle visibleRect = getVisibleRect();

        if (!groups.isEmpty()) {
            final Graphics2D graphics2D = (Graphics2D) g.create();
            graphics2D.setColor(Color.BLACK);

            final Graphics2D greyGraphics = (Graphics2D) g.create();
            greyGraphics.setColor(IGVConstants.VERY_LIGHT_GRAY);

            int regionX = 0;
            int regionY = 0;

            for (Iterator<TrackGroup> groupIter = groups.iterator(); groupIter.hasNext();) {
                TrackGroup group = groupIter.next();

                if (group.isVisible()) {
                    if (groups.size() > 1) {
                        if (regionY + IGVConstants.groupGap >= visibleRect.y &&
                                regionY < visibleRect.getMaxY()) {
                            greyGraphics.fillRect(0, regionY + 1, getWidth(),
                                    IGVConstants.groupGap - 1);
                        }
                        regionY += IGVConstants.groupGap;
                    }

                    if (group.isDrawBorder() && regionY + IGVConstants.groupGap >= visibleRect.y &&
                            regionY < visibleRect.getMaxY()) {
                        g.drawLine(0, regionY - 1, getWidth(), regionY - 1);
                    }

                    int previousRegion = -1;
                    List<Track> tmp = new ArrayList(group.getTracks());
                    for (Track track : tmp) {

                        track.setTop(regionY);

                        int trackHeight = track.getHeight();

                        if (track.isVisible()) {

                            previousRegion = regionY;

                            if (regionY + trackHeight >= visibleRect.y &&
                                    regionY < visibleRect.getMaxY()) {
                                int width = getWidth();
                                int height = track.getHeight();

                                Rectangle region = new Rectangle(regionX,
                                        regionY, width, height);
                                addMousableRegion(new MouseableRegion(region, track));
                                draw(graphics2D, track, regionX, regionY, width,
                                        height, visibleRect);

                            }
                            regionY += trackHeight;

                        }
                    }

                    if (group.isDrawBorder()) {
                        g.drawLine(0, regionY, getWidth(), regionY);
                    }
                }

            }
        }
    }

    private void draw(Graphics2D graphics, Track track, int trackX, int trackY,
            int trackWidth,
            int trackHeight,
            Rectangle visibleRect) {

        Rectangle rect = new Rectangle(trackX, trackY, trackWidth, trackHeight);

        track.renderName(graphics, rect, visibleRect);

    }

    private void init() {

        setBorder(javax.swing.BorderFactory.createLineBorder(Color.black));
        setBackground(new java.awt.Color(255, 255, 255));
        org.jdesktop.layout.GroupLayout dataTrackNamePanelLayout = new org.jdesktop.layout.GroupLayout(
                this);
        setLayout(dataTrackNamePanelLayout);
        dataTrackNamePanelLayout.setHorizontalGroup(
                dataTrackNamePanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING).add(
                0, 148, Short.MAX_VALUE));
        dataTrackNamePanelLayout.setVerticalGroup(
                dataTrackNamePanelLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING).add(
                0, 528, Short.MAX_VALUE));

        NamePanelMouseAdapter mouseAdapter = new NamePanelMouseAdapter();
        addMouseListener(mouseAdapter);
        addMouseMotionListener(mouseAdapter);

        DropListener dndListener = new DropListener(this);
        addGhostDropListener(dndListener);
    }

    /**
     * Method description
     *
     *
     * @param x
     * @param y
     *
     * @return
     */
    public String getPopupMenuTitle(int x, int y) {

        Collection<Track> tracks = getSelectedTracks();

        String popupTitle;
        if (tracks.size() == 1) {
            popupTitle = tracks.iterator().next().getDisplayName();
        } else {
            popupTitle = "Total Tracks Selected: " + tracks.size();
        }

        return popupTitle;
    }

    /**
     * Method description
     *
     *
     * @param x
     * @param y
     *
     * @return
     */
    public String getTooltipTextForLocation(int x, int y) {

        List<MouseableRegion> mouseableRegions = TrackNamePanel.this.getMouseableRegions();

        for (MouseableRegion mouseableRegion : mouseableRegions) {
            if (mouseableRegion.containsPoint(x, y)) {
                return mouseableRegion.getText();
            }
        }
        return "";
    }

    static void addGhostDropListener(DropListener listener) {
        if (listener != null) {
            dropListeners.add(listener);
        }
    }

    static void removeGhostDropListener(DropListener listener) {
        if (listener != null) {
            dropListeners.remove(listener);
        }
    }

    /**
     * Listener for scroll pane events
     * @param evt
     */
    public void adjustmentValueChanged(AdjustmentEvent evt) {
        if (evt.getValueIsAdjusting()) {
            // The user is dragging the knob
            return;
        }
        repaint();

    }
    BufferedImage dndImage = null;

    private synchronized void createDnDImage() {
        dndImage = new BufferedImage(getWidth(), 2, BufferedImage.TYPE_INT_ARGB);
        Graphics g = dndImage.getGraphics();
        g.setColor(Color.blue);
        g.drawLine(1, 0, getWidth() - 2, 0);
        g.drawLine(1, 1, getWidth() - 2, 1);

    }

    protected void shiftSelectTracks(MouseEvent e) {
        for (MouseableRegion mouseRegion : mouseableRegions) {
            if (mouseRegion.containsPoint(e.getX(), e.getY())) {
                Collection<Track> clickedTracks = mouseRegion.getTracks();
                if (clickedTracks != null && clickedTracks.size() > 0) {
                    Track t = clickedTracks.iterator().next();
                    TrackManager.getInstance().shiftSelectTracks(t);
                }
                return;
            }
        }
    }

    /**
     * Mouse adapter for the track name panel.  Supports multiple selection,
     * popup menu, and drag & drop within or between name panels.
     */
    class NamePanelMouseAdapter extends MouseInputAdapter {

        boolean isDragging = false;
        Point dragStart = null;

        @Override
        /**
         * Mouse down.  Track selection logic goes here.
         */
        public void mousePressed(MouseEvent e) {

            if(log.isDebugEnabled()) {
                log.debug("Enter mousePressed");
            }

            requestFocus();
            grabFocus();

            if (e.isPopupTrigger()) {
                if (!isTrackSelected(e)) {
                    clearTrackSelections();
                    selectTracks(e);
                }
                openPopupMenu(e);
            } // meta (mac) or control,  toggle selection]
            else if (e.getButton() == MouseEvent.BUTTON1) {
                if (e.isMetaDown() || e.isControlDown()) {
                    toggleTrackSelections(e);
                } else if (e.isShiftDown()) {
                    shiftSelectTracks(e);
                } else if (!isTrackSelected(e)) {
                    clearTrackSelections();
                    selectTracks(e);
                }
            } else if (!isTrackSelected(e)) {
                clearTrackSelections();
                selectTracks(e);
            }


            IGVMainFrame.getInstance().repaintNamePanels();

        }

        public void mouseReleased(MouseEvent e) {

            if(log.isDebugEnabled()) {
                log.debug("Enter mouseReleased");
            }

            if (isDragging) {


                Component c = e.getComponent();

                IGVMainFrame.getInstance().endDnD();
                GhostGlassPane glassPane = IGVMainFrame.getInstance().getDnDGlassPane();

                Point p = (Point) e.getPoint().clone();
                SwingUtilities.convertPointToScreen(p, c);

                Point eventPoint = (Point) p.clone();
                SwingUtilities.convertPointFromScreen(p, glassPane);

                glassPane.setPoint(p);
                glassPane.setVisible(false);
                glassPane.setImage(null);

                fireGhostDropEvent(new GhostDropEvent(dragStart, eventPoint));
            }

            if (e.isPopupTrigger()) {
                openPopupMenu(e);
            } else {
                if (!isDragging && !e.isMetaDown() && !e.isControlDown() &&
                        !e.isShiftDown()) {
                    clearTrackSelections();
                    selectTracks(e);
                    IGVMainFrame.getInstance().repaintNamePanels();
                }
            }

            isDragging = false;
            dndImage = null;

        }

        public void mouseDragged(MouseEvent e) {
      
            Component c = e.getComponent();
            if(e.isPopupTrigger()) {
                return;
            }
            if (!isDragging) {
                dragStart = e.getPoint();
                dragStart.x = getWidth() / 2;
                IGVMainFrame.getInstance().startDnD();

                if (dndImage == null) {
                    createDnDImage();
                }
                IGVMainFrame.getInstance().getDnDGlassPane().setImage(dndImage);
                isDragging = true;
            // Code below paints target component on the dndImage.  It needs modified to paint some representation
            // of the selectect tracks, probably the track names printed as a list.
            }
            if (isDragging) {

                final GhostGlassPane glassPane = IGVMainFrame.getInstance().getDnDGlassPane();

                Point p = (Point) e.getPoint().clone();
                p.x = getWidth() / 2;
                SwingUtilities.convertPointToScreen(p, c);
                SwingUtilities.convertPointFromScreen(p, glassPane);

                glassPane.setPoint(p);

                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {
                        glassPane.paintImmediately(glassPane.getBounds());
                    }
                });
            //glassPane.repaint();
            }
        }

        public void mouseMoved(MouseEvent e) {
            int x = e.getX();
            int y = e.getY();
            setToolTipText(getTooltipTextForLocation(x, y));
        }

        protected void fireGhostDropEvent(GhostDropEvent evt) {
            Iterator it = TrackNamePanel.dropListeners.iterator();
            while (it.hasNext()) {
                ((GhostDropListener) it.next()).ghostDropped(evt);
            }
        }
    }

    class DropListener extends AbstractGhostDropManager {

        TrackNamePanel panel;

        public DropListener(TrackNamePanel target) {
            super(target);
            this.panel = panel;

        }

        public void ghostDropped(GhostDropEvent e) {
            Point startPoint = e.getStartLocation();
            Point dropPoint = getTranslatedPoint(e.getDropLocation());

            if (isInTarget(dropPoint)) {
                tracksDropped(startPoint, dropPoint);
            }
        }
    }

    void tracksDropped(Point startPoint, Point dropPoint) {

        // This cast is horrid but we can't fix everything at once.
        String viewType = ((TrackSetView) this.getParent()).getName();
        List<MouseableRegion> regions = getMouseableRegions();


        // Find the regions containing the startPoint and point
        boolean before = true;
        MouseableRegion dropRegion = null;
        MouseableRegion startRegion = null;
        for (MouseableRegion region : regions) {
            if (region.containsPoint(dropPoint.x, dropPoint.y)) {
                dropRegion = region;
                Rectangle bnds = dropRegion.getBounds();
                int dy1 = (dropPoint.y - bnds.y);
                int dy2 = bnds.height - dy1;
                before = dy1 < dy2;
            }
            if (region.containsPoint(startPoint.x, startPoint.y)) {
                startRegion = region;
            }
            if (dropRegion != null && startRegion != null) {
                break;
            }
        }

        // If the dropRegion is not found,  or the dropRegion & startRegions are equal return
        // DISABLED
        //if (dropRegion == null || (startRegion != null && (startRegion == dropRegion))) {
        //    return;
        //}

        // Get the track, the interface for getTracks returns a set but in the
        // name panel (here) there is always one
        if (dropRegion == null) {
            TrackManager.getInstance().moveSelectedTracksTo(getSelectedTracks(),
                    viewType, null, before);
        } else {
            Iterator<Track> tmp = dropRegion.getTracks().iterator();
            if (tmp.hasNext()) {
                Track dropTrack = tmp.next();
                TrackManager.getInstance().moveSelectedTracksTo(
                        getSelectedTracks(), viewType, dropTrack, before);
            }
        }
        IGVMainFrame.getInstance().doResizeTrackPanels();

    }
}
