/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * TrackPanel.java
 *
 * Created on Sep 5, 2007, 4:09:39 PM
 *
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ui.panel;

import org.broad.igv.ui.*;
import org.broad.igv.feature.Chromosome;
import org.broad.igv.feature.Genome;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;
import javax.swing.JPanel;
import javax.swing.event.MouseInputAdapter;
import static org.broad.igv.IGVConstants.*;

/**
 *
 * @author jrobinso
 * 
 * Lucida Blackletter
Lucida Bright
Lucida Calligraphy
Lucida Fax
Lucida Grande
Lucida Handwriting
Lucida Sans
Lucida Sans Typewriter
 */
public class RulerPanel extends JPanel {

    // TODO -- get from preferences
    boolean drawSpan = true;
    boolean drawEllipsis = false;
    private Font tickFont = FontManager.getScalableFont(Font.BOLD, 9);
    private Font spanFont = FontManager.getScalableFont(Font.BOLD, 12);
    private Font chrFont = FontManager.getScalableFont(10);
    private List<ClickLink> chromosomeRects = new ArrayList();

    public RulerPanel() {
        initialize();
    }

    private boolean isWholeGenomeView() {
        return getViewContext().getChrName().equals("All");
    }

    @Override
    protected void paintComponent(Graphics g) {

        super.paintComponent(g);


        g.setColor(Color.black);

        if (isWholeGenomeView())
        {
            drawChromosomeTicks(g);
        } else
        {
            // Clear panel
            drawTicks(g);
            if (drawSpan)
            {
                drawSpan(g);
            }
            if (drawEllipsis)
            {
                drawEllipsis(g);
            }
        }
    /*
    // Draw regions of interest?
    ViewContext viewContext = IGVMainFrame.theInstance.getViewContext();
    if(viewContext.isShowRegionsOfInterestOn()) {
    boolean drawBackground = false;
    viewContext.drawRegionsOfInterest(g, getHeight(), drawBackground);
    }
     */

    /*
    // Manually draw border because setBorder() shifts the layout
    g.setColor(Color.BLACK);  
    int x = getX();
    int y = getY();
    int width = getWidth();
    int height = getHeight();
    g.drawLine(0, 0, 0, height); // left border    
    g.drawLine(width-1, 0, width-1, height); // right border     
     */
    }

    private ViewContext getViewContext() {
        return IGVModel.getInstance().getViewContext();
    }

    private void drawSpan(Graphics g) {

        //TODO -- hack

        int w = getViewContext().getDataPanelWidth();

        g.setFont(spanFont);

        // Positions are 1/2 open, subtract 1 to compensate
        int range = (int) (getViewContext().getScale() * w) - 1;

        TickSpacing ts = findSpacing(range);
        String rangeString = formatNumber((double) range / ts.getUnitMultiplier()) +
            " " + ts.getMajorUnit();
        int strWidth = g.getFontMetrics().stringWidth(rangeString);
        int strHeight = g.getFontMetrics().getAscent();
        int strPosition = (w - strWidth) / 2;

        int lineY = getHeight() - 35 - strHeight / 2;

        g.drawLine(0, lineY, (w - strWidth) / 2 - 10, lineY);
        int[] arrowX = {0, 10, 10};
        int[] arrowY = {lineY, lineY + 3, lineY - 3};
        g.fillPolygon(arrowX, arrowY, arrowX.length);

        g.drawLine((w + strWidth) / 2 + 10, lineY, w, lineY);
        arrowX = new int[]{w, w - 10, w - 10};
        g.fillPolygon(arrowX, arrowY, arrowX.length);

        g.drawString(rangeString, strPosition, getHeight() - 35);

    }

    private void drawEllipsis(Graphics g) {
        double cytobandScale =
            ((double) getViewContext().getChromosomeLength()) / getViewContext().getDataPanelWidth();

        double maxPixel = getViewContext().getMaxPixel();
        //visibleFraction = maxPixel < 0 ? 0 : ((double) getViewContext().getDataPanelWidth()) / maxPixel;

        int start = (int) ((getViewContext().getOrigin()) / cytobandScale);
        int span =
            (int) ((getViewContext().getDataPanelWidth() * getViewContext().getScale()) / cytobandScale);
        int end = start + span;

        g.drawLine(start, 0, 0, getHeight());
        g.drawLine(end, 0, getWidth(), getHeight());

    }

    private void drawTicks(Graphics g) {

        int w = getViewContext().getDataPanelWidth();

        g.setFont(tickFont);

        int range = (int) (w * getViewContext().getScale());
        TickSpacing ts = findSpacing(range);
        double spacing = ts.getMajorTick();

        // Find starting point closest to the current origin
        int nTick = (int) (getViewContext().getOrigin() / spacing) - 1;
        int l = (int) (nTick * spacing);
        int x = getViewContext().getPixelPosition(l);
        //int strEnd = Integer.MIN_VALUE;
        while (x < getWidth())
        {
            l = (int) (nTick * spacing);
            x = getViewContext().getPixelPosition(l);
            String chrPosition = formatNumber((double) l / ts.getUnitMultiplier()) +
                " " + ts.getMajorUnit();
            int strWidth = g.getFontMetrics().stringWidth(chrPosition);
            int strPosition = x - strWidth / 2;
            //if (strPosition > strEnd) {

            if (nTick % 2 == 0)
            {
                g.drawString(chrPosition, strPosition, getHeight() - 15);
            }
            //strEnd = strPosition + strWidth;
            //}
            g.drawLine(x, getHeight() - 10, x, getHeight() - 2);
            nTick++;
        }
    }

    private void drawChromosomeTicks(Graphics g) {

        //this.removeAll();
        this.setLayout(null);

        // TODO -- remove hardcoded value
        int locationUnit = 1000;

        g.setFont(chrFont);

        Genome genome = getViewContext().getGenome();
        boolean even = true;
        long offset = 0;
        chromosomeRects.clear();
        for (String chrName : genome.getChromosomeNames())
        {
            Chromosome c = genome.getChromosome(chrName);
            int chrLength = c.getLength();

            int x = (int) (offset / (locationUnit * getViewContext().getScale()));

            g.drawLine(x, getHeight() - 10, x, getHeight() - 2);

            int dw = (int) (chrLength / (locationUnit * getViewContext().getScale()));
            int center = x + dw / 2;
            String chrNumber = chrName.replace("chr", "");
            int strWidth = g.getFontMetrics().stringWidth(chrNumber);
            int strPosition = center - strWidth / 2;


            int y = (even ? getHeight() - 35 : getHeight() - 25);
            g.drawString(chrNumber, strPosition, y);

            Rectangle clickRect = new Rectangle(strPosition, y - 15, 15, 15);
            String tooltipText = "Jump to chromosome: " + chrName;
            chromosomeRects.add(new ClickLink(clickRect, chrName, tooltipText));

            even = !even;

            offset += chrLength;
        }
    }

    private String formatNumber(double position) {

        //NumberFormatter f = new NumberFormatter();
        DecimalFormat formatter = new DecimalFormat();
        return formatter.format((int) position);
    //return f.valueToString(position);

    }

    TickSpacing findSpacing(long maxValue) {

        if(maxValue < 10) {
            return new TickSpacing(1, "bp", 1);
        }
        // TODO -- hack, assumes location unit for whole genome is kilo-base
        boolean scaleInKB = getViewContext().getChrName().equals(CHR_ALL);


        // Now man zeroes?
        int nZeroes = (int) Math.log10(maxValue);
        String majorUnit = scaleInKB ? "kb" : "bp";
        int unitMultiplier = 1;
        if (nZeroes > 9)
        {
            majorUnit = scaleInKB ? "tb" : "gb";
            unitMultiplier = 1000000000;
        }
        if (nZeroes > 6)
        {
            majorUnit = scaleInKB ? "gb" : "mb";
            unitMultiplier = 1000000;
        } else if (nZeroes > 3)
        {
            majorUnit = scaleInKB ? "mb" : "kb";
            unitMultiplier = 1000;
        }

        double nMajorTicks = maxValue / Math.pow(10, nZeroes - 1);
        if (nMajorTicks < 25)
        {
            return new TickSpacing(Math.pow(10, nZeroes - 1), majorUnit, unitMultiplier);
        } else
        {
            return new TickSpacing(Math.pow(10, nZeroes) / 2, majorUnit, unitMultiplier);
        }
    }

    private void initialize() {

        if (!isWholeGenomeView())
        {
            setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
            this.setToolTipText("Click to center view.");
        } else
        {

            setCursor(Cursor.getDefaultCursor());
            this.setToolTipText("Click on a chromosome number to jump to that chromosome.");
        }

        MouseInputAdapter mouseAdapter = new MouseInputAdapter() {

            int lastMousePressX;

            @Override
            public void mouseClicked(MouseEvent evt) {
                final MouseEvent e = evt;
                LongRunningTask.submit(new Runnable() {

                    public void run() {
                        if (!isWholeGenomeView())
                        {
                            double newLocation = IGVModel.getInstance().getViewContext().getChromosomePosition(e.getX());
                            getViewContext().centerOnLocation(newLocation);
                        } else
                        {
                            for (ClickLink link : chromosomeRects)
                            {
                                if (link.region.contains(e.getPoint()))
                                {
                                    getViewContext().setChrName(link.value);
                                    // TODO -- get rid of this ugly reference to IGVMainFrame.theInstance
                                    IGVMainFrame.getInstance().chromosomeChangeEvent();

                                    return;
                                }
                                setCursor(Cursor.getDefaultCursor());

                            }
                        }
                    }
                });
            }

            @Override
            public void mouseMoved(MouseEvent e) {
                if (isWholeGenomeView())
                {
                    for (ClickLink link : chromosomeRects)
                    {
                        if (link.region.contains(e.getPoint()))
                        {
                            setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
                            setToolTipText(link.tooltipText);
                            return;
                        }
                    }
                    setCursor(Cursor.getDefaultCursor());
                    setToolTipText("Click on a chromosome number to jump to that chromosome.");

                }
            }

            @Override
            public void mouseEntered(MouseEvent e) {

                if (isWholeGenomeView())
                {
                    setCursor(Cursor.getDefaultCursor());
                    setToolTipText("Click on a chromosome number to jump to that chromosome.");
                } else
                {
                    setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
                    setToolTipText("Click on a location to center the view.");
                }

            }
        };

        addMouseMotionListener(mouseAdapter);

        addMouseListener(mouseAdapter);
    }

    class TickSpacing {

        private double majorTick;
        private double minorTick;
        private String majorUnit = "";
        private int unitMultiplier = 1;

        TickSpacing(double majorTick, String majorUnit, int unitMultiplier) {
            this.majorTick = majorTick;
            this.minorTick = majorTick / 10.0;
            this.majorUnit = majorUnit;
            this.unitMultiplier = unitMultiplier;
        }

        public double getMajorTick() {
            return majorTick;
        }

        public double getMinorTick() {
            return minorTick;
        }

        public String getMajorUnit() {
            return majorUnit;
        }

        public void setMajorUnit(String majorUnit) {
            this.majorUnit = majorUnit;
        }

        public int getUnitMultiplier() {
            return unitMultiplier;
        }

        public void setUnitMultiplier(int unitMultiplier) {
            this.unitMultiplier = unitMultiplier;
        }
    }

// TODO -- possibly generalize?
    class ClickLink {

        Rectangle region;
        String value;
        String tooltipText;

        ClickLink(Rectangle region, String value, String tooltipText) {
            this.region = region;
            this.value = value;
            this.tooltipText = tooltipText;
        }
    }
}
