/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ui.panel;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.track.Track;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Point;
import java.awt.event.MouseEvent;


import java.util.HashSet;
import java.util.Set;
import javax.swing.JComponent;
import javax.swing.JScrollBar;
import javax.swing.JViewport;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.IGVTool;
import org.broad.igv.ui.LongRunningTask;
import org.broad.igv.ui.ViewContext;

/**
 *
 * @author eflakes
 */
public class PanAndZoomTool extends IGVTool {

    private int previousYDirection = 0;    // Drag Directions: 1=up, 0=none 0r -1=down
    private int lastMousePressedY;
    private int cumulativeDeltaX;
    private int cumulativeDeltaY;
    private Point lastMousePoint;
    private JViewport viewport;
    private Container panel;
    private JScrollBar verticalScrollBar;
    private boolean isDragging = false;
    private Cursor dragCursor;

    /**
     * Constructs ...
     *
     *
     * @param owner
     */
    public PanAndZoomTool(Component owner) {
        super(owner, IGVMainFrame.handCursor);    // Cursor.getDefaultCursor());
        this.dragCursor = IGVMainFrame.fistCursor;
        setName("Zoom");
    }

    /**
     * Method description
     *
     *
     * @return
     */
    @Override
    public Cursor getCursor() {
        return isDragging ? dragCursor : super.getCursor();
    }

    /**
     * Method description
     *
     *
     * @param e
     */
    @Override
    public void mousePressed(final MouseEvent e) {

        isDragging = true;

        panel = (Container) e.getSource();

        panel.setCursor(dragCursor);

        lastMousePoint = e.getPoint();
        lastMousePressedY = (int) e.getPoint().getY();
        cumulativeDeltaX = 0;
        cumulativeDeltaY = 0;

        // Vertical scrolling get the viewport
        if ((panel != null) && (panel instanceof DataPanel)) {

            verticalScrollBar = ((DataPanel) panel).getVerticalScrollbar();
            Container parentContainer = panel.getParent();
            if (parentContainer != null) {
                Container parentOfParent = parentContainer.getParent();
                if ((parentOfParent != null) && (parentOfParent instanceof JViewport)) {
                    viewport = (JViewport) parentOfParent;
                }
            }
        }
    }

    /**
     * Method description
     *
     *
     * @param e
     */
    @Override
    public void mouseReleased(final MouseEvent e) {
        viewport = null;
        if (isDragging) {
            getViewContext().snapToGrid();
            isDragging = false;
            DragEventManager.getInstance().dragStopped();
        }
        ((JComponent) e.getSource()).setCursor(getCursor());
    }

    /**
     * Method description
     *
     *
     * @param e
     */
    @Override
    final public void mouseDragged(final MouseEvent e) {

        // ((JComponent) e.getSource()).setCursor(IGVMainFrame.handCursor);
        try {

            if (lastMousePoint == null) {
                lastMousePoint = e.getPoint();
                return;
            }

            double deltaX = lastMousePoint.getX() - e.getX();
            double deltaY = lastMousePoint.getY() - e.getY();
            cumulativeDeltaX += Math.abs(deltaX);
            cumulativeDeltaY += Math.abs(deltaY);

            // Test for horizontal vs vertical panning.
            if (cumulativeDeltaX > cumulativeDeltaY) {

                // Horizontal scrolling
                getViewContext().shiftOriginPixels(deltaX);
                return;
            }

            int totalYChange = lastMousePressedY - e.getY();

            // Vertical Scrolling
            if ((viewport != null) && (totalYChange != 0)) {

                // This section handles false drag direction changes
                int currentYDirection = 0;
                try {

                    // Figure out the current drag direction
                    currentYDirection = totalYChange / Math.abs(totalYChange);

                    // If the previous direction is 0 we were not moving before
                    if (previousYDirection != 0) {

                        // See if we changed direction
                        boolean changedYDirection = currentYDirection != previousYDirection;
                        if (changedYDirection) {

                            // Don't save lastMousePressedPoint because may
                            // be incorrect (this is the problem we are
                            // solving with the direction flag) instead
                            // we'll just check the next drag Point to be
                            // sure of the correct direction.
                            return;
                        }
                    }

                } finally {

                    // Save the current direction indicator for next time
                    previousYDirection = currentYDirection;
                }

                // If we have a vertical scrollbar use it to move
                if (verticalScrollBar != null) {
                    int adjustedScrollbarValue = verticalScrollBar.getValue();
                    adjustedScrollbarValue += totalYChange;
                    verticalScrollBar.setValue(adjustedScrollbarValue);
                }
            }
        } finally {
            lastMousePoint = e.getPoint();    // Always save the last Point
        }
    }

    /**
     * Method description
     *
     *
     * @param e
     */
    @Override
    public void mouseClicked(MouseEvent e) {
        final ViewContext viewContext = getViewContext();

        Object source = e.getSource();
        if (source instanceof DataPanel) {
            Track track = ((DataPanel) source).getTrack(e.getX(), e.getY());
            if (track != null) {
                if (track.handleClick(e)) {
                    return;
                }
            }
        }

        // The shift and alt keys are alternative undocumented zoom options
        // Shift zooms by 8x,  alt zooms out by 2x
        if (e.isShiftDown() || e.isAltDown() || (e.getClickCount() > 1)) {
            int currentZoom = viewContext.getZoom();
            final int newZoom = e.isAltDown()
                    ? Math.max(currentZoom - 1, 0)
                    : (e.isShiftDown() ? currentZoom + 3 : currentZoom + 1);
            final double locationClicked = viewContext.getChromosomePosition(e.getX());

            LongRunningTask.submit(new Runnable() {

                public void run() {
                    viewContext.zoomTo(newZoom, locationClicked);
                }
            });
        }
    }
}
