/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */



package org.broad.igv.ui.panel;

//~--- non-JDK imports --------------------------------------------------------

import org.broad.igv.ui.*;
import org.apache.log4j.Logger;

import org.broad.igv.IGVConstants;
import org.broad.igv.track.FeatureTrack;
import org.broad.igv.track.RenderContext;
import org.broad.igv.track.Track;
import org.broad.igv.track.TrackGroup;
import org.broad.igv.track.TrackManager;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Rectangle;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import java.util.List;

/**
 *
 * @author jrobinso
 */
public class DataPanelPainter {

    private static Logger log = Logger.getLogger(DataPanelPainter.class);

    private ViewContext getViewContext() {
        return IGVModel.getInstance().getViewContext();
    }

    /**
     * Method description
     *
     *
     * @param groups
     * @param context
     * @param width
     * @param height
     * @param background
     */
    public void paint(Collection<TrackGroup> groups, RenderContext context, int width, int height,
                      Color background) {
        paint(groups, context, width, height, background, null);
    }

    /**
     * Method description
     *
     *
     * @param groups
     * @param context
     * @param width
     * @param height
     * @param background
     * @param visibleRect
     */
    public void paint(Collection<TrackGroup> groups, RenderContext context, int width, int height,
                      Color background, Rectangle visibleRect) {

        Graphics2D graphics2D = null;

        try
        {
            graphics2D = (Graphics2D) context.getGraphics().create();
            String chromosomeName = context.getChr();

            graphics2D.setBackground(background);
            graphics2D.clearRect(0, 0, width, height);
            graphics2D.setColor(Color.BLACK);

            final Graphics2D greyGraphics =
                context.getGraphic2DForColor(IGVConstants.VERY_LIGHT_GRAY);
            int trackX = 0;
            int trackY = 0;


            for (Iterator<TrackGroup> groupIter = groups.iterator(); groupIter.hasNext(); )
            {
                TrackGroup group = groupIter.next();

                if (visibleRect != null && (trackY > visibleRect.y + visibleRect.height))
                {
                    break;
                }

                if (group.isVisible())
                {
                    if (groups.size() > 1)
                    {
                        greyGraphics.fillRect(0, trackY + 1, width, IGVConstants.groupGap - 1);
                        trackY += IGVConstants.groupGap;
                    }

                    // Draw a line just above group.   TODO use Java2D line
                    if (group.isDrawBorder())
                    {
                        graphics2D.drawLine(0, trackY - 1, width, trackY - 1);
                    }

                    // Copy tracks to prevent concurrent modfication exception
                    List<Track> trackList = new ArrayList(group.getTracks());
                    for (Track track : trackList)
                    {
                        int trackHeight = track.getHeight();

                        if (visibleRect != null)
                        {
                            if (trackY > visibleRect.y + visibleRect.height)
                            {
                                break;
                            }
                            else if (trackY + trackHeight < visibleRect.y)
                            {
                                if (track.isVisible())
                                {
                                    trackY += trackHeight;
                                }
                                continue;
                            }
                        }


                        if (track.isVisible())
                        {

                            // This code fixes IGV-348
                            if ((track instanceof FeatureTrack)
                                    && ((FeatureTrack) track).isExpanded())
                            {
                            //    ((FeatureTrack) track).getFeaturesByLevels(chromosomeName, 0, 0);
                            }

                            trackY = draw(track, trackX, trackY, width, trackHeight, context);
                        }
                    }

                    // Draw a line just below group.   TODO use Java2D line
                    if (group.isDrawBorder())
                    {
                        graphics2D.drawLine(0, trackY, width, trackY);
                    }
                }
            }

            // Now draw borders.  This is done in a second loop to prevent data from
            // one track from overwriting border from an adjacent track
            trackX = 0;
            trackY = 0;
            for (Iterator<TrackGroup> groupIter = groups.iterator(); groupIter.hasNext(); )
            {

                TrackGroup group = groupIter.next();
                if (group.isVisible())
                {
                    if (groups.size() > 1)
                    {
                        trackY += IGVConstants.groupGap;
                    }

                    // Draw borders  -- copy track list to prevent concurrent modification exception
                    // Copy tracks to prevent concurrent modfication exception
                    List<Track> trackList = new ArrayList(group.getTracks());
                    for (Track track : trackList)
                    {
                        if (track.isVisible())
                        {
                            Rectangle rect = new Rectangle(trackX, trackY, width,
                                                           track.getHeight());
                            track.renderBorder(context, rect);
                            trackY += track.getHeight();
                        }
                    }
                }
            }


        }
        finally
        {
            graphics2D.dispose();
        }
    }

    final private int draw(Track track, int trackX, int trackY, int trackWidth, int trackHeight,
                           RenderContext context) {


        // The rectangle in which the track is drawn.  The height is reduced
        Rectangle rect = new Rectangle(trackX, trackY, trackWidth, trackHeight);



        track.render(context, rect);

        // Get overlays

        List<Track> overlayTracks = TrackManager.getInstance().getOverlayTracks(track);
        if (overlayTracks != null)
        {
            for (Track overlayTrack : overlayTracks)
            {

                // Don't overlay on self
                if (overlayTrack != track)
                {
                    overlayTrack.overlay(context, rect);
                }
            }
        }

        // Change Y to point to the start of the next track
        trackY += trackHeight;

        return trackY;

    }
}
