/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ui.panel;

import org.broad.igv.ui.*;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.util.Collection;
import java.util.concurrent.Callable;
import org.apache.log4j.Logger;
import org.broad.igv.ui.util.GenericUtilities;
import org.broad.igv.track.TrackGroup;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import javax.swing.JComponent;
import org.broad.igv.track.RenderContext;
import org.broad.igv.util.LRUCache;

/**
 *
 * @author jrobinso
 */
public class DataPanelImageManager {

    private static Logger log = Logger.getLogger(DataPanelImageManager.class);
    public static int CACHE_SIZE = 8;
    public static int IMAGE_WIDTH = 500;  // Width of each tile in pixels
    private DataPanelPainter painter = new DataPanelPainter();
    private LRUCache<String, CachedImageWrapper> cache = new LRUCache(CACHE_SIZE);
    private ExecutorService threadExecutor;

    
    // Creating a thread pool with a single instance.  Profiling has revealed lots of thread
    // contention for synchronized objects during image creation.
    public DataPanelImageManager() {
        threadExecutor = Executors.newFixedThreadPool(1);

    }

    public void clearCache() {
        cache.clear();
    }

    public int size() {
        return cache.size();
    }

    public CachedImageWrapper getCachedImage(JComponent requestor, final String chr, int t, final int zoomLevel) {
        String key = generateTileKey(chr, t, zoomLevel);
        CachedImageWrapper wi = cache.get(key);
        if (wi == null) {
            log.error("Missing image for key: " + key);
        }
        return wi;

    }

    /** Build an image for the given groups, orign, and resolution scale
     *  and store it with the given key.
     * @param key
     * @param groups
     * @param origin  in base pairs (not screen pixels)
     * @param scale   in base pairs per pixel
     * @param imageHeight  in pixels
     * @param background  
     */
    public void buildImage(
        Collection<TrackGroup> groups,
        String key,
        int yPosition,
        double origin,
        double scale,
        int imageHeight,
        Color background) {


        if (cache.containsKey(key)) {
        // already have this image.  Nothing to do
        } else {
            BufferedImage image = createImage(groups, origin, imageHeight, background);
            CachedImageWrapper wrapper = new CachedImageWrapper(yPosition, origin, scale, image);
            synchronized (cache) {
                cache.put(key, wrapper);
            }
        }

    }

    private BufferedImage createImage(Collection<TrackGroup> groups, double origin, int imageHeight, Color background) {
        // Create an image the size of the track panel
        BufferedImage image = GenericUtilities.getDeviceCompatibleImage(IMAGE_WIDTH, imageHeight);
        // paint image starting at the new offset and then store it
        Graphics2D g = (Graphics2D) image.getGraphics();

        ViewContext vc = IGVModel.getInstance().getViewContext();
        double scale = vc.getScale();
        String chromosomeName = vc.getChrName();
        double startLocation = origin;
        double endLocation = (startLocation + IMAGE_WIDTH * scale + 1);
        int zoom = vc.getZoom();

        RenderContext context =
            new RenderContext(chromosomeName, startLocation, endLocation, zoom,
            origin, scale, null, g);

        painter.paint(groups, context, IMAGE_WIDTH, imageHeight, background);
        image.flush();
        g.dispose();
        long availableMemory = Runtime.getRuntime().maxMemory() - Runtime.getRuntime().totalMemory();
        if (availableMemory < 100000000) {
            Runtime.getRuntime().gc();
        }
        return image;

    }

    public CachedImageWrapper getCachedImageWrapper(String key) {
        return cache.get(key);
    }

    public void submit(Callable callable) {
        threadExecutor.submit(callable);
    }

    java.lang.String generateTileKey(final String chr, int t, final int zoomLevel) {

        // Fetch image for this chromosome, zoomlevel, and tile.  If found
        // draw immediately
        java.lang.String key = chr + "_z_" + zoomLevel + "_t_" + t;
        return key;
    }
}
