/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */



/*
* LocationPanel.java
*
* Created on September 11, 2007, 2:29 PM
*
* To change this template, choose Tools | Template Manager
* and open the template in the editor.
 */
package org.broad.igv.ui.panel;

//~--- non-JDK imports --------------------------------------------------------

import org.broad.igv.ui.*;
import org.broad.igv.IGVConstants;

//import com.sun.java.swing.SwingUtilities2;
import org.broad.igv.renderer.CytobandRenderer;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;

import javax.swing.JPanel;
import javax.swing.event.MouseInputAdapter;

/**
 *
 * @author jrobinso
 */
public class CytobandPanel extends JPanel {

    private static int fontHeight = 10;
    private static int bandHeight = 10;
    private static String fontFamilyName = "Lucida Sans";
    private boolean isDragging = false;
    private double viewOrigin;
    double cytobandScale;

    /** Creates a new instance of LocationPanel */
    public CytobandPanel() {

        FontManager.getScalableFont(fontHeight);
        this.setFont(new Font(fontFamilyName, Font.BOLD, fontHeight));


        // TODO SwingUtilities2 is not compatible with 1.6
        // this.putClientProperty(SwingUtilities2.AA_TEXT_PROPERTY_KEY, new Boolean(true));
        initialize();

    }

    /**
     * The panel is divided into 2 regions. (1) the cytoband itself, and 2)
     * location ellipsis
     */
    @Override
    protected void paintComponent(Graphics g) {

        super.paintComponent(g);

        if (getViewContext().getChrName().equals(IGVConstants.CHR_ALL))
        {
            Graphics g2 = g.create();
            g2.setFont(FontManager.getScalableFont(Font.ITALIC, 12));
            String text = "Whole genome view.  To jump to a chromosome click on its label.";
            g2.drawString(text, 20, getHeight() - 5);
            return;
        }

        int dataPanelWidth = getViewContext().getDataPanelWidth();
        Rectangle cytoRect = new Rectangle(0, 10, dataPanelWidth, bandHeight);

        (new CytobandRenderer()).draw(getViewContext().getChromosome().getCytobands(),
                                      (Graphics2D) g, cytoRect);

        int chromosomeLength = getViewContext().getChromosomeLength();
        cytobandScale = ((double) chromosomeLength) / dataPanelWidth;

        // The test is true if we are zoomed in (that is, if the virtual image size is greater
        // than the viewport width).
        if (getViewContext().getMaxPixel() > dataPanelWidth)
        {

            double scale = getViewContext().getScale();

            double origin = isDragging ? viewOrigin : getViewContext().getOrigin();

            int start = (int) (origin / cytobandScale);
            double scaledDataPanelWidth = dataPanelWidth * scale;
            int span = (int) (scaledDataPanelWidth / cytobandScale);

            // Draw Cytoband current region viewer
            int height = (int) cytoRect.getHeight();
            g.setColor(Color.RED);
            int y = (int) (cytoRect.getY()) + CytobandRenderer.CYTOBAND_Y_OFFSET;

            g.drawRect(start, y, span, height);
            g.drawRect(start - 1, (y - 1), span + 2, height + 2);
            g.drawRect(start - 2, (y - 2), span + 4, height + 4);
            if (span < 2)
            {

                g.drawRect(start - 2, (y - 2), span + 4, height + 4);
            }
        }
    }
    int toolOrigin = -1;

    private void initialize() {

        setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
        this.setToolTipText(
            "<html>Click anywhere on the cytoband <p>to center view at that location.");



        MouseInputAdapter mouseAdapter = new MouseInputAdapter() {

            int lastMousePressX;

            public void mouseClicked(MouseEvent e) {
                final int mouseX = e.getX();
                final int clickCount = e.getClickCount();
                LongRunningTask.submit(new Runnable() {

                    public void run() {
                        double newLocation = cytobandScale * mouseX;
                        if (clickCount > 1)
                        {
                            final int newZoom = getViewContext().getZoom() + 1;
                            getViewContext().zoomTo(newZoom, newLocation);
                        }
                        else
                        {
                            getViewContext().centerOnLocation(newLocation);
                        }

                    }
                });
            }

            @Override
            public void mousePressed(MouseEvent e) {
                lastMousePressX = e.getX();
            }

            @Override
            public void mouseReleased(MouseEvent e) {
                if (isDragging)
                {
                    LongRunningTask.submit(new Runnable() {

                        public void run() {
                            getViewContext().setOrigin(viewOrigin);
                        }
                    });
                }
                isDragging = false;
            }

            @Override
            public void mouseDragged(MouseEvent e) {
                if (!isDragging)
                {
                    isDragging = true;
                    viewOrigin = getViewContext().getOrigin();
                }

                // Dragging disabled until image thread is improved.  Currently
                // dragging will overwhelm the queue

                /*
                 *   int start = (int) (origin / cytobandScale);
                 * double scaledDataPanelWidth = dataPanelWidth * scale;
                 * int span = (int) (scaledDataPanelWidth / cytobandScale);
                 */
                int w = getViewContext().getDataPanelWidth();
                double scale = getViewContext().getScale();
                int x = (int) Math.max(0, Math.min(e.getX(), w * (cytobandScale - scale)));
                int delta = x - lastMousePressX;
                if ((delta != 0) && (cytobandScale > 0))
                {
                    viewOrigin = Math.max(0, Math.min(viewOrigin + delta * cytobandScale,
                                                      w * (cytobandScale - scale)));
                    repaint();
                }
                lastMousePressX = x;

                // Dragging disabled until image thread is improved.  Currently
                // dragging will overwhelm the queue

                /*
                 * int x = e.getX();
                 * int delta = x - lastMousePressX;
                 * if(delta != 0 && visibleFraction > 0) {
                 * getViewContext().shiftOriginPixels((int) (delta / visibleFraction));
                 * }
                 * lastMousePressX = x;
                 */

            }

            @Override
            public void mouseEntered(MouseEvent e) {}

            @Override
            public void mouseExited(MouseEvent e) {
                
            }
        };

        addMouseMotionListener(mouseAdapter);
        addMouseListener(mouseAdapter);
    }

    // TODO remove reference to IGVMainFrame.theInstance
    private ViewContext getViewContext() {
        return IGVModel.getInstance().getViewContext();
    }
}
