/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ui.panel;

import java.awt.Image;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Iterator;
import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.stream.ImageOutputStream;
import org.apache.log4j.Logger;

/**
 *
 * @author jrobinso
 */
public class CachedImageWrapper {

    private static Logger log = Logger.getLogger(CachedImageWrapper.class);
    
    /** Set to true to enable image compression.  It was hoped this would increase
     * perfomance by decreasing the amount of memory required by large images.
     * This did not pan out,  but the option is retained and defaulted to "false".
     */
    private static boolean useCompression = false;
    /**
     * Origin in base pairs
     */
    private double origin;
    /**
     * Scale in bp / pixel
     */
    private double scale;
    
    /**
     * Vertical offset for this tile in pixels
     */
    private int yPosition;
    
    /**
     * The image
     */
    private BufferedImage image;
    private byte[] compressedBytes;

    public CachedImageWrapper(int yOffset, double origin, double scale, BufferedImage image) {
        this.yPosition = yOffset;
        this.origin = origin;
        this.scale = scale;
        if (useCompression) {
            compressImage(image, 0.5f);
        } else {
            this.image = image;
        }
    }
    
    public int getYPosition() {
        return yPosition;
        
    }

    public double getOrigin() {
        return origin;
    }

    public double getScale() {
        return scale;
    }

    public Image getImage() {
        if(image == null) {
            return decompressImage();
        }
        return image;
    }

    private void compressImage(BufferedImage image, float quality) {

        try {

            // Get a ImageWriter for jpeg format.
            Iterator<ImageWriter> writers = ImageIO.getImageWritersBySuffix("jpeg");
            if (!writers.hasNext()) {
                throw new IllegalStateException("No writers found");
            }
            ImageWriter writer = (ImageWriter) writers.next();
            // Create the ImageWriteParam to compress the image.
            ImageWriteParam param = writer.getDefaultWriteParam();
            param.setCompressionMode(ImageWriteParam.MODE_EXPLICIT);
            param.setCompressionQuality(quality);
            // The output will be a ByteArrayOutputStream (in memory)
            ByteArrayOutputStream bos = new ByteArrayOutputStream(32768);
            ImageOutputStream ios = ImageIO.createImageOutputStream(bos);
            writer.setOutput(ios);
            writer.write(null, new IIOImage(image, null, null), param);
            ios.flush(); // otherwise the buffer size will be zero!
            
            compressedBytes = bos.toByteArray();
            this.image = null;
            
            System.out.println("Image byte array size = " + compressedBytes.length);

        } catch (IOException iOException) {
            log.error("Error compressing image", iOException);
            this.image = image;
        }

    }

    private Image decompressImage() {
        if (compressedBytes == null) {
            return null;
        }
        try {
            ByteArrayInputStream in = new ByteArrayInputStream(compressedBytes);
            return ImageIO.read(in);

        } catch (IOException iOException) {
            log.error("Error decompressing image");
            return null;
        }

    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        //System.out.println("Finalizing cached image");
    }
    
    
}
