/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ui.panel;

import org.broad.igv.ui.*;
import java.awt.Color;
import java.awt.Graphics2D;
import java.util.Collection;
import java.util.Stack;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.Callable;
import org.apache.log4j.Logger;
import org.broad.igv.track.TrackGroup;
import org.broad.igv.ui.WaitCursorManager.CursorToken;

/**
 *
 * @author eflakes
 */
public class CachedDataPanelPainter implements CachedIGVDataPanelPainter {

    private double requestCount;
    private double processedCount;
    private double discardedCount;
    private boolean isDebugEnabled = false;
    private Timer debugTimer = new Timer("CachedDataPanelPainter Debug Thread");
    private TimerTask debuggingTask = new TimerTask() {

        public void run() {
            printDebuggingStatistics();
        }
    };
    private final int MAX_TILE_STACK_SIZE = 3;
    private static Logger logger = Logger.getLogger(CachedDataPanelPainter.class);
    // Class to manage cached images.
    private  DataPanelImageManager imageManager = new DataPanelImageManager();
    /**
     * Keep track of image tiles currently being built.
     * TODO move to image manager
     */
    private final Stack<Integer> tileStack = new Stack();

    /**
     * Do a cached image paint.
     * 
     * @param dataPanel
     * @param groups
     * @param graphics2D
     */
    final public void paint(final DataPanel dataPanel,
            final Collection<TrackGroup> groups, Graphics2D graphics2D) {

        final ViewContext viewContext = dataPanel.getViewContext();
        
        final String chr = viewContext.getChrName();
        final double scale = viewContext.getScale();
        final double startLocation = viewContext.getOrigin();
        final double endLocation = (startLocation + dataPanel.getWidth() * scale);
        final int zoomLevel = viewContext.getZoom();
        final double scaledImageWidth = DataPanelImageManager.IMAGE_WIDTH * scale;

        // Compute the start and end "tile" (image index) 
        // to cover the requested region
        final int startTile = (int)Math.max(0, (startLocation/scaledImageWidth));
        final int endTile =(int)(endLocation/scaledImageWidth);

        // Loop through tiles
        for (int tileNumber = startTile; tileNumber <= endTile; tileNumber++) {

            final int tile = tileNumber;

            // Remove duplicate before adding the latest requested tile
            synchronized (tileStack) {
                int index = tileStack.indexOf(tileNumber);
                if(index != -1) tileStack.remove(index);
            }

            // Fetch image for this chromosome, zoomlevel, and tile.
            // If found draw immediately
            String key = dataPanel.generateTileKey(chr, tile, zoomLevel);
            CachedImageWrapper imageWrapper = imageManager.getCachedImageWrapper(key);
            if (imageWrapper != null) {
                int xOffset = (int) ((imageWrapper.getOrigin() - startLocation) / scale);   
               
                // TODO -- something like...
                // int yOffset = imageWrapper.getYPosition() - scrollYPoistion,   or maybe vice versa
                int yOffset = 0;
                
                graphics2D.drawImage(imageWrapper.getImage(), xOffset, yOffset, null);
            } else {
                
                // Write message on canvas indicating image is loading TODO -- center the string
                final int tileOrigin = (int) (tile * scaledImageWidth);
                int leftPixelPos = Math.max(0, viewContext.getPixelPosition(tileOrigin) + 10);
                for (int w = leftPixelPos; w < dataPanel.getWidth(); w += 250) {
                    graphics2D.drawString(" loading ...", w, dataPanel.getHeight() / 2);
                }

                // Image was not found and must be built. It takes time to build 
                // an image, use a new thread to prevent blocking swing thread.
                tileStack.push(tile);
                ++requestCount;

                //No work to do if the stack is empty
                if (!tileStack.isEmpty()) {
                    Callable<Integer> imageBuilderCallable =
                            createCallable(dataPanel, groups, zoomLevel, chr, scale, tileStack);
                    imageManager.submit(imageBuilderCallable);
                }
            }

        }
    }

    /**
     * Create the callable thread object.
     * 
     * @param dataPanel
     * @param groups
     * @param zoomLevel
     * @param chromosome
     * @param scale
     * @return
     */
    private Callable<Integer> createCallable(final DataPanel dataPanel,
            final Collection<TrackGroup> groups, final int zoomLevel,
            final String chromosome, final double scale,
            final Stack<Integer> tilesToLoadStack) {

        // Now every Callable callable shares the resposibility of cleaning 
        // up the same LIFO queue
        final Callable<Integer> imageBuilderCallable = new Callable() {

            final public Integer call() {

               CursorToken token = WaitCursorManager.showWaitCursor();
                try {

                    // Process all requested images
                    // TODO get this to clear out all but the top 3? tile in 
                    // queue (performance enhancement)
                    while (true) {

                        int t;
                        synchronized (tilesToLoadStack) {
                            if (tilesToLoadStack.isEmpty()) {
                                break;
                            } // All images processed
                            t = tilesToLoadStack.pop();
                        }

                        doImageBuild(t);

                        ++processedCount;


                        if (!dataPanel.isPaintingTile()) {
                            dataPanel.repaint();
                        }

                        // If the stack has more items on it than we want
                        // we should trim it back by removing the oldest 
                        // draw requests which are not really necessary
                        synchronized (tilesToLoadStack) {

                            int stackSize = tilesToLoadStack.size();
                            if (stackSize > MAX_TILE_STACK_SIZE) {

                                int indexOfLastStackItem = stackSize - 1;
                                while (tilesToLoadStack.size() > MAX_TILE_STACK_SIZE) {
                                    tilesToLoadStack.remove(indexOfLastStackItem--);
                                    ++discardedCount;
                                }
                            }
                        }
                    }

                    return 1;

                } catch (Exception ex) {
                    logger.error("Error in repaint call", ex);
                    return -1;
                } finally {
                    WaitCursorManager.removeWaitCursor(token);

                }
            }

            final public void doImageBuild(int tile) {

                String key = dataPanel.generateTileKey(chromosome, tile, zoomLevel);
                double tileOrigin = (tile * scale) * DataPanelImageManager.IMAGE_WIDTH;

                Color background = dataPanel.getBackground();
                
                // TODO,  input the yPosition for this tile 
                int yPosition = 0;
                // TODO, something like height = Math.min(dataPanel.getHeight(), DataPanelImageManager.IMAGE_HEIGHT);
                int height = dataPanel.getHeight();
                               
                imageManager.buildImage(groups, key, yPosition, tileOrigin, scale,
                        height, background);

            }
        };

        return imageBuilderCallable;
    }

    public void setDebuggingEnabled(boolean enabled) {

        isDebugEnabled = enabled;
        if (isDebugEnabled) {
            debugTimer.schedule(debuggingTask, 10000, 3000);
        } else {
            debuggingTask.cancel();
        }
    }

    public void clearImageCache() {
        imageManager.clearCache();
    }

    synchronized public void printDebuggingStatistics() {

        System.out.println("\n\nGetting statistics");
        System.out.println("\trequestCount: " + requestCount);
        System.out.println("\tprocessedCount: " + processedCount);
        System.out.println("\tdiscardedCount: " + discardedCount);
    }
}
