/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/



/*
* TrackPanel.java
*
* Created on Sep 5, 2007, 4:09:39 PM
*
* To change this template, choose Tools | Templates
* and open the template in the editor.
 */

package org.broad.igv.ui.panel;

//~--- non-JDK imports --------------------------------------------------------

import org.broad.igv.ui.*;
import org.broad.igv.PreferenceManager;
import org.broad.igv.track.Track;
import org.broad.igv.track.TrackManager;

//Static imports
import static org.broad.igv.ui.IGVMainFrame.*;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridLayout;
import java.awt.event.MouseEvent;
import java.awt.geom.AffineTransform;

import java.util.List;

import javax.swing.JPanel;
import javax.swing.event.MouseInputAdapter;
import org.broad.igv.ui.util.Packable;

/**
 *
 * @author jrobinso
 */
public class AttributeHeaderPanel extends JPanel implements Packable {

    private int attributeCount = 0;
    final static int MAXIMUM_FONT_SIZE = 10;
    final static int ATTRIBUTE_COLUMN_WIDTH = 10;
    final static int COLUMN_BORDER_WIDTH = 2;

    /**
     * Constructs ...
     *
     */
    public AttributeHeaderPanel() {
        setBorder(javax.swing.BorderFactory.createLineBorder(Color.black));
    }

    @Override
    protected void paintComponent(final Graphics graphics) {

        super.paintComponent(graphics);

        final Dimension size = getSize();

        // Don't want to destroy the components original graphics
        // context because of the border so we need to create one
        final Graphics2D graphics2 = (Graphics2D) graphics.create();
        graphics2.setColor(getBackground());
        graphics2.fillRect(0, 0, getWidth(), getHeight());
        graphics2.setColor(Color.BLACK);

        final List<String> keys = getInstance().getDisplayableAttributes();

        // Divide the remaining space to get column widths
        int columnWidth = getAttributeColumnWidth();

        // Create font and font size
        int fontSize = (int) (0.9 * columnWidth);
        if (fontSize > MAXIMUM_FONT_SIZE)
        {
            fontSize = MAXIMUM_FONT_SIZE;
        }
        Font font = FontManager.getScalableFont(fontSize);

        final int count = keys.size();
        if (count > 0)
        {

            if (attributeCount != count)
            {

                removeAll();    // Remove Mouse Regions
                if (keys != null)
                {
                    setLayout(new GridLayout(1, count));
                }
            }

            // Change the origin for the text
            AffineTransform transform = AffineTransform.getTranslateInstance(0,
                                            size.height - COLUMN_BORDER_WIDTH);
            graphics2.transform(transform);

            // Now rotate text counter-clockwise 90 degrees
            transform = AffineTransform.getRotateInstance(-Math.PI / 2);
            graphics2.transform(transform);
            graphics2.setFont(font);
            FontMetrics fm = graphics2.getFontMetrics();
            int fontAscent = fm.getHeight();

            int i = 1;
            int x = 0;
            for (String key : keys)
            {
                int columnLeftEdge = ((COLUMN_BORDER_WIDTH + ATTRIBUTE_COLUMN_WIDTH) * i++);
                x = columnLeftEdge
                    + ((COLUMN_BORDER_WIDTH + ATTRIBUTE_COLUMN_WIDTH) - fontAscent) / 2;
                graphics2.drawString(key, 0, x);
            }
        }

        GuiUtilities.invokeOnEventThread(new Runnable() {

            public void run() {

                for (String key : keys)
                {

                    if (attributeCount != count)
                    {
                        addMousableRegion(key);
                    }
                }

                if (attributeCount != count)
                {
                    attributeCount = count;
                    doLayout();
                }
            }
        });

    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getAttributeColumnWidth() {
        return ATTRIBUTE_COLUMN_WIDTH;
    }




    /**
     * Method description
     *
     */
    public void packHorizontally() {

        int newWidth = calculatePackWidth();

        Dimension dimension = getSize();
        dimension = new Dimension(newWidth, dimension.height);
        setMinimumSize(dimension);
        setMaximumSize(dimension);
        setSize(dimension);
        setPreferredSize(dimension);
    }

    private int calculatePackWidth() {

        if (!PreferenceManager.getInstance().getShowAttributeView())
        {
            return 0;
        }

        List<String> attributeKeys = getInstance().getDisplayableAttributes();
        int numberOfAttributes = attributeKeys.size();
        int packWidth = (numberOfAttributes) * (ATTRIBUTE_COLUMN_WIDTH + COLUMN_BORDER_WIDTH)
                        + COLUMN_BORDER_WIDTH;
        return packWidth;
    }


    /**
     * Method description
     *
     */
    public void packComponent() {


        int newWidth = calculatePackWidth();

        Dimension dimension = getSize();
        dimension = new Dimension(newWidth, dimension.height);
        setMinimumSize(dimension);
        setMaximumSize(dimension);
        setSize(dimension);
        setPreferredSize(dimension);

    }

    /**
     * Method description
     *
     *
     * @param x
     * @param y
     * @param width
     * @param height
     */
    @Override
    public void setBounds(int x, int y, int width, int height) {
        super.setBounds(x, y, calculatePackWidth(), height);
    }

    private static class Region extends JPanel {

        /**
         * Method description
         *
         *
         * @return
         */
        @Override
        public boolean isOpaque() {
            return false;
        }
    }


    private boolean isSortAscending = true;
    private void addMousableRegion(final String tooltip) {

        // Create a clickable region
        final Region region = new Region();
        region.setName(tooltip);
        region.setToolTipText(tooltip);

        MouseInputAdapter listener = new MouseInputAdapter() {

            @Override
            public void mouseClicked(MouseEvent e) {

                String[] selectedAttribute = { region.getName() };
                boolean[] sortAscending = { isSortAscending };
                sortTrackByAttribute(selectedAttribute, sortAscending);
                isSortAscending = !isSortAscending;
            }

            @Override
            public void mouseEntered(MouseEvent e) {

                region.setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));

            }

            @Override
            public void mouseExited(MouseEvent e) {}

            @Override
            public void mouseReleased(MouseEvent e) {}
        };
        region.addMouseMotionListener(listener);
        region.addMouseListener(listener);
        add(region);
    }


    /**
     * Method description
     *
     *
     * @param selectedSortKeys
     * @param isSortAscending
     */
    final public void sortTrackByAttribute(String[] selectedSortKeys, boolean[] isSortAscending) {

        if (selectedSortKeys != null)
        {

            TrackManager.getInstance().sortAllTracksByAttributes(selectedSortKeys, isSortAscending);
            IGVMainFrame.getInstance().fireViewChangedEvent();
            IGVMainFrame.getInstance().getContentPane().repaint();
        }
    }

}
