/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ui.action;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;

import org.broad.igv.feature.Chromosome;
import org.broad.igv.feature.Feature;
import org.broad.igv.feature.FeatureDB;
import org.broad.igv.ui.ViewContext;

//~--- JDK imports ------------------------------------------------------------


import javax.swing.JOptionPane;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.LongRunningTask;

/**
 * A class for performing search actions.  The class takes a view context and
 * search string as parameters.   The search string can be either
 *   (a) a feature (e.g. gene),  or
 *   (b) a locus string in the UCSC form,  e.g. chr1:100,000-200,000
 *
 * Note:  Currently the only recognized features are genes
 *
 * @author jrobinso
 */
public class SearchCommand implements Command {

    private static Logger logger = Logger.getLogger(SearchCommand.class);
    String searchString;
    ViewContext viewContext;

    /**
     * Constructs ...
     *
     *
     * @param viewContext
     * @param searchString
     */
    public SearchCommand(ViewContext viewContext, String searchString) {
        this.viewContext = viewContext;
        this.searchString = searchString.trim();
    }

    /**
     * Method description
     *
     */
    public void execute() {

        // The search string can be a feature (e.g. gene) name or a locus
        // description.  Try feature name first

        LongRunningTask.submit(new Runnable() {

            public void run() {
                Feature feature = FeatureDB.getFeature(searchString.toUpperCase().trim());
                if (feature != null) {
                    int start = Math.max(0, (int) feature.getStart());
                    int end = (int) feature.getEnd();
                    viewContext.jumpTo(feature.getChromosome(), start, end);
                    return;
                }

                // Apparently not a feature. Either a locus or track name.  Track names can be quoted,
                // loci are never quoted.

                if (!searchString.contains("\"")) {
                    String chr = null;
                    int[] startEnd = null;
                    int colon = searchString.indexOf(":");

                    if (colon > 0) {

                        // The chromosome is that portion of the search string up to the colon.
                        chr = searchString.substring(0, colon);
                        String posString = searchString.substring(colon).replace(":", "");
                        startEnd = getStartEnd(posString);

                        if (startEnd != null) {
                            viewContext.jumpTo(chr, startEnd[0], startEnd[1]);
                            return;
                        }
                    } else {

                        // No chromosome delimiter (color),  The search string is either chromosome name
                        // or a locus in the current chromosome.
                        if (searchString.contains("-")) {

                            // Presense of a dash indicates this is a locus string in the current chromosome
                            startEnd = getStartEnd(searchString);
                            if (startEnd != null) {
                                viewContext.jumpTo(null, startEnd[0], startEnd[1]);
                                return;
                            }
                        } else {

                            // No dash, this is either a chromosome or an unkown search string
                            Chromosome chromosome = viewContext.getGenome().getChromosome(searchString);
                            if (chromosome != null) {
                                viewContext.setChromosomeName(searchString, true);
                                return;
                            }
                        }
                    }
                }

                if (!IGVMainFrame.getInstance().scrollToTrack(searchString.replaceAll("\"", ""))) {
                    showError("Cannot find feature or locus: " + searchString);
                }
            }
        });


    }

    /**
     * Return the start and end positions as a 2 element array for the input
     * position string.  UCSC conventions  are followed for coordinates,
     * specifically the internal representation is "zero" based (first base is
     * numbered 0) but the display representation is "one" based (first base is
     * numbered 1).   Consequently 1 is substracted from the parsed positions
     *
     */
    private int[] getStartEnd(String posString) {
        try {
            String[] posTokens = posString.split("-");
            String startString = posTokens[0].replaceAll(",", "");
            int start = Math.max(0, Integer.parseInt(startString)) - 1;

            // Default value for end
            int end = start + 1;
            if (posTokens.length >= 2) {
                String endString = posTokens[1].replaceAll(",", "");

                // Add 1 bp to end position t make it "inclusive"
                end = Integer.parseInt(endString);
            }

            return new int[]{Math.min(start, end), Math.max(start, end)};
        } catch (NumberFormatException numberFormatException) {
            return null;
        }

    }

    /**
     *  TODO -- should this be here?  If not here where?
     * @param message
     */
    private void showError(String message) {
        JOptionPane.showMessageDialog(null, message);
    }
}
