/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ui.action;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.PrintWriter;
import java.util.Collection;
import javax.swing.Icon;
import org.apache.log4j.Logger;
import org.broad.igv.IGVConstants;
import org.broad.igv.PreferenceManager;
import org.broad.igv.ui.util.FileChooser;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.IGVModel;
import org.broad.igv.ui.RegionOfInterest;

/**
 *
 * @author jrobinso
 */
public class RegionsBaseMenuAction extends MenuAction {

    static Logger log = Logger.getLogger(RegionsBaseMenuAction.class);
    
    IGVMainFrame mainFrame;

    protected enum Direction {

        IMPORT, EXPORT

    };

    public   RegionsBaseMenuAction(String name, Icon icon  ,
           int mnemonic) {
        super(name, icon, mnemonic);
    }

    public final void importExportRegionsOfInterest(Direction direction) {
        File exportRegionDirectory = PreferenceManager.getInstance().getLastExportedRegionDirectory();
        if (exportRegionDirectory == null) {
            exportRegionDirectory = new File(IGVConstants.getDefaultUserDirectory());
        }
        
        FileChooser exportedRegionFileChooser = new FileChooser(exportRegionDirectory);
        String title = null;
        if (direction == Direction.EXPORT) {
            title = "Export Regions of Interest ...";
        } else {
            title = "Import Regions of Interest ...";
        }
        exportedRegionFileChooser.setDialogTitle(title);
        File file = selectExportedRegionsFile(exportedRegionFileChooser, new File("regions.bed"), direction == Direction.EXPORT);

        if (file == null) {
            return;
        }

        // Read or write the ROI file
        if (direction == Direction.EXPORT) {
            writeRegionsOfInterestFile(file);
        } else {
            readRegionsOfInterestFile(file);
        }
    }

    private void readRegionsOfInterestFile(File roiFile) {

        if (roiFile == null) {
            log.info("A blank Region of Interest import file was supplied!");
            return;
        }

        if (!roiFile.exists()) {
            org.broad.igv.ui.util.UIUtilities.showErrorMessage(mainFrame, "Region of Interest export file not found!");
            return;
        }
        try {
            BufferedReader reader = null;
            try {
                reader = new BufferedReader(new FileReader(roiFile));
                while (true) {
                    String dataRecord = reader.readLine();
                    if (dataRecord == null) {
                        return;
                    }
                    String[] data = dataRecord.split("\t");
                    if (data.length >= 3) {
                        try {
                            RegionOfInterest regionOfInterest = new RegionOfInterest(data[0], Integer.parseInt(data[1]), Integer.parseInt(data[2]), "ROI");
                            mainFrame.addRegionOfInterest(regionOfInterest);
                        } catch (NumberFormatException numberFormatException) {
                        }
                    }
                }
            } finally {

                if (reader != null) {
                    reader.close();
                }
                mainFrame.doRefresh();
            }
        } catch (Exception e) {
            log.error("Failed to write Region of Interest export file!", e);
        }
    }

    private File selectExportedRegionsFile(FileChooser exportedRegionFileChooser, File currentFile, boolean isSave) {

        exportedRegionFileChooser.setSelectedFile(currentFile);

        // Display the dialog
        if (isSave) {
            exportedRegionFileChooser.showSaveDialog(mainFrame);
        } else {
            exportedRegionFileChooser.showOpenDialog(mainFrame);
        }

        mainFrame.resetStatusMessage();
        File file = exportedRegionFileChooser.getSelectedFile();
        if (file != null) {
            File directory = exportedRegionFileChooser.getCurrentDirectory();
            if (directory != null) {
                PreferenceManager.getInstance().setLastExportedRegionDirectory(directory);
            }
        }

        return file;
    }

    private void writeRegionsOfInterestFile(File roiFile) {

        if (roiFile == null) {
            log.info("A blank Region of Interest export file was supplied!");
            return;
        }
        try {
            Collection<RegionOfInterest> regions = IGVModel.getInstance().getAllRegionsOfInterest();

            if (regions == null || regions.isEmpty()) {
                return;
            }

            // Create export file
            roiFile.createNewFile();
            PrintWriter writer = null;
            try {
                writer = new PrintWriter(roiFile);
                for (RegionOfInterest regionOfInterest : regions) {
                    Integer regionStart = regionOfInterest.getStart();
                    if (regionStart == null) {
                        // skip - null starts are bad regions of interest
                        continue;
                    }
                    Integer regionEnd = regionOfInterest.getEnd();
                    if (regionEnd == null) {
                        regionEnd = regionStart;
                    }

                    // Write info in BED format
                    writer.print(regionOfInterest.getChromosomeName());
                    writer.print("\t");
                    writer.print(regionStart);
                    writer.print("\t");
                    writer.println(regionEnd);
                }
            } finally {

                if (writer != null) {
                    writer.close();
                }
            }
        } catch (Exception e) {
            log.error("Failed to write Region of Interest export file!", e);
        }
    }
}
