/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ui.action;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;

import org.broad.igv.PreferenceManager;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.util.Utilities;
import org.broad.igv.track.AttributeManager;
import org.broad.igv.track.TrackManager;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.ResourceCheckBoxList;
import org.broad.igv.ui.ResourceCheckBoxList.ResourceCheckBox;
import org.broad.igv.ui.ResourceTree;
import org.broad.igv.ui.UIConstants;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import org.xml.sax.SAXParseException;

//~--- JDK imports ------------------------------------------------------------

import java.awt.event.ActionEvent;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;

import java.net.ConnectException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.UnknownHostException;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import javax.swing.JOptionPane;
import javax.swing.JScrollPane;

import javax.xml.parsers.DocumentBuilderFactory;
import org.broad.igv.ui.GuiUtilities;
import org.broad.igv.util.ConnectionTimerTask;

/**
 *
 * @author jrobinso
 */
public class LoadFromServerAction extends MenuAction {

    static Logger log = Logger.getLogger(LoadFromServerAction.class);
    IGVMainFrame mainFrame;

    /**
     * Constructs ...
     *
     *
     * @param label
     * @param mnemonic
     * @param mainFrame
     */
    public LoadFromServerAction(String label, int mnemonic, IGVMainFrame mainFrame) {
        super(label, null, mnemonic);
        this.mainFrame = mainFrame;
        setToolTipText(UIConstants.LOAD_SERVER_DATA_TOOLTIP);
    }

    /**
     * Method description
     *
     *
     *
     * @param evt
     */
    @Override
    public void actionPerformed(ActionEvent evt) {

        GuiUtilities.invokeOnEventThread(new Runnable() {

            public void run() {
                try {
                    mainFrame.setStatusBarMessage("Loading ...");

                    try {

                        URL masterResourceFileURL =
                                new URL(PreferenceManager.getInstance().getDataServerURL());

                        URLConnection conn = masterResourceFileURL.openConnection();
                        conn.setReadTimeout(5000);
                        ConnectionTimerTask timerTask = new ConnectionTimerTask(conn);
                        (new java.util.Timer()).schedule(timerTask, 6000);

                        BufferedReader bufferedReader =
                                new BufferedReader(
                                new InputStreamReader(conn.getInputStream()));
                        LinkedHashSet<URL> urls = getResourceUrls(bufferedReader);
                        timerTask.cancel();


                        List<ResourceLocator> locators = selectResources(urls);
                        if (locators != null) {
                            mainFrame.loadTracks(locators);
                        }


                    } catch (UnknownHostException e) {
                        org.broad.igv.ui.util.UIUtilities.showErrorMessage(
                                mainFrame,
                                "The application could not contact the server: Unknown Host Failure!\nHost: " + e.getMessage());
                    } catch (ConnectException e) {
                        org.broad.igv.ui.util.UIUtilities.showErrorMessage(
                                mainFrame,
                                "The application could not contact the server: Connection Failure!\n" + e.getMessage());
                    } catch (Exception e) {
                        org.broad.igv.ui.util.UIUtilities.showErrorMessage(mainFrame,
                                "Could not load information from server!\n" + e.getMessage());
                    }
                } finally {
                    mainFrame.showLoadedTrackCount();
                }

            }
        });
    }

    /**
     * Method description
     *
     *
     * @param xmlUrls
     *
     * @return
     */
    public List<ResourceLocator> selectResources(final LinkedHashSet<URL> xmlUrls) {

        if ((xmlUrls == null) || xmlUrls.isEmpty()) {
            log.error("No valid xml resource from which to load");
            return null;
        }

        StringBuffer buffer = new StringBuffer();
        boolean xmlParsingError = false;
        try {

            buffer.append(
                    "<html>The following urls could not be processed due to load failures:<br>");

            Document masterDocument =
                    DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();

            Element rootNode = masterDocument.createElement("Global");
            rootNode.setAttribute("name", "Available Datasets");
            rootNode.setAttribute("version", "1");

            masterDocument.appendChild(rootNode);

            // Merge all documents into one xml document for processing
            for (URL url : xmlUrls) {

                try {

                    Document xmlDocument = null;
                    try {
                        xmlDocument = Utilities.createDOMDocumentFromXmlStream(url.openStream());
                    } catch (SAXParseException e) {
                        log.error("Invalid XML resource: " + url, e);

                        xmlParsingError = true;
                        buffer.append(url);
                        buffer.append("<br><i>");
                        if (url.toString().contains("iwww.broad")) {
                            buffer.append("File could not be loaded from the Broad Intranet");
                        } else {
                            buffer.append(e.getMessage());
                        }
                        buffer.append("");
                        continue;
                    } catch (FileNotFoundException e) {

                        String message = "Could not find file represented by " + url.toString();
                        log.error(message, e);

                        xmlParsingError = true;
                        buffer.append(url);
                        buffer.append("\t  [");
                        buffer.append(e.getMessage());
                        buffer.append("]\n");
                        continue;
                    }

                    NodeList elements = xmlDocument.getElementsByTagName("Global");
                    Element global = (Element) elements.item(0);
                    NodeList nodes = global.getChildNodes();
                    Element categoryNode = masterDocument.createElement("Category");
                    categoryNode.setAttribute("name", global.getAttribute("name"));
                    categoryNode.setAttribute("hyperlink", global.getAttribute("hyperlink"));
                    rootNode.appendChild(categoryNode);
                    int size = nodes.getLength();
                    for (int i = 0; i < size; i++) {
                        categoryNode.appendChild(masterDocument.importNode(nodes.item(i), true));
                    }
                } catch (Exception e) {
                    String message = "Cannot create an XML Document from " + url.toString();
                    log.error(message, e);
                    continue;
                }
            }
            if (xmlParsingError) {
                JOptionPane.showMessageDialog(mainFrame, buffer.toString());
            }

            /**
             * Resource Tree
             */
            HashSet<ResourceLocator> selectedLocators =
                    ResourceTree.getInstance().showResourceTreeDialog(mainFrame, masterDocument,
                    "Available Datasets");

            List<ResourceLocator> newLoadList = new ArrayList();

            Set<ResourceLocator> loadedResources =
                    TrackManager.getInstance().getDataResourceLocators();
            loadedResources.addAll(AttributeManager.getInstance().getLoadedResources());

            for (ResourceLocator locator : selectedLocators) {

                // Don't reload data that is already loaded
                if (loadedResources.contains(locator)) {
                    continue;
                }

                newLoadList.add(locator);
            }
            return newLoadList;
        } catch (Exception e) {
            log.error("Could not load information from server", e);
            return null;
        } finally {
            if (xmlParsingError) {
                log.error(buffer.toString());
            }
        }
    }

    /**
     * Returns the complete list of URLs from the master registry file.
     *
     * @param bufferedReader
     * @return
     * @throws java.net.MalformedURLException
     * @throws java.io.IOException
     */
    private LinkedHashSet<URL> getResourceUrls(BufferedReader bufferedReader)
            throws MalformedURLException, IOException {

        LinkedHashSet<URL> xmlFileUrls = new LinkedHashSet();
        while (true) {
            String xmlFileUrl = bufferedReader.readLine();
            if ((xmlFileUrl == null) || (xmlFileUrl.trim().length() == 0)) {
                break;
            }
            xmlFileUrl = xmlFileUrl.trim();
            xmlFileUrls.add(new URL(xmlFileUrl));
        }

        return xmlFileUrls;
    }

    /**
     * Popups a dialog from which the user can select one or more XML resource
     * files.  Not currelent used.
     *
     * @param bufferedReader
     * @return
     * @throws java.net.MalformedURLException
     * @throws java.io.IOException
     */
    private Set<URL> getResourceUrlsFromDialog(BufferedReader bufferedReader)
            throws MalformedURLException, IOException {

        List<ResourceCheckBox> xmlFileUrls = new ArrayList();

        while (true) {
            String xmlFileUrl = bufferedReader.readLine();
            if (xmlFileUrl == null) {
                break;
            }
            xmlFileUrl = xmlFileUrl.trim();

            ResourceCheckBox checkbox = new ResourceCheckBox(xmlFileUrl, new URL(xmlFileUrl));
            xmlFileUrls.add(checkbox);
        }

        Object[] data = xmlFileUrls.toArray();
        ResourceCheckBoxList list = new ResourceCheckBoxList(data);

        int status = JOptionPane.showConfirmDialog(mainFrame, new JScrollPane(list),
                "Resource File List", JOptionPane.OK_CANCEL_OPTION,
                JOptionPane.PLAIN_MESSAGE, null);

        if ((status == JOptionPane.CANCEL_OPTION) || (status == JOptionPane.CLOSED_OPTION)) {
            return new HashSet<URL>();
        }

        return list.getSelectedURLs();
    }
}
