/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ui.action;

import com.jidesoft.utils.SwingWorker;
import java.awt.event.ActionEvent;
import java.util.Collection;
import java.util.List;
import org.apache.log4j.Logger;
import org.broad.igv.track.Track;
import org.broad.igv.track.TrackGroup;
import org.broad.igv.ui.panel.DataPanel;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.UIConstants;
import org.broad.igv.ui.panel.TrackSetScrollPane;

/**
 *
 * @author jrobinso
 */
public class FitDataToWindowMenuAction extends MenuAction {

    static Logger log = Logger.getLogger(FitDataToWindowMenuAction.class);
    IGVMainFrame mainFrame;

    public FitDataToWindowMenuAction(String label, int mnemonic, IGVMainFrame mainFrame) {
        super(label, null, mnemonic);
        this.mainFrame = mainFrame;
        setToolTipText(UIConstants.FIT_DATA_TO_WINDOW_TOOLTIP);
    }

    @Override
    /**
     * The action method. A swing worker is used, so "invoke later" and explicit
     * threads are not neccessary.
     * 
     */
    public void actionPerformed(ActionEvent e) {

        Worker worker = new Worker();
        worker.execute();
    }

    /**
     * Adjust the height of all tracks so that all tracks fit in the available
     * height of the panel.  This is not possible in all cases as the 
     * minimum height for a track is 1 pixel.  
     * 
     * @param dataPanel
     * @return
     */
    private boolean fitTracksToPanel(DataPanel dataPanel) {

        boolean success = true;

        int visibleHeight = dataPanel.getVisibleHeight();
        int visibleTrackCount = 0;
        int geneTrackHeight = 0;

        // Process data tracks first
        Collection<TrackGroup> groups = dataPanel.getTrackGroups();

        // Count visible tracks and note gene track 'was found' and ist height
        for (TrackGroup group : groups)
        {
            List<Track> tracks = group.getTracks();
            for (Track track : tracks)
            {
                if (track.isVisible())
                {
                    ++visibleTrackCount;
                }
            }
        }

        // Auto resize the height of the visible tracks
        if (visibleTrackCount > 0)
        {

            int adjustedVisibleHeight = visibleHeight;

            if (adjustedVisibleHeight > 0)
            {

                int newTrackHeight = adjustedVisibleHeight / visibleTrackCount;

                // If the new track height is less than 1 theres nothing we 
                // can do to force all tracks to fit so we do nothing
                if (newTrackHeight < 1)
                {
                    newTrackHeight = 1;
                }

                for (TrackGroup group : groups)
                {
                    List<Track> tracks = group.getTracks();
                    for (Track track : tracks)
                    {
                        if (track.isVisible())
                        {
                            track.setHeight(newTrackHeight);
                        }
                    }
                }

                // Extra space to fill in
                int extraPixels = adjustedVisibleHeight % visibleTrackCount;

                for (TrackGroup group : groups)
                {
                    List<Track> tracks = group.getTracks();
                    for (Track track : tracks)
                    {
                        if (extraPixels == 0)
                        {
                            break;
                        }
                        if (track.isVisible())
                        {
                            track.setHeight(newTrackHeight + 1);
                            --extraPixels;
                        }
                    }
                }
            }
        }

        return success;
    }

    /**
     * The worker class for this action.
     * 
     * @param String
     * @param Void
     */
    class Worker<String, Void> extends SwingWorker {

        boolean success;

        @Override
        /**
         * This method is executed in a background thread (NOT the swing
         * thread).
         */
        protected Object doInBackground() throws Exception {
            // Process Data and Feature Tracks
            success = false;
            for (TrackSetScrollPane sp : IGVMainFrame.getInstance().getTrackSetScrollPanes())
            {
                success = success || fitTracksToPanel(sp.getDataPanel());
            }

            return success;

        }

        @Override
        /**
         * This method is executed in the swing thread when doInBackground 
         * completes.
         */
        protected void done() {
            mainFrame.clearImageCacheWithNoRepaint();
            mainFrame.updateTrackState();

        }
    }
}
