/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ui.action;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Collections;
import java.util.List;
import javax.swing.ButtonGroup;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;
import org.apache.log4j.Logger;
import org.broad.igv.util.Filter;
import org.broad.igv.track.AttributeManager;
import org.broad.igv.track.Track;
import org.broad.igv.track.TrackManager;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.TrackFilter;
import org.broad.igv.ui.TrackFilterPane;
import org.broad.igv.ui.UIConstants;

/**
 *
 * @author jrobinso
 */
public class FilterTracksMenuAction extends MenuAction {

    //static Logger log = Logger.getLogger(FilterTracksMenuAction.class);
    IGVMainFrame mainFrame;

    // Current track filter
    private TrackFilter trackFilter;

    private JCheckBox showAllTracksFilterCheckBox = new JCheckBox();

    private JCheckBox matchAllCheckBox = new JCheckBox();

    private JCheckBox matchAnyCheckBox = new JCheckBox();

    private TrackFilterPane trackFilterPane;

    public FilterTracksMenuAction(String label, int mnemonic, IGVMainFrame mainFrame) {
        super(label, null, mnemonic);
        this.mainFrame = mainFrame;
        setToolTipText(UIConstants.FILTER_TRACKS_TOOLTIP);
    }

    @Override
    public void actionPerformed(ActionEvent e) {
        doFilterTracks();
    }

    private void doFilterTracks() {

        boolean previousDisableFilterState =
                showAllTracksFilterCheckBox.isSelected();

        boolean previousMatchAllState =
                matchAllCheckBox.isSelected();

        List<String> uniqueAttributeKeys =
                AttributeManager.getInstance().getAttributeKeys();

        // Sort the attribute keys if we have any
        if (uniqueAttributeKeys != null) {
            Collections.sort(uniqueAttributeKeys,
                    AttributeManager.getInstance().getAttributeComparator());
        } else // If we have no attribute we can't display the
        // track filter dialog so say so and return
        if (uniqueAttributeKeys == null ||
                uniqueAttributeKeys.isEmpty()) {

            JOptionPane.showMessageDialog(mainFrame,
                    "No attributes found to use in a filter");
            return;
        }

        if (trackFilterPane == null) {

            trackFilterPane = new TrackFilterPane(uniqueAttributeKeys, "Show tracks whose attribute", trackFilter);

        } else {

            trackFilterPane.setItems(uniqueAttributeKeys);

            // Backup the initial state for restores
            trackFilterPane.backup();
            Filter filter = trackFilterPane.getFilter();
            if (filter == null || filter.isEmpty()) {
                trackFilterPane.more();
            }
        }

        trackFilterPane.clearTracks();
        trackFilterPane.addTracks(TrackManager.getInstance().getAllTracks(false));

        while (true) {

            Integer response =
                    createFilterTrackDialog(mainFrame, trackFilterPane,
                    "Filter Tracks");

            if (response == null) {
                continue;
            }

            if (response.intValue() == JOptionPane.CANCEL_OPTION) {

                // Restore previous filter state
                boolean disableFilterState =
                        showAllTracksFilterCheckBox.isSelected();
                if (disableFilterState != previousDisableFilterState) {
                    showAllTracksFilterCheckBox.setSelected(previousDisableFilterState);
                }

                // Restore previous boolean match state
                boolean matchAllState = matchAllCheckBox.isSelected();
                if (matchAllState != previousMatchAllState) {
                    matchAllCheckBox.setSelected(previousMatchAllState);
                    matchAnyCheckBox.setSelected(!previousMatchAllState);
                }
                // Reset state
                trackFilterPane.restore();
                return;
            } else if (response.intValue() == JOptionPane.OK_OPTION) {

                configureFilterStateOfTracks(trackFilterPane);
                break;
            }
        }

        // Update the state of the current tracks
        mainFrame.updateTrackState();

    }

    private Integer createFilterTrackDialog(Frame parent,
            final TrackFilterPane trackFilterPane,
            String title) {

        JScrollPane scrollPane = new JScrollPane(trackFilterPane);
        scrollPane.setVerticalScrollBarPolicy(
                JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
        scrollPane.setHorizontalScrollBarPolicy(
                JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);

        int optionType = JOptionPane.OK_CANCEL_OPTION;
        int messageType = JOptionPane.PLAIN_MESSAGE;

        JPanel panel = new JPanel();
        panel.setLayout(new BorderLayout());

        JPanel filterHeaderPanel = new JPanel();
        filterHeaderPanel.setBackground(Color.WHITE);
        filterHeaderPanel.setLayout(new GridLayout(0, 1));
        filterHeaderPanel.add(new JLabel("For attributes that:"));

        ButtonGroup booleanButtonGroup = new ButtonGroup();
        booleanButtonGroup.add(matchAllCheckBox);
        booleanButtonGroup.add(matchAnyCheckBox);

        showAllTracksFilterCheckBox.setText("Show All Tracks");
        matchAllCheckBox.setText("Match all of the following");
        matchAnyCheckBox.setText("Match any of the following");
        boolean matchAll = trackFilterPane.getMatchAll();
        if (matchAll) {
            matchAllCheckBox.setSelected(true);
        } else {
            matchAnyCheckBox.setSelected(true);
        }

        matchAllCheckBox.addActionListener(new java.awt.event.ActionListener() {

            public void actionPerformed(java.awt.event.ActionEvent evt) {
                trackFilterPane.setMatchAll(true);
            }
        });
        matchAnyCheckBox.addActionListener(new java.awt.event.ActionListener() {

            public void actionPerformed(java.awt.event.ActionEvent evt) {
                trackFilterPane.setMatchAll(false);
            }
        });

        JPanel controls = new JPanel();
        FlowLayout layoutManager = new FlowLayout();
        layoutManager.setAlignment(FlowLayout.LEFT);
        controls.setLayout(layoutManager);
        controls.add(matchAllCheckBox);
        controls.add(matchAnyCheckBox);
        controls.add(showAllTracksFilterCheckBox);
        controls.setBackground(Color.WHITE);
        controls.setOpaque(true);
        filterHeaderPanel.add(controls);

        panel.setOpaque(true);
        panel.add(filterHeaderPanel, BorderLayout.NORTH);
        panel.add(scrollPane, BorderLayout.CENTER);

        final JOptionPane optionPane =
                new JOptionPane(panel, messageType, optionType);

        optionPane.setPreferredSize(new Dimension(600, 400));
        optionPane.setOpaque(true);
        optionPane.setBackground(Color.WHITE);
        optionPane.addPropertyChangeListener(JOptionPane.VALUE_PROPERTY,
                new PropertyChangeListener() {

                    public void propertyChange(PropertyChangeEvent e) {

                        Object value = e.getNewValue();
                        if (value instanceof Integer) {

                            int option = (Integer) value;
                            if (option == JOptionPane.OK_OPTION) {

                                if (trackFilterPane.isFilterValid())
                                {
                                    trackFilterPane.applyFilterMatching();                                    
                                    trackFilterPane.save();
                                }
                            }
                        }
                    }
                });

        JDialog dialog = optionPane.createDialog(parent, title);
        dialog.setBackground(Color.WHITE);
        dialog.getContentPane().setBackground(Color.WHITE);

        Dimension maximumSize = new Dimension(dialog.getSize().width, 100);
        if (maximumSize != null) {
            dialog.setMaximumSize(maximumSize);
        }

        Component[] children = optionPane.getComponents();
        if (children != null) {
            for (Component child : children) {
                child.setBackground(Color.WHITE);
            }

        }

        dialog.pack();
        dialog.setVisible(true);

        Object selectedValue = optionPane.getValue();
        if (selectedValue == null) {
            return JOptionPane.CANCEL_OPTION;
        } else if (((Integer) selectedValue).intValue() == JOptionPane.OK_OPTION) {
            if (!trackFilterPane.isFilterValid() &&
                    !showAllTracksFilterCheckBox.isSelected()) {
                JOptionPane.showMessageDialog(
                        parent,
                        "Some of the filter values are missing." +
                        "\nPlease enter all value before pressing ok.");

                selectedValue =
                        null;
            }

        }

        return ((Integer) selectedValue);
    }

    private void configureFilterStateOfTracks(TrackFilterPane trackFilterPane) {

        boolean showAllTracks = showAllTracksFilterCheckBox.isSelected();
        if (showAllTracks) {

            List<Track> tracks = TrackManager.getInstance().getAllTracks(false);
            for (Track track : tracks) {
                track.setVisible(showAllTracks);
            }

        } else {
            trackFilter = trackFilterPane.getFilter();
            // Evaluate the filter elements
            trackFilter.evaluate();
        }

    }

    public void resetTrackFilter() {
        trackFilterPane = null;
        trackFilter = null;
        setFilterShowAllTracks(false);
    }

    public void setFilterShowAllTracks(boolean value) {
        if (showAllTracksFilterCheckBox != null) {
            showAllTracksFilterCheckBox.setSelected(value);
        }
    }

    public boolean isFilterShowAllTracks() {
        if (showAllTracksFilterCheckBox != null) {
            return showAllTracksFilterCheckBox.isSelected();
        }
        return false;
    }

    public JCheckBox getShowAllTracksFilterCheckBox() {
        return showAllTracksFilterCheckBox;
    }

    public TrackFilter getTrackFilter() {

        TrackFilter filter = null;
        if (trackFilterPane != null) {
            filter = trackFilterPane.getFilter();
        }
        return filter;
    }

    public void setTrackFilter(TrackFilter trackFilter) {

        if (trackFilter == null) {
            return;
        }

        List<String> uniqueAttributeKeys =
                AttributeManager.getInstance().getAttributeKeys();


        // If we have no attribute we can't display the
        // track filter dialog so say so and return
        if (uniqueAttributeKeys == null ||
                uniqueAttributeKeys.isEmpty()) {

            JOptionPane.showMessageDialog(mainFrame,
                    "No attributes found to use in a filter");
            return;
        }

        trackFilterPane = new TrackFilterPane(uniqueAttributeKeys, "Show tracks whose attribute", trackFilter);

        trackFilterPane.clearTracks();
        trackFilterPane.addTracks(TrackManager.getInstance().getAllTracks(false));

        // Evaluate the filter elements
        trackFilter.evaluate();
    }

    public void setFilterMatchAll(boolean value) {
        if (trackFilterPane != null) {
            trackFilterPane.setMatchAll(value);
        }
    }

    public boolean isFilterMatchAll() {
        if (trackFilterPane != null) {
            return trackFilterPane.getMatchAll();
        }
        return false;
    }
}
