/*
* The Broad Institute
* SOFTWARE COPYRIGHT NOTICE AGREEMENT
* This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
* of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
* (the "License"); you may not use this file except in compliance with
*  the License.  You may obtain a copy of the License at
*
*    http://www.opensource.org/licenses/gpl-2.0.php
*
* This software is supplied without any warranty or guaranteed support
* whatsoever. Neither the Broad Institute nor MIT can be responsible for its
* use, misuse, or functionality.
*/







package org.broad.igv.ui;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Cursor;

import java.util.*;


/**
 * Utility class for managing IGV cursors.  The main purpose of the class is to centrally manage
 * a global wait cursor.  When in "wait" mode component set cursor events are ignored, or rather
 * saved in a cached until the wait cursor is removed.
 *
 * @author jrobinso
 */
public class WaitCursorManager {


    /**
     * A set of tokens, one for each call to "showWaitCursor".  These are removed in the
     * "removeWaitCursor" method.  The presence of a token in this list indicates that IGV is
     * in the wait state.
     */
    static Set<CursorToken> tokens = new HashSet();

    /**
     * The wait cursor, defined statically for convenience.
     */
    static Cursor waitCursor = Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR);

    /**
     * Show the wait cursor on all components.  Add a token to represent this invocation of
     * showWaitCursor
     *
     * @return  token representing this invocation.  This token should be used by clients to remove
     * the wait cursr.  This should be done in a finally block to insure removal.
     */
    public static CursorToken showWaitCursor() {
        synchronized (tokens)
        {
            IGVMainFrame.getInstance().getGlassPane().setVisible(true);
            CursorToken token = new CursorToken();
            tokens.add(token);
            // Return a token representing this wait cursor set.  The token is used to release the
            // wait cursor.
            return token;
        }
    }

    /**
     * Remove the token for a showWaitCursor() invocation.  This indicates that the client has completed
     * its task and removed the wait cursor request.  If the last token has been removed reset
     * the cursors on the components to their last requested cursor, or the default cursor if
     * there are no outstanding requests.
     *
     * @param token
     */
    public static void removeWaitCursor(CursorToken token) {
        synchronized (tokens)
        {
            tokens.remove(token);
            if (tokens.isEmpty())
            {
                IGVMainFrame.getInstance().getGlassPane().setVisible(false);
            }

        }
    }


    /**
     * A class to represent a token.
     */
    public static class CursorToken {}

}
