/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.ui;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.ui.panel.DataPanel;
import org.broad.igv.IGVConstants;
import org.broad.igv.PreferenceManager;

import org.broad.igv.feature.Chromosome;
import org.broad.igv.feature.Genome;
import org.broad.igv.feature.GenomeManager;
import org.broad.igv.track.Track;
import org.broad.igv.track.TrackGroup;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Rectangle;

import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;

import java.text.NumberFormat;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;

import java.util.List;
import javax.swing.JOptionPane;
import org.broad.igv.ui.panel.DragEventManager;
import org.broad.igv.ui.panel.TrackSetScrollPane;

/**
 *
 * @author jrobinso
 */
public class ViewContext {

    private String genomeId;
    Genome genome;
    /**
     * The nominal viewport width in pixels. Don't change this value, or
     * try to compute it dynamically.
     */
    public int binsPerTile = 700;
    // The following object is dynamic and change in response to zoom and
    // pan actions.
    /**
     * The chromosome currently in view
     */
    private String chrName = "chrAll";
    private DataPanel dataPanel;
    /**
     * The minimum zoom level for the current screen size + chromosome combination.
     *
     */
    private int minZoom = 0;
    /**
     * The current zoom level.  Zoom level -1 corresponds to the whole
     * genome view (chromosome "all")
     */
    private int zoom = minZoom;
    /**
     * The maximum zoom level.  Set to prevent integer overflow.  This is a function
     * of chromosom length.
     */
    public static int maxZoom = 23;
    /**
     * The number of tiles for this zoom level, = 2^zoom
     */
    private int nTiles = 1;
    /**
     * The maximum virtual pixel value.
     */
    private double maxPixel;
    /**
     * The origin in bp
     */
    private double origin = 0;
    /**
     * The location (x axis) locationScale in base pairs / virtual pixel
     */
    private double locationScale;
    private boolean locationScaleValid = false;
    /**
     * Indicates if the data is stretched to fill the window.  This happens at low
     * resolution views when drawing at the resolution of the zoom level would
     * only partially fill the screen.
     */
    private boolean stretched = false;
    /**
     * Regions of Interest
     */
    private LinkedHashMap<String, LinkedHashSet<RegionOfInterest>> regionsOfInterest =
            new LinkedHashMap<String, LinkedHashSet<RegionOfInterest>>();

    /**
     * Constructs ...
     *
     */
    public ViewContext() {
        String lastChromosomeViewed = PreferenceManager.getInstance().getLastChromosomeViewed();

        if ((lastChromosomeViewed == null) || lastChromosomeViewed.trim().equals("")) {
            this.chrName = getHomeChr();
        } else {
            this.chrName = lastChromosomeViewed;

        }
    }

    /**
     * Return the "home chromosome" for the current genome.  This should probably be a method
     * on the Genome class.
     * @return
     */
    public String getHomeChr() {
        if (genome != null && genome.getChromosomeNames().size() == 1) {
            return genome.getChromosomeNames().get(0);
        } else {
            return IGVConstants.CHR_ALL;
        }
    }

    /**
     * Compute the maximum zoom level, which is a function of chromosome length.
     */
    private void computeMaxZoom() {

        // Compute max zoom.  Assume window size @ max zoom of ~ 50 bp
        if (genome != null && chrName != null && genome.getChromosome(chrName) != null) {
            if (chrName.equals(IGVConstants.CHR_ALL)) {
                maxZoom = 0;
            } else {
                int chrLength = genome.getChromosome(chrName).getLength();

                maxZoom = (int) (Math.log(chrLength / 50.0) / log2) + 1;
            }

            if (zoom > maxZoom) {
                setZoom(maxZoom);
            }
        }


        // TODO -- fire chromosome changed event rather than this
    }

    private void setZoom(int newZoom) {
        zoom = Math.min(maxZoom, newZoom);
        nTiles = (int) Math.pow(2, Math.max(minZoom, zoom));
        maxPixel = getTilesTimesBinsPerTile();
        invalidateLocationScale();

        // TODO -- do this with events,
        if (IGVMainFrame.hasInstance()) {
            IGVMainFrame.getInstance().repaintStatusAndZoomSlider();
        }

    }

    /**
     * Method description
     *
     *
     * @param increment
     */
    public void incrementZoom(int increment) {
        zoomAndCenter((int) zoom + increment);
    }

    /**
     * Method description
     *
     *
     * @param newZoom
     */
    public void zoomAndCenterAdjusted(int newZoom) {
        zoomAndCenter(minZoom + newZoom);
    }

    /**
     * Method description
     *
     *
     * @param newZoom
     */
    public void zoomAndCenter(int newZoom) {

        // Zoom but remain centered about current center
        double currentCenter = origin + ((dataPanel.getWidth() / 2) * getScale());

        zoomTo(newZoom, currentCenter);
    }

    /**
     * Method description
     *
     *
     * @param newZoom
     * @param newCenter
     */
    public void zoomTo(final int newZoom, final double newCenter) {

        if (chrName.equals(IGVConstants.CHR_ALL)) {
            chrName = getHomeChr();
        }

        if (chrName.equals(IGVConstants.CHR_ALL)) {

            // DISABLE ZOOMING FOR GENOME VIEW
            // Translate the location to chromosome number
            jumpToChromosomeForGenomeLocation(newCenter);
            IGVMainFrame.getInstance().chromosomeChangeEvent();
        } else {
            if (zoom != newZoom) {

                setZoom(newZoom);
                computeLocationScaleImmediately();

                double newLocationScale = getScale();

                // Adjust origin so newCenter is centered
                double newOrigin = Math.round(
                        newCenter - ((dataPanel.getWidth() / 2) * newLocationScale));

                setOrigin(newOrigin);

            }
        }
        // This is a hack,  this is not a drag event but is a "jump"
        DragEventManager.getInstance().dragStopped();

    }

    private void jumpToChromosomeForGenomeLocation(double locationMB) {
        double startMB = 0;

        for (String chr : getGenome().getChromosomeNames()) {
            double endMB = startMB + getGenome().getChromosome(chr).getLength() / 1000.0;

            if ((locationMB > startMB) && (locationMB <= endMB)) {

                // this.jumpTo(chr, -1, -1);
                this.setChromosomeName(chr);
                break;
            }

            startMB = endMB;
        }
    }

    public void pageLeft() {
        shiftOriginPixels(-dataPanel.getWidth());
    }

    public void pageRight() {
        shiftOriginPixels(dataPanel.getWidth());
    }

    /**
     * Method description
     *
     *
     * @param delta
     */
    public void shiftOriginPixels(double delta) {
        double shiftBP = delta * getScale();
        setOrigin(origin + shiftBP);
    }

    public void snapToGrid() {
        setOrigin(Math.round(origin));
    }

    /**
     * Method description
     *
     *
     * @param chrLocation
     */
    public void centerOnLocation(String chr, double chrLocation) {
        if (!chrName.equals(chr)) {
            chrName = chr;
            computeMaxZoom();
            if (zoom > maxZoom) {
                setZoom(maxZoom);
            }
        }
        centerOnLocation(chrLocation);
    }

    /**
     * Method description
     *
     *
     * @param chrLocation
     */
    public void centerOnLocation(double chrLocation) {
        double windowWidth = (dataPanel.getWidth() * getScale()) / 2;

        setOrigin(Math.round(chrLocation - windowWidth));
    }

    public boolean windowAtEnd() {
        double windowLengthBP = dataPanel.getWidth() * getScale();
        return origin + windowLengthBP + 1 > getChromosomeLength();

    }


    /* Keep origin within data range */
    /**
     * Method description
     *
     *
     * @param newOrigin
     */
    public void setOrigin(double newOrigin) {
        int windowLengthBP = (int) (dataPanel.getWidth() * getScale());

        origin = Math.max(0, Math.min(newOrigin, getChromosomeLength() - windowLengthBP));

        for (TrackSetScrollPane sp : IGVMainFrame.getInstance().getTrackSetScrollPanes()) {
            preloadTrackData(sp.getDataPanel());
        }


        // Repaint
        IGVMainFrame.getInstance().repaintDataAndHeaderPanels();
        IGVMainFrame.getInstance().repaintStatusAndZoomSlider();
    }
    static double log2 = Math.log(2);

    /**
     * Method description
     *
     *
     * @param chr
     * @param start
     * @param end
     */
    public void jumpTo(String chr, int start, int end) {

        // Switch chromosomes if not null
        if (chr != null) {
             if (genome.getChromosome(chr) == null &&  !chr.contains(IGVConstants.CHR_ALL)) {
                JOptionPane.showMessageDialog(IGVMainFrame.getInstance(),
                        chr + " is not a valid chromosome.");
                return;
            }
            setChromosomeName(chr);
        }

        if (start >= 0) {

            // Estmate zoom level
            int z = (int) (Math.log(getChromosomeLength() / (end - start)) / log2) + 1;

            if (z != this.zoom) {
                zoom = Math.min(maxZoom, Math.max(minZoom, z));
                nTiles = (int) Math.pow(2, zoom);
                maxPixel = getTilesTimesBinsPerTile();
            }

            // overide computed scale
            int w = dataPanel.getWidth();

            setLocationScale(((double) (end - start)) / w);

            origin = start;
        }

        for (TrackSetScrollPane sp : IGVMainFrame.getInstance().getTrackSetScrollPanes()) {
            preloadTrackData(sp.getDataPanel());
        }

        // Repaint
        IGVMainFrame.getInstance().repaintDataAndHeaderPanels();
        IGVMainFrame.getInstance().repaintStatusAndZoomSlider();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public double getOrigin() {
        return origin;
    }

    public double getCenter() {
        return origin + getScale() * dataPanel.getWidth() / 2;
    }

    public double getEnd() {
        return origin + getScale() * dataPanel.getWidth();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getZoom() {
        return zoom;
    }

    /**
     * Return the maximum zoom level
     *
     * @return
     */
    public int getMaxZoom() {
        return maxZoom;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getAdjustedZoom() {
        return zoom - minZoom;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getNTiles() {
        return nTiles;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public double getMaxPixel() {
        return maxPixel;
    }

    /**
     * Method description
     *
     *
     * @param panel
     */
    public void setDataPanel(DataPanel panel) {

        if (panel != null) {

            dataPanel = panel;
            dataPanel.addComponentListener(new ComponentAdapter() {

                @Override
                public void componentResized(ComponentEvent e) {
                    invalidateLocationScale();
                }
            });
        }
    }

    /**
     * Method description
     *
     *
     * @param name
     */
    public void setChrName(String name) {
        this.setChromosomeName(name);
    }

    /**
     * @ deprecated, replace with calls to setChrName();
     * @param name
     * @param force
     */
    public void setChromosomeName(String name, boolean force) {

        if ((chrName == null) || !name.equals(chrName) || force) {
            chrName = name;
            origin = 0;
            setZoom(0);
            computeMaxZoom();
        }
    }

    /**
     * @ deprecated, replace with calls to setChrName();
     * @param name
     */
    public void setChromosomeName(String name) {
        setChromosomeName(name, false);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getChrName() {
        return chrName;
    }

    public String getNextChrName(String chr) {
        List<String> chrList = genome.getChromosomeNames();
        for (int i = 0; i < chrList.size() - 1; i++) {
            if (chrList.get(i).equals(chr)) {
                return chrList.get(i + 1);
            }
        }
        return null;
    }

    public String getPrevChrName(String chr) {
        List<String> chrList = genome.getChromosomeNames();
        for (int i = chrList.size() - 1; i > 0; i--) {
            if (chrList.get(i).equals(chr)) {
                return chrList.get(i - 1);
            }
        }
        return null;
    }

    // TODO -- this parameter shouldn't be stored here.  Maybe in a specialized
    // layout manager?
    /**
     * Method description
     *
     *
     * @return
     */
    public int getDataPanelWidth() {
        return dataPanel.getWidth();
    }

    /**
     * Return the current locationScale in base pairs / pixel
     *
     * @return
     */
    public double getScale() {
        if ((locationScale == 0) || !isLocationScaleValid()) {
            computeLocationScaleImmediately();
        }

        if (locationScale < 0) {
            System.err.println("Negative scale");
        }

        return locationScale;
    }

    /**
     * Method description
     *
     */
    public void invalidateLocationScale() {
        //Thread.dumpStack();
        setLocationScaleValid(false);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isLocationScaleValid() {
        return locationScaleValid;
    }

    /**
     * Method description
     *
     */
    public void computeLocationScaleImmediately() {

        if (genome != null) {

            computeMinZoom();

            double virtualPixelSize = getTilesTimesBinsPerTile();

            stretched = virtualPixelSize < dataPanel.getWidth();

            double nPixel = Math.max(virtualPixelSize, dataPanel.getWidth());

            setLocationScale(((double) getChromosomeLength()) / nPixel);
        }
    }

    /**
     * Compute the minimum zoom level for the data panel width.  This is defined as the maximum
     * zoom level for which all data bins will fit in the window without loss of
     * data,  i.e. the maximum zoom level for which nBins < nPixels.  The number
     * of bins is defined as
     *    nBins =  2^z
     * so minZoom is the value z such that nBins < dataPanel.getWidth()
     */
    private void computeMinZoom() {
        if (this.chrName.equals(IGVConstants.CHR_ALL)) {
            minZoom = 0;
        } else {
            minZoom = Math.max(0, (int) (Math.log((dataPanel.getWidth() / binsPerTile)) / log2));

            if (zoom < minZoom) {
                zoom = minZoom;
                nTiles = (int) Math.pow(2, zoom);
                maxPixel = getTilesTimesBinsPerTile();
            }
        }

    }

    /**
     * Return the chromosome position corresponding to the pixel index.  The
     * pixel index is the pixel "position" translated by -origin.
     *
     * @param pixelIndex
     *
     * @return
     */
    public double getChromosomePosition(int pixelIndex) {
        return origin + (getScale() * pixelIndex);
    }

    /**
     * Return the pixel position corresponding to the chromosomal position.
     *
     * @param chromosomePosition
     *
     * @return
     */
    public int getPixelPosition(double chromosomePosition) {
        return (int) ((chromosomePosition - origin) / getScale());
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Chromosome getChromosome() {
        if (genome == null) {
            throw new RuntimeException("Genome not loaded!");
        }

        return genome.getChromosome(chrName);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getChromosomeLength() {

        if (chrName.equals("All")) {

            // TODO -- remove the hardcoded unit divider ("1000")
            return (int) (genome.getLength() / 1000);

            // return genome.getLength();
        } else {
            if (getChromosome() == null) {
                System.out.println("Null chromosome: " + chrName);
            }

            return getChromosome().getLength();
        }
    }

    // /////////////////////////////////////////////////////////////////////////
    // Genome methods /////////////////////////////////////////////////////////
    /**
     * Method description
     *
     *
     * @param newGenome
     */
    public void setGenomeId(String newGenome) {

        boolean startUp = (genomeId == null);

        genomeId = newGenome;
        genome = GenomeManager.getInstance().getGenome(genomeId);

        if (genome == null) {

            // The previously used genome is unavailable.   Load hg18, we are assuming that
            // this is always available.  // TODO -- permit IGV starting with no selected genome
            genomeId = "hg18";
            GenomeManager.getInstance().findGenomeAndLoad(genomeId);
            genome = GenomeManager.getInstance().getGenome(genomeId);
            this.setChrName(getHomeChr());
        }

        // We don't know what chromosomes the new genome has, set to "all" (whole genome)
        if (!startUp) {
            this.setChrName(getHomeChr());
        }
        computeMaxZoom();
        invalidateLocationScale();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getGenomeId() {
        return genomeId;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Genome getGenome() {
        return genome;
    }

    /**
     * Return the collection of chromosome names for the currently loaded
     * genome
     *
     * @return
     */
    public Collection<String> getChromosomeNames() {
        return genome.getChromosomeNames();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isStretched() {
        return stretched;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public double getTilesTimesBinsPerTile() {
        return (double) nTiles * (double) binsPerTile;
    }

    /**
     * Get the UCSC style locus string corresponding to the current view.  THe UCSC
     * conventions are followed for coordinates,  specifically the internal representation
     * is "zero" based (first base is numbered 0) but the display representation is
     * "one" based (first base is numbered 1).   Consequently 1 is added to the
     * computed positions.
     * @return
     */
    public String getCurrentLocusString() {

        if (zoom == 0) {
            return getChrName();
        } else {

            int start = 0;
            int end = getDataPanelWidth();
            int startLoc = (int) getChromosomePosition(start) + 1;
            int endLoc = (int) getChromosomePosition(end);
            String startStr = NumberFormat.getInstance().format(startLoc);
            String endStr = NumberFormat.getInstance().format(endLoc);
            String position = getChrName() + ":" + startStr + "-" + endStr;

            return position;
        }
    }

    private void setLocationScale(double locationScale) {
        this.locationScale = locationScale;
        setLocationScaleValid(true);

    }

    /**
     * Method description
     *
     *
     * @param locationScaleValid
     */
    public void setLocationScaleValid(boolean locationScaleValid) {
        this.locationScaleValid = locationScaleValid;
    }

    /**
     *
     */
    private void preloadTrackData(DataPanel dp) {

        Rectangle visibleRect = dp.getVisibleRect();
        int start = (int) origin;
        int end = (int) (start + dp.getWidth() * this.locationScale);
        int trackY = 0;
        Collection<TrackGroup> groups = dp.getTrackGroups();

        for (Iterator<TrackGroup> groupIter = groups.iterator(); groupIter.hasNext();) {
            TrackGroup group = groupIter.next();
            if (group.isVisible()) {
                if (groups.size() > 1) {
                    trackY += IGVConstants.groupGap;
                }
                for (Track track : group.getTracks()) {
                    int trackHeight = track.getHeight();

                    if (visibleRect != null) {
                        if (trackY > visibleRect.y + visibleRect.height) {
                            break;
                        } else if (trackY + trackHeight < visibleRect.y) {
                            trackY += trackHeight;
                            continue;
                        }
                    }


                    if (track.isVisible()) {
                        track.preloadData(chrName, start, end, zoom);

                        trackY += track.getHeight();
                    }
                }
            }

        }
    }
}
