/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.ui;

//~--- non-JDK imports --------------------------------------------------------
import ncsa.hdf.hdf5lib.H5;
import ncsa.hdf.hdf5lib.exceptions.HDF5LibraryException;

import org.apache.log4j.Logger;

import org.broad.igv.IGVConstants;

import org.broad.igv.h5.HDF5RemoteReader;

//~--- JDK imports ------------------------------------------------------------

import java.io.File;
import java.io.FileFilter;

import java.util.Arrays;
import java.util.Comparator;

/**
 * This thread is registered upon startup and will get executed upon exit.
 */
public class ShutdownThread extends Thread {

    private static Logger log = Logger.getLogger(ShutdownThread.class);

    /**
     * Method description
     *
     */
    @Override
    public void run() {
        try {

            // Release local and remote HDF5 resources
            log.info("Closing HDF5");
            H5.H5close();
            HDF5RemoteReader.shutdown();

            // Cleanup jnlp files
            if (IGVConstants.IS_MAC) {
                File desktop = new File(System.getProperty("user.home") + "/Desktop");
                if (desktop.exists() && desktop.isDirectory()) {
                    cleanupJnlpFiles(desktop);
                }
                File downloads = new File(System.getProperty("user.home") + "/Downloads");
                if (downloads.exists() && downloads.isDirectory()) {
                    cleanupJnlpFiles(downloads);
                }

            }
        } catch (HDF5LibraryException ex) {
            ex.printStackTrace();
        }
    }

    /**
     * Cleanup extra jnlp files.  This method is written specifcally for Mac os.
     *
     */
    public static void cleanupJnlpFiles(File desktop) {

        File downloads = new File(System.getProperty("user.home") + "/Downloads");
        if (desktop.exists() && desktop.isDirectory()) {
            File[] jnlpFiles = desktop.listFiles(new FileFilter() {

                public boolean accept(File arg0) {
                    return arg0.getName().startsWith("igv") && arg0.getName().endsWith(".jnlp");
                }
            });

            // Sort files by ascending version number
            Arrays.sort(jnlpFiles, new Comparator<File>() {

                public int compare(File file1, File file2) {
                    return getVersionNumber(file1.getName()) - getVersionNumber(file2.getName());
                }

                private int getVersionNumber(String fn) {
                    int idx = fn.indexOf(".jnlp");
                    int idx2 = fn.lastIndexOf("-");
                    if (idx2 < 0) {
                        return 0;
                    } else {
                        try {
                            return Integer.parseInt(fn.substring(idx2 + 1, idx));

                        } catch (NumberFormatException numberFormatException) {
                            return 0;
                        }
                    }

                }
            });

            // Delete all but the highest version (newest) jnlp file
            for (int i = 0; i < jnlpFiles.length - 1; i++) {
                jnlpFiles[i].delete();
            }

            // Strip the version nuber fro the newest file
            if (jnlpFiles.length > 1) {
                File newestFile = jnlpFiles[jnlpFiles.length - 1];
                String fn = newestFile.getName();
                int dotIndex = fn.indexOf(".jnlp");
                int dashIndex = fn.lastIndexOf("-");
                if (dashIndex > 1) {
                    String newName = fn.substring(0, dashIndex) + fn.substring(dotIndex);
                    newestFile.renameTo(new File(newestFile.getParentFile(), newName));
                }
            }
        }
    }
}
