/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/




package org.broad.igv.ui;

//~--- non-JDK imports --------------------------------------------------------

import org.broad.igv.roc.ROC;

import org.broad.igv.track.RegionScoreType;


/**
 * Static imports
 */
import org.broad.igv.track.TrackManager;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;

import java.util.Collection;
import java.util.Iterator;

import java.util.List;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.event.MouseInputAdapter;
import org.broad.igv.PreferenceManager;
import org.broad.igv.roc.ROCScore;
import org.broad.igv.roc.ROCUtils;

/**
 *
 * @author eflakes
 */
public class RegionOfInterest extends MouseInputAdapter {

    private String chromosomeName;
    private String description;
    private Integer start;    // In Chromosome coordinates
    private Integer end;      // In Chromosome coordinates
    private Color backgroundColor = Color.RED;
    private Color foregroundColor = Color.BLACK;
    private JPopupMenu popupMenu;
    private Component parent;
    private WaitCursorManager.CursorToken token;

    /**
     * A bounded region on a chromosome.
     *
     *
     * @param chromosomeName
     * @param start The region starting position on the chromosome.
     * @param end The region starting position on the chromosome.
     * @param description
     */
    public RegionOfInterest(String chromosomeName, Integer start, Integer end, String description) {

        this.chromosomeName = chromosomeName;
        this.description = description;
        this.start = start;
        this.end = end;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getChromosomeName() {
        return chromosomeName;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getDescription() {
        return description;
    }

    /**
     * Method description
     *
     *
     * @param end
     */
    public void setEnd(Integer end) {
        this.end = end;
    }

    /**
     * Method description
     *
     *
     * @param start
     */
    public void setStart(Integer start) {
        this.start = start;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Integer getEnd() {
        return end;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Integer getStart() {
        return start;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Color getBackgroundColor() {
        return backgroundColor;
    }

    /**
     * Method description
     *
     *
     * @param backgroundColor
     */
    public void setBackgroundColor(Color backgroundColor) {
        this.backgroundColor = backgroundColor;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Color getForegroundColor() {
        return foregroundColor;
    }

    /**
     * Method description
     *
     *
     * @param foregroundColor
     */
    public void setForegroundColor(Color foregroundColor) {
        this.foregroundColor = foregroundColor;
    }

    protected JPopupMenu getPopupMenu(final Point point) {

        popupMenu = new JPopupMenu();

        // Create and add a menu item

        JMenuItem item = new JMenuItem("Sort by amplification");
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {

                TrackManager.getInstance().sortGroup(RegionOfInterest.this,
                        RegionScoreType.AMPLIFICATION);
                IGVMainFrame.getInstance().clearImageCacheWithNoRepaint();
                IGVMainFrame.getInstance().getContentPane().repaint();
            }
        });
        popupMenu.add(item);


        item = new JMenuItem("Sort by deletion");
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {

                TrackManager.getInstance().sortGroup(RegionOfInterest.this,
                        RegionScoreType.DELETION);
                IGVMainFrame.getInstance().clearImageCacheWithNoRepaint();
                IGVMainFrame.getInstance().getContentPane().repaint();
            }
        });
        popupMenu.add(item);

        item = new JMenuItem("Sort by expression");
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {

                TrackManager.getInstance().sortGroup(RegionOfInterest.this,
                        RegionScoreType.EXPRESSION);
                IGVMainFrame.getInstance().clearImageCacheWithNoRepaint();
                IGVMainFrame.getInstance().getContentPane().repaint();
            }
        });

        popupMenu.add(item);

        item = new JMenuItem("Sort by value");
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {

                TrackManager.getInstance().sortGroup(RegionOfInterest.this, RegionScoreType.SCORE);
                IGVMainFrame.getInstance().fireViewChangedEvent();
                IGVMainFrame.getInstance().getContentPane().repaint();
            }
        });

        popupMenu.add(item);

        if (ROC.ENABLED)
        {
            item = new JMenuItem("Search for corrleated genes (ROC)");
            item.addActionListener(new ActionListener() {

                public void actionPerformed(ActionEvent e) {
                    String att = PreferenceManager.getInstance().getOverlayAttribute();
                    ViewContext vc = IGVModel.getInstance().getViewContext();
                    ROCUtils rocUtils = new ROCUtils(att, chromosomeName, start, end, vc.getZoom());
                    
                    float threshold = 0.1f;
                    List<ROCScore> scores = rocUtils.computeScores(threshold);
                    for(ROCScore score : scores) {
                        System.out.println(score.getGene() + "\t" + score.getAUC());
                    }
                }
            });

            popupMenu.add(item);

        }

        popupMenu.add(new JSeparator());

        item = new JMenuItem("Delete");
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {

                ViewContext viewContext = IGVModel.getInstance().getViewContext();
                Collection<RegionOfInterest> regionsOfInterest =
                    IGVModel.getInstance().getRegionsOfInterest(chromosomeName);

                if (regionsOfInterest != null)
                {

                    Iterator iterator = regionsOfInterest.iterator();
                    while (iterator.hasNext())
                    {
                        RegionOfInterest region = (RegionOfInterest) iterator.next();

                        Integer start = region.getStart();
                        Integer end = region.getEnd();

                        if ((start == null) || (end == null))
                        {
                            continue;
                        }

                        Integer regionStart = viewContext.getPixelPosition(start);
                        Integer regionEnd = viewContext.getPixelPosition(end);

                        if ((regionStart == null) || (regionEnd == null))
                        {
                            continue;
                        }

                        if ((point.x >= regionStart) && (point.x <= regionEnd))
                        {
                            iterator.remove();
                        }
                    }
                }
                IGVMainFrame.getInstance().repaintDataAndHeaderPanels();
            }
        });
        popupMenu.add(item);

        return popupMenu;
    }

    /**
     * Method description
     *
     *
     * @param e
     */
    @Override
    public void mousePressed(MouseEvent e) {
        showPopup(e);
    }

    /**
     * Method description
     *
     *
     * @param e
     */
    @Override
    public void mouseReleased(MouseEvent e) {
        showPopup(e);
    }

    /**
     * Method description
     *
     *
     * @param e
     */
    @Override
    public void mouseEntered(MouseEvent e) {}

    /**
     * Method description
     *
     *
     * @param e
     */
    @Override
    public void mouseExited(MouseEvent e) {}

    /**
     * Method description
     *
     *
     * @param e
     */
    @Override
    public void mouseMoved(MouseEvent e) {

        Point point = e.getPoint();

        ViewContext viewContext = IGVModel.getInstance().getViewContext();
        if ((point.x >= viewContext.getPixelPosition(start))
                && (point.x <= viewContext.getPixelPosition(end)))
        {

            parent.setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));

        }
        else
        {
            parent.setCursor(Cursor.getDefaultCursor());
        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Component getParent() {
        return parent;
    }

    private void showPopup(MouseEvent e) {

        ViewContext viewContext = IGVModel.getInstance().getViewContext();
        Point point = e.getPoint();

        if ((point.x >= viewContext.getPixelPosition(start))
                && (point.x <= viewContext.getPixelPosition(end)))
        {

            getPopupMenu(point).show(e.getComponent(), e.getX(), e.getY());

        }
    }

    /**
     * Method description
     *
     *
     * @param parent
     */
    public void setParent(Component parent) {
        this.parent = parent;
    }
}
