/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * PreprocessorDialog.java
 *
 * Created on June 3, 2008, 5:42 PM
 */
package org.broad.igv.ui;

import org.broad.igv.feature.GenomeDescriptor;
import com.jidesoft.combobox.FileChooserComboBox;
import com.jidesoft.dialog.JideOptionPane;
import com.jidesoft.utils.SwingWorker;
import java.awt.Cursor;
import java.awt.Rectangle;
import java.io.File;
import java.util.Map;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import org.broad.igv.PreferenceManager;
import org.broad.igv.data.IGVDataset;
import org.broad.igv.data.Dataset;
import org.broad.igv.data.GCTDataset;
import org.broad.igv.data.GCTDatasetParser;
import org.broad.igv.data.WiggleParser;
import org.broad.igv.feature.GenomeManager;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.preprocess.old.AbstractProcessor;
import org.broad.igv.preprocess.old.NonOverlappingProcessor;
import org.broad.igv.preprocess.old.OverlappingProcessor;
import org.broad.igv.preprocess.old.StatusMonitor;
import org.broad.igv.track.TrackType;

/**
 *
 * @author  jrobinso
 */
public class PreprocessorDialog extends javax.swing.JDialog implements StatusMonitor {

    static final String[] knownExtensions = new String[]{".gct", ".res", ".cn", ".xcn",
        ".snp", "igv", "wig", "tab"

    };

    static public class PreprocessorFileChooserComboBox extends FileChooserComboBox {

        int type;

        public PreprocessorFileChooserComboBox(int type) {
            this.type = type;
        }

        @Override
        protected void customizeFileChooser(JFileChooser chooser) {
            chooser.setDialogType(type);
            File lastDirectoryFile =
                PreferenceManager.getInstance().getLastTrackDirectory();
            if (lastDirectoryFile != null)
            {
                chooser.setCurrentDirectory(lastDirectoryFile);
            }
            super.customizeFileChooser(chooser);
        }
    }

    private enum State {

        INIT, PROCESSING, COMPLETE
    }
    private State state = State.INIT;
    private ProcessWorker processWorker;
    private double percentComplete = 0.0;
    private TrackType dataType = TrackType.OTHER;

    /** Creates new form PreprocessorDialog */
    public PreprocessorDialog(java.awt.Frame parent, boolean modal) {
        initComponents();
        Map<String, GenomeDescriptor> map = GenomeManager.getInstance().getGenomeDescriptorMap();
        genomeComboBox.setSelectedItem(map.get(IGVModel.getInstance().getViewContext().getGenomeId()));
        setModal(false);
        setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
        setLocationRelativeTo(parent);
    }

    private Object[] getAllGenomes() {
        Map<String, GenomeDescriptor> map =
            GenomeManager.getInstance().getGenomeDescriptorMap();
        return map.values().toArray();
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        inputFileControl = new PreprocessorDialog.PreprocessorFileChooserComboBox(JFileChooser.OPEN_DIALOG);
        jLabel1 = new javax.swing.JLabel();
        outputFileControl = new PreprocessorDialog.PreprocessorFileChooserComboBox(JFileChooser.SAVE_DIALOG);
        jLabel2 = new javax.swing.JLabel();
        genomeComboBox = new javax.swing.JComboBox();
        jLabel3 = new javax.swing.JLabel();
        okButton = new javax.swing.JButton();
        cancelButton = new javax.swing.JButton();
        progressBar = new javax.swing.JProgressBar();
        missingDataExplanation = new javax.swing.JLabel();
        dataTypeComboBox = new javax.swing.JComboBox();
        jLabel4 = new javax.swing.JLabel();

        setDefaultCloseOperation(javax.swing.WindowConstants.DISPOSE_ON_CLOSE);
        setTitle("Preprocess Datasets");

        inputFileControl.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                inputFileControlActionPerformed(evt);
            }
        });

        jLabel1.setText("Input File");

        outputFileControl.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                outputFileControlActionPerformed(evt);
            }
        });

        jLabel2.setText("Output File");

        genomeComboBox.setModel(new javax.swing.DefaultComboBoxModel(getAllGenomes()));

        jLabel3.setText("Genome");

        okButton.setText("Run");
        okButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                okButtonActionPerformed(evt);
            }
        });

        cancelButton.setText("Cancel");
        cancelButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cancelButtonActionPerformed(evt);
            }
        });

        missingDataExplanation.setFont(new java.awt.Font("Lucida Grande", 2, 12));
        missingDataExplanation.setText("<html>Use this form to preprocess gct, res, cn, xcn, and snp files. <br>Preprocessing can significantly improve load times and overall performance."); // NOI18N

        dataTypeComboBox.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "", "GENE_EXPRESSION", "COPY_NUMBER", "ALLELE_SPECIFIC_COPY_NUMBER", "LOH", "DNA_METHYLATION", "OTHER" }));
        dataTypeComboBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                dataTypeComboBoxActionPerformed(evt);
            }
        });

        jLabel4.setText("Data Type");

        org.jdesktop.layout.GroupLayout layout = new org.jdesktop.layout.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(layout.createSequentialGroup()
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(org.jdesktop.layout.GroupLayout.TRAILING, layout.createSequentialGroup()
                        .addContainerGap()
                        .add(okButton)
                        .add(24, 24, 24)
                        .add(cancelButton))
                    .add(org.jdesktop.layout.GroupLayout.TRAILING, layout.createSequentialGroup()
                        .addContainerGap()
                        .add(missingDataExplanation))
                    .add(org.jdesktop.layout.GroupLayout.TRAILING, layout.createSequentialGroup()
                        .add(37, 37, 37)
                        .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                            .add(jLabel3)
                            .add(jLabel2)
                            .add(jLabel1)
                            .add(jLabel4))
                        .add(39, 39, 39)
                        .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.TRAILING)
                            .add(inputFileControl, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 331, Short.MAX_VALUE)
                            .add(org.jdesktop.layout.GroupLayout.LEADING, genomeComboBox, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 171, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                            .add(org.jdesktop.layout.GroupLayout.LEADING, outputFileControl, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 331, Short.MAX_VALUE)
                            .add(org.jdesktop.layout.GroupLayout.LEADING, dataTypeComboBox, 0, 331, Short.MAX_VALUE)))
                    .add(layout.createSequentialGroup()
                        .addContainerGap()
                        .add(progressBar, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 459, Short.MAX_VALUE)))
                .add(35, 35, 35))
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(org.jdesktop.layout.GroupLayout.TRAILING, layout.createSequentialGroup()
                .addContainerGap()
                .add(missingDataExplanation)
                .add(44, 44, 44)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(dataTypeComboBox, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(jLabel4))
                .add(28, 28, 28)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.TRAILING)
                    .add(layout.createSequentialGroup()
                        .add(inputFileControl, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 22, Short.MAX_VALUE)
                        .add(27, 27, 27)
                        .add(outputFileControl, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 22, Short.MAX_VALUE))
                    .add(layout.createSequentialGroup()
                        .add(jLabel1)
                        .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED, 39, Short.MAX_VALUE)
                        .add(jLabel2)))
                .add(27, 27, 27)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(genomeComboBox, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(jLabel3))
                .add(39, 39, 39)
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(okButton)
                    .add(cancelButton))
                .add(35, 35, 35)
                .add(progressBar, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                .add(38, 38, 38))
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents
    private void inputFileControlActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_inputFileControlActionPerformed
        if (outputFileControl.getSelectedItem() == null && inputFileControl.getSelectedItem() != null)
        {
            String inputFile = inputFileControl.getSelectedItem().toString();
            String outputFile = inputFile + ".h5";
            outputFileControl.setSelectedItem(new File(outputFile));
        //datasetNameField.setText(inputFileControl.getSelectedItem().toString());
        }
    }//GEN-LAST:event_inputFileControlActionPerformed

    private void okButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_okButtonActionPerformed

        if (state == State.INIT && validateInput())
        {
            setState(State.PROCESSING);
            process();
        } else if (state == State.PROCESSING)
        {
            if (processWorker != null)
            {
                processWorker.cancel(true);
            }
            String inputFile = inputFileControl.getSelectedItem().toString();
            if (inputFile != null)
            {
                File directory = new File(inputFile).getParentFile();
                PreferenceManager.getInstance().setLastTrackDirectory(directory);
            }
        } else if (state == State.COMPLETE)
        {
            setVisible(false);
        }
        
       
    }//GEN-LAST:event_okButtonActionPerformed

    private void cancelButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cancelButtonActionPerformed

        setVisible(false);
    }//GEN-LAST:event_cancelButtonActionPerformed

private void outputFileControlActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_outputFileControlActionPerformed
// TODO add your handling code here:
}//GEN-LAST:event_outputFileControlActionPerformed

private void dataTypeComboBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_dataTypeComboBoxActionPerformed
    String dataTypeString = dataTypeComboBox.getSelectedItem().toString();
    if(dataTypeString != null && dataTypeString.length() > 0) {
        dataType = TrackType.valueOf(dataTypeString);
    }
}//GEN-LAST:event_dataTypeComboBoxActionPerformed

    void setState(State state) {
        this.state = state;
        switch (state) {
            case INIT:
                setStatus(0.0);
                okButton.setText("Run");
                okButton.setEnabled(true);
                cancelButton.setEnabled(true);
                break;
            case PROCESSING:
                cancelButton.setEnabled(false);
                okButton.setText("Stop");
                break;
            case COMPLETE:
                okButton.setText("Done");
                okButton.setEnabled(true);
                setStatus(100.0);
        }
    }

    void preprocessingComplete() {
        processWorker = null;
        GuiUtilities.invokeOnEventThread(new Runnable() {
            public void run() {setCursor(null);
                setState(State.COMPLETE);
            }
        });
    }

    void preprocessingCanceled() {
        setCursor(null);
        processWorker = null;
        GuiUtilities.invokeOnEventThread(new Runnable() {
            public void run() {
        setCursor(null);
                setState(State.INIT);
            }
        });

    }

    private void process() {
        processWorker = new ProcessWorker();
        processWorker.execute();
    }

    public void setStatus(final double percent) {

        if(percent > 0 && percent < 1.0d) {
            percentComplete = Math.ceil(percent);
            
            // This line is a workaround
            percentComplete = 3.0d; // TODO Remove this line once we figure
                                         // out why the progress bar ignore values
                                         // less than 3.0
        }
        else {
            this.percentComplete = percent;
        }
        
        GuiUtilities.invokeOnEventThread(new Runnable() {

            public void run() {
                progressBar.setValue((int) percentComplete);
                progressBar.updateUI();
                Rectangle progressRect = progressBar.getBounds();
                progressBar.paintImmediately(progressRect);
            }
        });
        
        //System.out.println("Percent Complete: "+this.percent);
    }
 
    public void incrementStatus(final double increment) {
        if(progressBar.getValue() + increment >= 100) {
            System.out.println("Done");
        }
        setStatus(percentComplete + increment);

    }

    private boolean validateInput() {

        if (inputFileControl.getSelectedItem() == null ||
                outputFileControl.getSelectedItem() == null ||
                genomeComboBox.getSelectedItem() == null) {
            JideOptionPane.showMessageDialog(this, "All fields are required");
            return false;
        }
        
        // Check file extension
        String inputFile = inputFileControl.getSelectedItem().toString();
        String tmp = (inputFile.endsWith(".txt") ? 
            inputFile.substring(0, inputFile.length() - 4) :
            inputFile);
        boolean validExtension = false;
        for(String ext : knownExtensions) {
            if(tmp.endsWith(ext)) {
                validExtension = true;
                break;
            }
        }
        if(!validExtension) {
            JideOptionPane.showMessageDialog(this, "Unknown file type: " + inputFile);
            return false;
        }
        
        return true;
    }
    

    /**
     * 
     * @param <String>
     * @param <Void>
     */
    class ProcessWorker<String, Void> extends SwingWorker {

        private boolean success;

        @Override
        protected Object doInBackground() throws Exception {

            try {
                setCursor( Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR) );
                java.lang.String inputFile = inputFileControl.getSelectedItem().toString();
                java.lang.String outputFile = outputFileControl.getSelectedItem().toString();
                java.lang.String genomeId = ((GenomeDescriptor) genomeComboBox.getSelectedItem()).getId();
                java.lang.String name = inputFile;
                if (!name.endsWith(".h5")) {
                    name = name + ".h5";
                }

                Dataset ds = null;
                AbstractProcessor proc = null;

                java.lang.String tmp = (inputFile.endsWith(".txt") ? 
                inputFile.substring(0, inputFile.length() - 4) :
                inputFile);
                if (tmp.endsWith(".gct") || tmp.endsWith(".res") || inputFile.endsWith(".tab")) {
                    GCTDatasetParser parser = new GCTDatasetParser(new ResourceLocator(inputFile), (File) null, genomeId);
                    ds = parser.parse();
                    ((GCTDataset) ds).setType(TrackType.GENE_EXPRESSION);
                    ((GCTDataset) ds).setNormalized(true);
                    ((GCTDataset) ds).setLogValues(true);             
                    proc = new OverlappingProcessor(ds, PreprocessorDialog.this);
                    proc.setZoomMax(2);
                    ds.setName((new File(inputFile)).getName().replace("/.h5", ""));
                    success = proc.process(outputFile);

                } else if (tmp.endsWith(".snp") || tmp.endsWith(".cn") || tmp.endsWith(".xcn")) {
                    ds = new IGVDataset(genomeId, new ResourceLocator(inputFile));
                    proc = new NonOverlappingProcessor(ds, PreprocessorDialog.this);
                    proc.setZoomMax(3);
                    ds.setName((new File(inputFile)).getName().replace("/.h5", ""));
                    success = proc.process(outputFile);

                }  else if (tmp.endsWith(".igv")) {
                    ds = new IGVDataset(genomeId, new ResourceLocator(inputFile));
                    proc = new OverlappingProcessor(ds, PreprocessorDialog.this);
                    proc.setZoomMax(3);
                    ds.setName((new File(inputFile)).getName().replace("/.h5", ""));
                    success = proc.process(outputFile);
               } else if (tmp.endsWith(".wig")) {
                    java.lang.String dsName = (new File(inputFile)).getName();
                    
                    ds = (new WiggleParser(new ResourceLocator(inputFile), genomeId)).parse();

                    
                    proc = new OverlappingProcessor(ds, PreprocessorDialog.this);
                    proc.setZoomMax(4);
                    ds.setName((new File(inputFile)).getName().replace("/.h5", ""));
                    success = proc.process(outputFile);
                    
                } else {
                    return false;
                }

            }
            catch(Exception e) {
                e.printStackTrace();
                JOptionPane.showMessageDialog(PreprocessorDialog.this, "Processing error: " + e.getMessage());
            }
            finally {
                setCursor(null);
            }

            return success;

        }

        @Override
        protected void done() {
            // Update the UI.  Must invoke update code with SwingUtilities since
            // this is not running on the swing thread.
            if (success) {
                preprocessingComplete();
            } else {
                preprocessingCanceled();
            }

        }
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String args[]) {
        java.awt.EventQueue.invokeLater(new Runnable() {

            public void run() {
                PreprocessorDialog dialog = new PreprocessorDialog(new javax.swing.JFrame(), true);
                dialog.addWindowListener(new java.awt.event.WindowAdapter() {

                    @Override
                    public void windowClosing(java.awt.event.WindowEvent e) {
                        System.exit(0);
                    }
                });
                dialog.setVisible(true);
            }
        });
    }
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton cancelButton;
    private javax.swing.JComboBox dataTypeComboBox;
    private javax.swing.JComboBox genomeComboBox;
    private com.jidesoft.combobox.FileChooserComboBox inputFileControl;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel missingDataExplanation;
    private javax.swing.JButton okButton;
    private com.jidesoft.combobox.FileChooserComboBox outputFileControl;
    private javax.swing.JProgressBar progressBar;
    // End of variables declaration//GEN-END:variables
}
