/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * Class to hold state information that is, potentially, shared by multiple
 * components.  An attempt to reduce the bloat of IGVMainFrame.
 * 
 * A static class.  Could be a singleton, is there any advantage one way or the other?
 */
package org.broad.igv.ui;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import org.broad.igv.IGVConstants;
import org.broad.igv.track.TrackManager;

/**
 *
 * @author jrobinso
 */
public class IGVModel {

    /** Supports a state machine for the application. States are
     * INITIAL:  No data tracks loaded
     * GENOME:   Data loaded, whole genome view
     * CHROMOSOME:  Data loaded, viewing chromosome
     * 
     */
    enum ApplicationState {

        INITIAL, GENOME, CHROMOSOME, CHROMOSOME_ZOOMED
    };
    private static IGVModel instance;
    private ViewContext viewContext = new ViewContext();
    /** Map of chromosome -> regions of interest */
    private Map<String, Collection<RegionOfInterest>> regionsOfInterest = new LinkedHashMap();
    private String infoText = "";

    private IGVModel() {
    }

    public static IGVModel getInstance() {
        if (instance == null) {
            instance = new IGVModel();

        }
        return instance;
    }

    public String getInfoText() {
        return infoText;
    }

    public void setInfoText(String aInfoText) {
        infoText = aInfoText;
    }

    public ViewContext getViewContext() {
        return viewContext;
    }

    public Collection<RegionOfInterest> getRegionsOfInterest(String chr) {
        return regionsOfInterest.get(chr);
    }

    public Collection<RegionOfInterest> getAllRegionsOfInterest() {
        ArrayList<RegionOfInterest> roiList = new ArrayList();
        for (Collection<RegionOfInterest> roi : regionsOfInterest.values()) {
            roiList.addAll(roi);
        }
        return roiList;
    }

    public void addRegionOfInterestWithNoListeners(RegionOfInterest roi) {
        String chr = roi.getChromosomeName();
        Collection<RegionOfInterest> roiList = regionsOfInterest.get(chr);
        if (roiList == null) {
            roiList = new ArrayList();
            regionsOfInterest.put(chr, roiList);
        }
        roiList.add(roi);
    }

    public void clearRegionsOfInterest() {
        if (regionsOfInterest != null) {
            regionsOfInterest.clear();
        }
    }
}
