/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * IGVMainFrame.java
 *
 * Created on September 28, 2007, 2:14 PM
 */
package org.broad.igv.ui;

import java.util.concurrent.ExecutionException;
import java.util.logging.Level;
import org.broad.igv.ui.legend.LegendDialog;
import org.broad.igv.ui.panel.RulerPanel;
import org.broad.igv.ui.panel.AttributeHeaderPanel;
import org.broad.igv.ui.panel.CytobandPanel;
import org.broad.igv.ui.panel.HeaderPanel;
import org.broad.igv.ui.panel.InfoPanel;
import org.broad.igv.ui.panel.TrackSetView;
import org.broad.igv.ui.panel.DataPanel;
import org.broad.igv.feature.GenomeDescriptor;
import org.broad.igv.ui.action.*;
import org.broad.igv.ui.util.FileChooser;
import com.jidesoft.plaf.LookAndFeelFactory;
import com.jidesoft.swing.JideBoxLayout;
import com.jidesoft.swing.JideScrollPane;
import com.jidesoft.swing.JideSplitPane;
import com.jidesoft.utils.SwingWorker;
import java.awt.AlphaComposite;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.HeadlessException;
import java.awt.Image;
import java.awt.Insets;
import java.awt.KeyboardFocusManager;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.NoRouteToHostException;
import java.net.URL;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import java.util.Vector;
import javax.swing.AbstractButton;
import javax.swing.DefaultComboBoxModel;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.JWindow;
import javax.swing.RepaintManager;
import javax.swing.ToolTipManager;
import javax.swing.UIManager;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.filechooser.FileFilter;
import javax.swing.plaf.basic.BasicBorders;
import org.apache.log4j.Logger;
import org.broad.igv.IGVConstants;
import org.broad.igv.PreferenceManager;
import org.broad.igv.session.Session;
import org.broad.igv.session.SessionManager;
import org.broad.igv.feature.FeatureDB;
import org.broad.igv.feature.GenomeManager;
import org.broad.igv.feature.GenomeManager.GenomeListItem;
import org.broad.igv.feature.MaximumContigGenomeException;
import org.broad.igv.ui.util.ProgressMonitor;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.util.Utilities;
import org.broad.igv.ui.util.ApplicationStatusBar;
import org.broad.igv.ui.util.FileChooserDialog;
import org.broad.igv.ui.util.GenericUtilities;
import org.broad.igv.ui.util.IconFactory;
import org.broad.igv.ui.panel.View;
import org.broad.igv.ui.util.MenuAndToolbarUtils;
import org.broad.igv.ui.util.ProgressBar;
import org.broad.igv.roc.ROC;
import org.broad.igv.track.AttributeManager;
import org.broad.igv.track.TrackManager;
import org.broad.igv.ui.dnd.GhostGlassPane;
import org.broad.igv.ui.event.GlobalKeyDispatcher;
import org.broad.igv.ui.panel.TrackSetScrollPane;
import org.broad.igv.ui.util.IndefiniteProgressMonitor;
import org.broad.igv.util.BrowserLauncher;


// Static Imports
import org.jdesktop.layout.GroupLayout;
import static org.broad.igv.ui.util.GenericUtilities.*;
import static org.broad.igv.ui.util.UIUtilities.*;
import static org.broad.igv.ui.WaitCursorManager.CursorToken;
import static org.broad.igv.IGVConstants.*;
import static org.broad.igv.track.TrackManager.*;

/**
 *
 * @author  jrobinso
 */
public class IGVMainFrame extends javax.swing.JFrame {

    public static final String DATA_PANEL_NAME = "DataPanel";
    public static final String FEATURE_PANEL_NAME = "FeaturePanel";
    private static Logger log = Logger.getLogger(IGVMainFrame.class);
    private static IGVMainFrame theInstance;
    // Object to hold state and reduce bloat of this class
    private IGVModel model = IGVModel.getInstance();

    // Force constants to load early


    static {
        System.out.println(IGVConstants.APPLICATION_LONG_NAME + " starting...");
    }


    static {
        org.broad.igv.Utilities.addRollingAppenderToRootLogger();
        log.info("Default User Directory: " + IGVConstants.getDefaultUserDirectory());
    }
    /**
     * The view context.  Manages location, zoom, chromosome
     */
    private JMenu toolMenu;
    private IGVCommandBar igvCommandBar;
    private javax.swing.JPanel applicationHeaderView;
    private javax.swing.JPanel attributeHeaderPanel;
    private javax.swing.JPanel centerPanel;
    private javax.swing.JPanel cytobandPanel;
    private javax.swing.JScrollPane dataTrackScrollPane;
    private javax.swing.JScrollPane featureTrackScrollPane;
    private javax.swing.JPanel headerPanel;
    private javax.swing.JScrollPane headerScrollPane;
    private javax.swing.JPanel infoPanel;
    private javax.swing.JPanel regionOfInterestPane;
    private javax.swing.JPanel rulerPanel;
    private com.jidesoft.status.StatusBar statusBar;
    private JideSplitPane centerSplitPane;
    public String currentSessionFilePath;

    // TODO -- Should this really be stored in the frame instance?
    private AttributeCheckList attributeCheckList;

    // Loaded session files
    final private LinkedHashSet<String> loadedTrackFiles =
            new LinkedHashSet<String>();

    // Most recent sessions
    final private LinkedList<String> recentSessionList =
            new LinkedList<String>();

    // Cursors
    public static Cursor handCursor;
    public static Cursor fistCursor;
    public static Cursor zoomInCursor;
    public static Cursor zoomOutCursor;
    public static Cursor dragNDropCursor;

    // FileChooser Dialogs
    private FileChooserDialog trackFileChooser;
    private FileChooser snapshotFileChooser;
    private FileChooser genomeImportFileChooser;

    // TODO -- move this to the preferences manager
    private boolean showRegionsOfInterestBarsOn = true;

    // Current track filter action.
    //TODO -- A lot of state is passed between the embedded filter in this
    // action and the session during save and restore.  Refactor to pass
    // this state in a single object to/from the session.
    FilterTracksMenuAction filterTracksAction;
    //
    final static String ROI_BUTTON_TEXT = "Define a region of interest";

    // MenuItems that need to be exposed because
    // they need to be accessed by app code
    private JCheckBoxMenuItem showAttributeMenuItem;
    private boolean isExportingSnapshot = false;
    private boolean areResourceNodesCheckable = false;
    private Set<String> inaccessibleGenomeIds = new HashSet();
    private boolean igvInitialized = false;
    RemoveUserDefinedGenomeMenuAction removeImportedGenomeAction;

    // Glass panes
    Component glassPane;
    GhostGlassPane dNdGlassPane;

    // Tracksets
    private final Map<String, TrackSetScrollPane> trackSetScrollPanes = new Hashtable();

    /**
     * Creates new form IGVMainFrame
     */
    private IGVMainFrame() {

        // Create cursors
        createHandCursor();
        createZoomCursors();
        createDragAndDropCursor();

        setupIGV();
        igvInitialized = true;

        // Setup a glass pane to implement a blocking wait cursor
        glassPane = getGlassPane();
        getGlassPane().setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        getGlassPane().addMouseListener(new MouseAdapter() {
        });

        // A pane for D&D
        dNdGlassPane = new GhostGlassPane();
    }

    public GhostGlassPane getDnDGlassPane() {
        return dNdGlassPane;
    }

    public void startDnD() {
        setGlassPane(dNdGlassPane);
        getGlassPane().setVisible(true);
    }

    public void endDnD() {
        setGlassPane(glassPane);
        getGlassPane().setVisible(false);
    }

    public boolean isIGVIntialized() {
        return igvInitialized;
    }

    private void setupIGV() {
        theInstance = this;

        Runtime.getRuntime().addShutdownHook(new ShutdownThread());

        // Command listner thread
        //Thread clThread = new Thread(new CommandListener());
        //clThread.start();


        disableGraphicAccelerators();

        // TODO Remove this line once we want to save settings again
        PreferenceManager.getInstance().setDisplayableAttributes(null);

        // Create the center split pane
        centerSplitPane = new JideSplitPane() {

            @Override
            public Insets getInsets(Insets insets) {
                return new Insets(0, 0, 0, 0);
            }
        };

        // Create the command bar
        igvCommandBar = new IGVCommandBar(this);
        igvCommandBar.setMinimumSize(new Dimension(250, 33));

        initializeDefaultUserDirectory();

        setTitle(IGVConstants.APPLICATION_NAME);

        initComponents();

        // Add the 2 default panels
        trackSetScrollPanes.put(DATA_PANEL_NAME, (TrackSetScrollPane) dataTrackScrollPane);
        if (!PreferenceManager.getInstance().isShowSingleTrackPane()) {
            trackSetScrollPanes.put(FEATURE_PANEL_NAME, (TrackSetScrollPane) featureTrackScrollPane);
        }

        setupDisplayableAttributePreferences(null);

        // TODO -- figure out why the view context needs this reference
        DataPanel dp = ((TrackSetScrollPane) dataTrackScrollPane).getDataPanel();
        model.getViewContext().setDataPanel(dp);

        pack();

        // Setup the content pane widgets
        configureContentPane();

        // Set delay properties for tooltips.
        ToolTipManager ttm = ToolTipManager.sharedInstance();
        //ttm.setInitialDelay(1000);
        ttm.setDismissDelay(15000);

        initializeSnapshot();
        initializeDialogs();



        // Build the toolbar
        createMenuAndToolbar();

        // Set the chromosome
        String chrName =
                PreferenceManager.getInstance().getLastChromosomeViewed();
        model.getViewContext().setChromosomeName(chrName);

        // Macs are double buffered natively.  Double buffering in java is redundant
        // and has a noticeable effect on performance on Macs.
        // If os is Mac turn double buffering off.
        if (IGVConstants.IS_MAC) {
            System.setProperty("apple.awt.graphics.UseQuartz", "true");
            System.setProperty("apple.awt.rendering", "speed");
            // NOTE:  This doesn't seem to have any effect on Leopard.  Retest on Tiger?
            RepaintManager.currentManager(this).setDoubleBufferingEnabled(false);
        }

        // Must call the exit routine
        addWindowListener(new WindowAdapter() {

            @Override
            public void windowClosing(WindowEvent e) {
                IGVMainFrame.this.doExitApplication();
            }
        });

        for (TrackSetScrollPane sp : trackSetScrollPanes.values()) {
            AttributeManager.getInstance().addPropertyChangeListener(sp.getAttributePanel());
        }

        packViews();

        addComponentListener(
                new ComponentAdapter() {

                    @Override
                    public void componentResized(ComponentEvent e) {

                        // Single track pane view if true
                        boolean isShowSingleTrackPane =
                                PreferenceManager.getInstance().getShowSingleTrackPane();

                        if (isShowSingleTrackPane) {
                            //centerSplitPane.setDividerLocation(1.0d);
                        } else {
                            //centerSplitPane.setDividerLocation(0.85d);
                        }
                    }
                });

        addWindowListener(new WindowAdapter() {

            @Override
            public void windowOpened(WindowEvent e) {
                adjustSplitPaneDivider();
            }
        });

        // Set the application's previous location and size
        Rectangle applicationBounds =
                PreferenceManager.getInstance().getApplicationFrameBounds();
        if (applicationBounds != null) {
            setBounds(applicationBounds);
        }

    }

    private void setupInitialGenome(final String[] args) {

        SwingWorker worker = new SwingWorker() {

            public Object doInBackground() {

                boolean successful = false;

                final ProgressMonitor monitor = new ProgressMonitor();
                ProgressBar bar = null;
                try {
                    bar = ProgressBar.showProgressDialog(IGVMainFrame.this,
                            "Initializing Genome...", monitor, false);
                    // Initialize the genome list and select default
                    igvCommandBar.initializeGenomeList(monitor);
                    successful = true;
                } catch (Exception e) {

                    log.error("Failed to initialize the default genome ", e);
                }

                if (!successful) {
                    successful = displayFailedToInitializeGenomeDialog();
                }

                // Done
                monitor.fireProgressChange(100);

                final ProgressBar progressBar = bar;

                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {

                        progressBar.close();
                    }
                });

                //If there is an argument assume it is a session file or url
                if (args.length > 0) {
                    final IndefiniteProgressMonitor indefMonitor = new IndefiniteProgressMonitor(60);
                    final ProgressBar bar2 = ProgressBar.showProgressDialog(IGVMainFrame.this,
                            "Loading session data", indefMonitor, false);

                    final String sessionRef = args[0];

                    // Second argument should be a locus string
                    final String locusString = (args.length > 1) ? args[1] : null;
                    try {
                        LongRunningTask.submit(new Runnable() {

                            public void run() {
                                indefMonitor.start();
                                try {
                                    if (sessionRef.startsWith("http:") || sessionRef.startsWith("file:")) {
                                        URL url = new URL(sessionRef);
                                        doRestoreSession(url, locusString);
                                    } else {
                                        File sessionFile = new File(sessionRef);
                                        if (sessionFile.exists()) {
                                            doRestoreSession(sessionFile, locusString, false);
                                        }
                                    }
                                } catch (Exception ex) {
                                    JOptionPane.showMessageDialog(IGVMainFrame.this, "<html>Error loading session: " + sessionRef + "<br>" + ex.toString());
                                    log.error("Error loading session: " + sessionRef, ex);
                                }
                                indefMonitor.stop();
                                bar2.close();
                            }
                        }).get();

                    } catch (InterruptedException ex) {
                    } catch (ExecutionException ex) {
                    }

                }
                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {
                        setVisible(true);
                    }
                });
                return null;
            }
        };


        worker.execute();

    }

    public static IGVMainFrame getInstance() {
        if (theInstance == null) {
            theInstance = new IGVMainFrame();
        }

        return theInstance;
    }

    public static boolean hasInstance() {
        return theInstance != null;
    }

    private boolean displayFailedToInitializeGenomeDialog() {

        boolean success = true;

        try {
            final String[] buttons = {"Select", "Exit"};

            int option =
                    JOptionPane.showOptionDialog(this, "Failed to initialize the default genome." +
                    "\nPlease select another genome or exit.",
                    "Genome Initialization Failure",
                    JOptionPane.ERROR_MESSAGE,
                    0,
                    null,
                    buttons,
                    buttons[2]);

            if (option == 0) {
                GenomeListItem selection = doImportGenomeFromCustomList(); // Select from narrowed genome list
                if (selection == null) {
                    return false;
                }

            } else {
                System.exit(-1);
            }

        } catch (Exception e) {
            log.error("Failure while initializing genome", e);
            success =
                    false;
        }

        return success;
    }

    public Set<String> getInaccessibleGenomeIds() {
        return inaccessibleGenomeIds;
    }

    public void addInaccessibleGenomeId(String id) {
        this.inaccessibleGenomeIds.add(id);
    }

    public void removeInaccessibleGenomeId(String id) {
        this.inaccessibleGenomeIds.remove(id);
    }

    /**
     * This method is only called by IGV initialization code. It's sole purpose
     * is to provide the user with the ability to select a genome to load
     * (from a narrowed list) because IGV failed to load the default genome. The
     * selection list is narrowed by removing the genome IGV failed to load.
     * @return
     */
    private GenomeListItem doImportGenomeFromCustomList() {

        Vector<Object> genomes = new Vector();

        // Build a single available genome list from both client, server
        // and cached information. This allows us to process
        // everything the same way.
        LinkedHashSet<GenomeListItem> serverSideItemList = null;
        try {
            serverSideItemList = GenomeManager.getInstance().getServerGenomeArchiveList(null);
        } catch (Exception e) {
            log.error("Error getting genome archive list", e);
        }

        LinkedHashSet<GenomeListItem> userDefinedItemList = null;
        try {
            userDefinedItemList = GenomeManager.getInstance().getUserDefinedGenomeArchiveList(null);
        } catch (Exception e) {
            log.error("Error getting user genome archive list", e);
        }

        if (userDefinedItemList != null && !userDefinedItemList.isEmpty()) {
            boolean foundValidIds = false;
            for (GenomeListItem item : userDefinedItemList) {
                if (inaccessibleGenomeIds.contains(item.getId())) {
                    continue; // skip it
                }

                genomes.add(item);
                foundValidIds =
                        true;
            }
// If true we have atleast one valid genome
// id so we need the separator

            if (foundValidIds) {
                genomes.add(IGVConstants.GENOME_LIST_SEPARATOR);
            }

        }

        if (serverSideItemList != null && !serverSideItemList.isEmpty()) {
            for (GenomeListItem item : serverSideItemList) {
                if (inaccessibleGenomeIds.contains(item.getId())) {
                    continue; // skip it
                }

                genomes.add(item);
            }

        }

        JComboBox genomeComboBox = new JComboBox(genomes);
        genomeComboBox.setRenderer(new IGVCommandBar.ComboBoxRenderer());

        int option = JOptionPane.showConfirmDialog(this, genomeComboBox,
                "Select Genome", JOptionPane.OK_CANCEL_OPTION);

        if (option != JOptionPane.OK_OPTION) {
            return null;
        }

        GenomeListItem genomeListItem = (GenomeListItem) genomeComboBox.getSelectedItem();
        selectGenomeFromList(genomeListItem.getId());
        return genomeListItem;
    }

    public void setCheckingSelectedResourceNodesAllowed(boolean value) {
        areResourceNodesCheckable = value;
    }

    public boolean isCheckingSelectedResourceNodesAllowed() {

        return areResourceNodesCheckable;
    }

    static void clearAllPreferences() {

        try {
            PreferenceManager.getInstance().clear();
        } catch (Exception e) {
            e.printStackTrace();
            String message = "Failed to clear all preferences!";
            showAndLogErrorMessage(IGVMainFrame.theInstance, message, log, e);
        }

    }

    private void adjustSplitPaneDivider() {

        // Single track pane view if true
        boolean isShowSingleTrackPane =
                PreferenceManager.getInstance().getShowSingleTrackPane();

        if (isShowSingleTrackPane) {
            //centerSplitPane.setDividerLocation(1.0d);
            //centerSplitPane.setDividerSize(0);
        } else {
            //centerSplitPane.setDividerLocation(dividerBeforeSingleTrack);
            //centerSplitPane.setDividerSize(3);
        }
    }

    @Override
    public Dimension getPreferredSize() {
        return IGVConstants.preferredSize;
    }

    private void initializeDefaultUserDirectory() {

        // Create the user directory
        File defaultUserDirectory = new File(DEFAULT_IGV_DIRECTORY);
        if (!defaultUserDirectory.exists()) {
            boolean exists = defaultUserDirectory.exists();
            if (!exists) {
                boolean wasSuccessful = defaultUserDirectory.mkdir();
                if (!wasSuccessful) {
                    log.error("Failed to create user directory!");
                }

            }
        }
    }

    private void initializeDialogs() {

        // Create Track Chooser
        //  Note --  why are these reused ? (JTR)
        trackFileChooser = new FileChooserDialog(this, true);
    }

    public FileChooserDialog getTrackFileChooser() {
        return trackFileChooser;
    }

    private void initializeSnapshot() {

        File snapshotDirectory =
                PreferenceManager.getInstance().getLastSnapshotDirectory();

        if (snapshotDirectory != null) {
            // Create the snapshot directory
            if (!snapshotDirectory.exists()) {
                boolean exists = snapshotDirectory.exists();
                if (!exists) {
                    boolean wasSuccessful = snapshotDirectory.mkdir();
                    if (!wasSuccessful) {
                        log.error("Failed to create snapshot directory!");
                    }

                }
            }
        }

        // File Filters
        FileFilter[] fileFilters = GenericUtilities.getAllSnapshotFileFilters();

        snapshotFileChooser =
                getFileChooser(snapshotDirectory, null, fileFilters);
        snapshotFileChooser.setDialogTitle("Snapshot File");

        snapshotFileChooser.addPropertyChangeListener(
                new PropertyChangeListener() {

                    public void propertyChange(PropertyChangeEvent e) {

                        File oldFile = null;
                        String property = e.getPropertyName();
                        if (JFileChooser.SELECTED_FILE_CHANGED_PROPERTY.equals(property)) {
                            oldFile = (File) e.getOldValue();
                            snapshotFileChooser.setPreviousFile(oldFile);
                        } else if (JFileChooser.FILE_FILTER_CHANGED_PROPERTY.equals(property)) {

                            if (e.getOldValue() instanceof SnapshotFileFilter &&
                                    e.getNewValue() instanceof SnapshotFileFilter) {

                                SnapshotFileFilter newFilter =
                                        (SnapshotFileFilter) e.getNewValue();

                                File currentDirectory = snapshotFileChooser.getCurrentDirectory();
                                File previousFile = snapshotFileChooser.getPreviousFile();
                                if (previousFile != null) {

                                    File file = null;
                                    if (currentDirectory != null) {
                                        file = new File(currentDirectory, previousFile.getName());
                                    } else {
                                        file = previousFile;
                                    }

                                    final File selectedFile = Utilities.changeFileExtension(
                                            file, newFilter.getExtension());

                                    GuiUtilities.invokeOnEventThread(new Runnable() {

                                        public void run() {
                                            snapshotFileChooser.setSelectedFile(selectedFile);
                                            snapshotFileChooser.validate();
                                        }
                                    });
                                }

                            }
                        }
                    }
                });
    }

    /**
     * Clear the image cache
     */
    public void clearImageCacheForTrackPanels() {
        clearImageCacheWithNoRepaint();
        repaint();

    }

    public void clearImageCacheWithNoRepaint() {
        for (TrackSetScrollPane tsv : trackSetScrollPanes.values()) {
            tsv.getDataPanel().clearImageCache();
        }

    }

    public void addRegionOfInterest(RegionOfInterest roi) {
        model.addRegionOfInterestWithNoListeners(roi);
        roi.setParent(regionOfInterestPane);
        regionOfInterestPane.addMouseListener(roi);
        regionOfInterestPane.addMouseMotionListener(roi);
    }

    void beginROI(JButton button) {
        for (TrackSetScrollPane tsv : trackSetScrollPanes.values()) {
            DataPanel dp = tsv.getDataPanel();
            RegionOfInterestTool regionOfInterestTool =
                    new RegionOfInterestTool(dp, button);
            dp.setCurrentTool(regionOfInterestTool);
        }

    }

    void endROI() {
        for (TrackSetScrollPane tsv : trackSetScrollPanes.values()) {
            DataPanel dp = tsv.getDataPanel();
            dp.setCurrentTool(null);
        }

    }

    // TODO -- move this to preferences manager class
    public void setShowRegionsOfInterestBarsOn(boolean enabled) {
        showRegionsOfInterestBarsOn = enabled;
    }

// TODO -- move this to preferences manager class
    public boolean isShowRegionsOfInterestBarsOn() {
        return showRegionsOfInterestBarsOn;
    }

   public void chromosomeChangeEvent() {
        chromosomeChangeEvent(true);
    }

    public void chromosomeChangeEvent(boolean updateCommandBar) {
        igvCommandBar.chromosomeChanged();
        if (model.getViewContext().getChrName().equals(IGVConstants.CHR_ALL)) {
            cytobandPanel.setCursor(Cursor.getDefaultCursor());
            cytobandPanel.setToolTipText("");
        } else {
            cytobandPanel.setToolTipText(
                    "<html>Click anywhere on the cytoband <p>to center view at that location.");
            cytobandPanel.setCursor(Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
        }

        ((InfoPanel) infoPanel).updateHelpText();
        repaintDataAndHeaderPanels(updateCommandBar);

    }

    /**
     * Repaint panels containing data, specifically the dataTrackPanel,
     * featureTrackPanel, and headerPanel.
     */
    public void repaintDataAndHeaderPanels() {
        repaintDataAndHeaderPanels(true);
    }

    public void repaintDataAndHeaderPanels(boolean updateCommandBar) {
        repaintDataPanels();
        headerPanel.repaint();
        if(updateCommandBar) {
            igvCommandBar.updateCurrentCoordinates();
        }
    }

    public void repaintDataPanels() {
        for (TrackSetScrollPane tsv : trackSetScrollPanes.values()) {
            tsv.getDataPanel().repaint();
        }

    }

    public void repaintNamePanels() {
        for (TrackSetScrollPane tsv : trackSetScrollPanes.values()) {
            tsv.getNamePanel().repaint();
        }

    }

    public void repaintStatusAndZoomSlider() {
        //   igvCommandBar.repaint();
        //   statusBar.repaint();
    }

    public void repaintCytobandPanel() {
        headerPanel.repaint();
    }

    private void loadGeneTrack(String id) {
        IGVModel.getInstance().getViewContext().setGenomeId(id);
        TrackManager.getInstance().loadGeneTrack(id);
        fireViewChangedEvent();

    }

    public void selectGenomeFromList(String genome) {
        try {
            igvCommandBar.selectGenomeFromList(genome);
        } catch (FileNotFoundException e) {
            log.error("File not found while intializing genome!", e);
        } catch (NoRouteToHostException e) {
            log.error("Error while intializing genome!", e);
        }

    }

    private void configureContentPane() {

        addComponentListener(
                new ComponentAdapter() {

                    @Override
                    public void componentResized(ComponentEvent e) {

                        GuiUtilities.invokeOnEventThread(new Runnable() {

                            public void run() {
                                doRefresh();
                            }
                        });
                    }
                });

        // Splitpane
        centerSplitPane.setDividerSize(3);
        //centerSplitPane.setResizeWeight(0.5d);
        centerSplitPane.setOrientation(JSplitPane.VERTICAL_SPLIT);
        centerSplitPane.add(dataTrackScrollPane, JSplitPane.TOP);
        if (!PreferenceManager.getInstance().isShowSingleTrackPane()) {
            centerSplitPane.add(featureTrackScrollPane, JSplitPane.BOTTOM);
        }

        centerSplitPane.addPropertyChangeListener(
                new PropertyChangeListener() {

                    public void propertyChange(final PropertyChangeEvent e) {
                        synchronized (IGVMainFrame.this.trackSetScrollPanes) {
                            for (TrackSetScrollPane tsv : getTrackSetScrollPanes()) {
                                tsv.getDataPanel().doResize();
                            }

                        }
                    }
                });

        // If data viewport size changes we need to resize to track set pane

        for (TrackSetScrollPane sp : trackSetScrollPanes.values()) {
            final DataPanel dataPanel = sp.getDataPanel();
            sp.getViewport().addChangeListener(
                    new ChangeListener() {

                        public void stateChanged(final ChangeEvent e) {

                            GuiUtilities.invokeOnEventThread(new Runnable() {

                                public void run() {
                                    dataPanel.doResize();
                                }
                            });
                        }
                    });
        }

        centerPanel.add(headerScrollPane, BorderLayout.NORTH);
        centerPanel.add(centerSplitPane, BorderLayout.CENTER);
    }

    private void createMenuAndToolbar() {

        // Setup the menus
        setJMenuBar(MenuAndToolbarUtils.createMenuBar(createMenus()));

        // Setup the toolbar panel.
        JPanel toolbarPanel = new JPanel();
        toolbarPanel.setBorder(new BasicBorders.MenuBarBorder(Color.GRAY, Color.GRAY));

        getContentPane().add(toolbarPanel, BorderLayout.NORTH);
        toolbarPanel.setLayout(new JideBoxLayout(toolbarPanel));

        // Nothing for this toolbar yet, basically used as a space
        //JPanel namePanelToolBar = new JPanel();
        //namePanelToolBar.setLayout(new JideBoxLayout(namePanelToolBar));
        //namePanelToolBar.setPreferredSize(new Dimension(180, 10));
        //toolbarPanel.add(namePanelToolBar, JideBoxLayout.FLEXIBLE);
        toolbarPanel.add(igvCommandBar, JideBoxLayout.VARY);
    }

    public void doDefineGenome() {
        doDefineGenome(null);
    }

    public void doDefineGenome(ProgressMonitor monitor) {

        ProgressBar bar = null;
        GenomeListItem genomeListItem = null;

        boolean doDefine = true;
        while (doDefine) {
            doDefine = false;

            File archiveFile = null;

            CursorToken token = WaitCursorManager.showWaitCursor();
            try {
                GenomeBuilderDialog genomeBuilderDialog =
                        new GenomeBuilderDialog(this, true);

                genomeBuilderDialog.setVisible(true);
                if (genomeBuilderDialog.isCanceled()) {
                    return;
                }

                if (isIGVIntialized() && monitor != null) {
                    bar = ProgressBar.showProgressDialog(IGVMainFrame.getInstance(),
                            "Defining Genome...", monitor, false);
                }

                String genomeZipLocation =
                        genomeBuilderDialog.getGenomeArchiveLocation();
                String cytobandFileName =
                        genomeBuilderDialog.getCytobandFileName();
                String refFlatFileName =
                        genomeBuilderDialog.getRefFlatFileName();
                String fastaFileName =
                        genomeBuilderDialog.getFastaFileName();
                String relativeSequenceLocation =
                        genomeBuilderDialog.getSequenceLocation();
                String genomeDisplayName =
                        genomeBuilderDialog.getGenomeDisplayName();
                String genomeId =
                        genomeBuilderDialog.getGenomeId();
                String genomeFileName =
                        genomeBuilderDialog.getArchiveFileName();

                genomeListItem =
                        GenomeManager.getInstance().defineGenome(
                        genomeZipLocation, cytobandFileName, refFlatFileName,
                        fastaFileName, relativeSequenceLocation, genomeDisplayName,
                        genomeId, genomeFileName, monitor, null);

                enableRemoveGenomes();

                igvCommandBar.addToUserDefinedGenomeItemList(genomeListItem);
                igvCommandBar.selectGenomeFromListWithNoImport(genomeListItem.getId());

                if (isIGVIntialized() && monitor != null) {
                    monitor.fireProgressChange(100);
                }

            } catch (MaximumContigGenomeException e) {

                String genomePath = "";
                if (archiveFile != null) {
                    genomePath = archiveFile.getAbsolutePath();
                }

                log.error("Failed to define genome: " + genomePath, e);

                JOptionPane.showMessageDialog(this, "Failed to define the current genome " +
                        genomePath + "\n" + e.getMessage());
            } catch (Exception e) {
                String genomePath = "";
                if (archiveFile != null) {
                    genomePath = archiveFile.getAbsolutePath();
                }

                log.error("Failed to define genome: " + genomePath, e);
                int option =
                        JOptionPane.showConfirmDialog(this, "Failed to define the current genome " +
                        genomePath + "\n" + "Would you like to define another?",
                        "Define Genome Failure", JOptionPane.OK_CANCEL_OPTION);

                if (option == JOptionPane.OK_OPTION) {
                    doDefine = true;
                }

            } finally {
                if (bar != null) {
                    bar.close();
                }

                WaitCursorManager.removeWaitCursor(token);
            }

        }
    }

    public GenomeListItem getGenomeSelectedInDropdown() {
        return igvCommandBar.getGenomeSelectedInDropdown();
    }

    /**
     * Gets the collection of genome display names currently in use.
     * @return Set of display names.
     */
    public Collection<String> getGenomeDisplayNames() {
        return igvCommandBar.getGenomeDisplayNames();
    }

    public GenomeListItem doLoadGenome() {
        return doLoadGenome(null);
    }

    public GenomeListItem doLoadGenome(
            ProgressMonitor monitor) {

        ProgressBar bar = null;
        GenomeListItem genomeListItem = null;
        boolean doImport = true;
        while (doImport) {

            doImport = false;
            File file = null;
            CursorToken token = WaitCursorManager.showWaitCursor();
            try {
                File importDirectory =
                        PreferenceManager.getInstance().getLastGenomeImportDirectory();
                if (importDirectory == null) {
                    PreferenceManager.getInstance().setLastGenomeImportDirectory(
                            new File(IGVConstants.getDefaultUserDirectory()));
                }

// File Filters
                FileFilter[] fileFilters = {
                    new GenericUtilities.GenomeArchiveFileFilter()
                };

                genomeImportFileChooser =
                        getFileChooser(importDirectory, null, fileFilters);
                genomeImportFileChooser.setDialogTitle("Load Genome");
                genomeImportFileChooser.addPropertyChangeListener(
                        new PropertyChangeListener() {

                            public void propertyChange(PropertyChangeEvent e) {

                                File oldFile = null;
                                String property = e.getPropertyName();
                                if (JFileChooser.SELECTED_FILE_CHANGED_PROPERTY.equals(property)) {
                                    oldFile = (File) e.getOldValue();
                                    genomeImportFileChooser.setPreviousFile(oldFile);
                                } else if (JFileChooser.FILE_FILTER_CHANGED_PROPERTY.equals(property)) {

                                    if (e.getOldValue() instanceof GenericUtilities.GenomeArchiveFileFilter &&
                                            e.getNewValue() instanceof GenericUtilities.GenomeArchiveFileFilter) {

                                        GenericUtilities.GenomeArchiveFileFilter newFilter =
                                                (GenericUtilities.GenomeArchiveFileFilter) e.getNewValue();

                                        File currentDirectory = genomeImportFileChooser.getCurrentDirectory();
                                        File previousFile = genomeImportFileChooser.getPreviousFile();
                                        if (previousFile != null) {

                                            File file = null;
                                            if (currentDirectory != null) {
                                                file = new File(currentDirectory,
                                                        previousFile.getName());
                                            } else {
                                                file = previousFile;
                                            }

                                            final File selectedFile = Utilities.changeFileExtension(
                                                    file, newFilter.getExtension());

                                            GuiUtilities.invokeOnEventThread(new Runnable() {

                                                public void run() {
                                                    genomeImportFileChooser.setSelectedFile(
                                                            selectedFile);
                                                    genomeImportFileChooser.validate();
                                                }
                                            });
                                        }

                                    }
                                }
                            }
                        });

                // Display the dialog
                genomeImportFileChooser.showOpenDialog(this);
                file =
                        genomeImportFileChooser.getSelectedFile();

                // If a file selection was made
                if (file != null) {
                    if (isIGVIntialized() && monitor != null) {
                        bar = ProgressBar.showProgressDialog(IGVMainFrame.getInstance(),
                                "Loading Genome...", monitor, false);
                    }

                    File directory = genomeImportFileChooser.getCurrentDirectory();
                    if (directory != null) {
                        PreferenceManager.getInstance().setLastGenomeImportDirectory(
                                directory);
                    }

                    try {

                        if (isIGVIntialized() && monitor != null) {
                            monitor.fireProgressChange(50);
                        }

// Import the genome
                        genomeListItem =
                                GenomeManager.getInstance().importGenome(
                                file.getAbsolutePath(),
                                false, true, monitor);

                        igvCommandBar.addToUserDefinedGenomeItemList(genomeListItem);
                        igvCommandBar.selectGenomeFromListWithNoImport(genomeListItem.getId());


                        if (isIGVIntialized() && monitor != null) {
                            monitor.fireProgressChange(100);
                        }

                        if (bar != null) {
                            bar.close();
                        }

                    } catch (Exception e) {
                        log.fatal("Could not import genome!", e);
                    } finally {
                    }
                }
            } catch (Exception e) {

                String genomePath = "";
                if (file != null) {
                    genomePath = file.getAbsolutePath();
                }

                log.error("Failed to load genome: " + genomePath, e);
                int option =
                        JOptionPane.showConfirmDialog(this, "Failed to load the current genome " +
                        genomePath + "\n" + "Would you like to load another?",
                        "Load Genome Failure", JOptionPane.OK_CANCEL_OPTION);

                if (option == JOptionPane.OK_OPTION) {
                    doImport = true;
                }

            } finally {
                WaitCursorManager.removeWaitCursor(token);
            }

        }

        return genomeListItem;
    }

    public IGVModel getModel() {
        return model;
    }

    private List<AbstractButton> createMenus() {

        List<AbstractButton> menus = new ArrayList<AbstractButton>();
        menus.add(createFileMenu());
        menus.add(createViewMenu());
        menus.add(createTracksMenu());

        toolMenu =
                createToolMenu();
        toolMenu.setVisible(false);
        menus.add(toolMenu);
        menus.add(createToolMenu());

        menus.add(createHelpMenu());

        // Experimental -- remove for production release

        return menus;
    }

    public void enableToolMenu() {
        toolMenu.setVisible(true);
    }

    /**
     * Load a collection of tracks in a background thread.
     *
     * @param locators
     */
    public void loadTracks(final Collection<ResourceLocator> locators) {
        loadTracks(locators, true);


    }

    /**
     * Load tracks corresponding to a collection of resource locations.
     *
     * // TODO -- why is this in the main frame (as opposed to TrackManager for example)?
     * @param locators
     */
    public void loadTracks(final Collection<ResourceLocator> locators, boolean doInBackground) {

        ((ApplicationStatusBar) statusBar).setMessage("Loading ...");
        Runnable loader = new Runnable() {

            public void run() {
                boolean tracksWereLoaded = false;

                try {
                    if (locators != null && !locators.isEmpty()) {

                        // get current track count per panel.  Needed to detect which panels
                        // changed.  Also record panel sizes
                        final HashMap<TrackSetScrollPane, Integer> trackCountMap = new HashMap();
                        final HashMap<TrackSetScrollPane, Integer> panelSizeMap = new HashMap();
                        for (TrackSetScrollPane sp : trackSetScrollPanes.values()) {
                            trackCountMap.put(sp, sp.getDataPanel().getAllTracks().size());
                            panelSizeMap.put(sp, sp.getDataPanel().getHeight());
                        }

                        tracksWereLoaded = TrackManager.getInstance().loadResources(locators);

                        if (tracksWereLoaded) {
                            GuiUtilities.invokeOnEventThread(new Runnable() {

                                public void run() {
                                    clearImageCacheWithNoRepaint();

                                    for (TrackSetScrollPane sp : trackSetScrollPanes.values()) {
                                        if (trackCountMap.containsKey(sp)) {
                                            int prevTrackCount = trackCountMap.get(sp).intValue();
                                            if (prevTrackCount != sp.getDataPanel().getAllTracks().size()) {
                                                sp.getDataPanel().doResize();
                                                int scrollPosition = panelSizeMap.get(sp);
                                                if (prevTrackCount != 0 && sp.getVerticalScrollBar().isShowing()) {
                                                    sp.getVerticalScrollBar().setMaximum(
                                                            sp.getDataPanel().getHeight());
                                                    sp.getVerticalScrollBar().setValue(
                                                            scrollPosition);
                                                }

                                            }
                                        }
                                    }

                                    // Adjust divider for data panel.  The data panel divider can be
                                    // zero if there are no data tracks loaded.
                                    TrackSetScrollPane dsp = trackSetScrollPanes.get(IGVMainFrame.DATA_PANEL_NAME);
                                    if (dsp.getDataPanel().getAllTracks().size() > 0 &&
                                            centerSplitPane.getDividerLocation(0) < 10) {
                                        centerSplitPane.setDividerLocation(0, 40);
                                    }

                                    doRefresh();

                                }
                            });

                        }

                    }


                } finally {
                    showLoadedTrackCount();

                }

            }
        };

        if (doInBackground) {
            LongRunningTask.submit(loader);
        } else {
            loader.run();
        }

    }

    public void updateAttributePanel() {
        // Need this because TrackManager loads some attributes
        // internally. So we need to pick them up.
        setupDisplayableAttributePreferences(AttributeManager.getInstance().getAttributeKeys());
    }

    private JMenu createFileMenu() {

        List<JComponent> menuItems = new ArrayList<JComponent>();
        MenuAction menuAction = null;

        menuItems.add(new JSeparator());

        // Load menu items
        menuAction =
                new LoadFilesMenuAction("Load ...", KeyEvent.VK_L, this);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        menuAction =
                new LoadFromServerAction("Load From Server...", KeyEvent.VK_S, this);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        menuItems.add(new JSeparator());

        // Session menu items
        menuAction =
                new NewSessionMenuAction("New Session...", KeyEvent.VK_N, this);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        menuAction =
                new OpenSessionMenuAction("Open Session...", KeyEvent.VK_O, this);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        menuAction =
                new SaveSessionMenuAction("Save Session...", KeyEvent.VK_V, this);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        menuItems.add(new JSeparator());

        menuAction =
                new MenuAction(IGVConstants.IMPORT_GENOME_LIST_MENU_ITEM, null, KeyEvent.VK_D) {

                    @Override
                    public void actionPerformed(ActionEvent event) {

                        SwingWorker worker = new SwingWorker() {

                            public Object doInBackground() {

                                ProgressMonitor monitor = null;
                                if (isIGVIntialized()) {
                                    monitor = new ProgressMonitor();
                                }

                                doDefineGenome(monitor);
                                return null;
                            }
                        };
                        worker.execute();
                    }
                };

        menuAction.setToolTipText(UIConstants.IMPORT_GENOME_TOOLTIP);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        boolean hasImportedGenomes = true;
        try {
            hasImportedGenomes = !GenomeManager.getInstance().getUserDefinedGenomeArchiveList(null).isEmpty();

        } catch (IOException iOException) {
            // Ignore
        }
        removeImportedGenomeAction = new RemoveUserDefinedGenomeMenuAction(
                IGVConstants.REMOVE_GENOME_LIST_MENU_ITEM, KeyEvent.VK_R);
        removeImportedGenomeAction.setEnabled(hasImportedGenomes);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(removeImportedGenomeAction));

        //menuAction = new ClearGenomeCacheAction("Clear Genome Cache...");
        //menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        menuItems.add(new JSeparator());

        // ***** Snapshots
        // Snapshot Application
        menuAction =
                new MenuAction("Save Image ...", null, KeyEvent.VK_A) {

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        doApplicationSnapshot();

                    }
                };

        menuAction.setToolTipText(UIConstants.SAVE_IMAGE_TOOLTIP);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        menuItems.add(new JSeparator());

        // Export Regions
        menuAction =
                new ExportRegionsMenuAction("Export Regions ...", KeyEvent.VK_E, this);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));


        // Import Regions
        menuAction =
                new ImportRegionsMenuAction("Import Regions ...", KeyEvent.VK_I, this);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        // Separator
        menuItems.add(new JSeparator());
        menuAction =
                new MenuAction("Preprocess ...", null, KeyEvent.VK_P) {

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        (new PreprocessorDialog(IGVMainFrame.this, false)).setVisible(true);
                    }
                };

        menuAction.setToolTipText(UIConstants.IMPORT_REGION_TOOLTIP);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));


        // Separator
        menuItems.add(new JSeparator());

        // Exit
        menuAction =
                new MenuAction("Exit", null, KeyEvent.VK_X) {

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        doExitApplication();
                    }
                };

        menuAction.setToolTipText(UIConstants.EXIT_TOOLTIP);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));


        // Empty the recent sessions list before we start to do
        // anything with it
        recentSessionList.clear();

        // Retrieve the stored session paths
        String recentSessions = PreferenceManager.getInstance().getRecentSessions();
        if (recentSessions != null) {
            String[] sessions = recentSessions.split(";");
            for (String sessionPath : sessions) {
                if (!recentSessionList.contains(sessionPath)) {
                    recentSessionList.add(sessionPath);
                }

            }
        }

        if (!recentSessionList.isEmpty()) {

            menuItems.add(new JSeparator());

            // Now add menu items
            for (final String session : recentSessionList) {
                OpenSessionMenuAction osMenuAction = new OpenSessionMenuAction(session, new File(
                        session), this);
                menuItems.add(MenuAndToolbarUtils.createMenuItem(osMenuAction));
            }

        }

        MenuAction fileMenuAction = new MenuAction("File", null, KeyEvent.VK_F);
        return MenuAndToolbarUtils.createMenu(menuItems, fileMenuAction);
    }

    private JMenu createTracksMenu() {

        List<JComponent> menuItems = new ArrayList<JComponent>();
        MenuAction menuAction = null;

        // Sort Context
        menuAction =
                new SortTracksMenuAction("Sort Tracks ...", KeyEvent.VK_S, this);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        menuAction =
                new GroupTracksMenuAction("Group Tracks  ... ", KeyEvent.VK_G, this);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        // Filter Tracks
        filterTracksAction =
                new FilterTracksMenuAction("Filter Tracks ...", KeyEvent.VK_F, this);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(filterTracksAction));

        menuItems.add(new JSeparator());

        // Reset Tracks
        menuAction =
                new FitDataToWindowMenuAction("Fit Data to Window", KeyEvent.VK_W, this);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));


        // Set track height
        menuAction =
                new SetTrackHeightMenuAction("Set Track Height...", KeyEvent.VK_H, this);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));


        MenuAction dataMenuAction = new MenuAction("Tracks", null, KeyEvent.VK_K);
        return MenuAndToolbarUtils.createMenu(menuItems, dataMenuAction);
    }

    private JMenu createViewMenu() {

        List<JComponent> menuItems = new ArrayList<JComponent>();
        MenuAction menuAction = null;

        // Preferences
        menuAction =
                new MenuAction("Preferences...", null, KeyEvent.VK_P) {

                    @Override
                    public void actionPerformed(ActionEvent e) {

                        GuiUtilities.invokeOnEventThread(new Runnable() {

                            public void run() {
                                doViewPreferences();
                            }
                        });
                    }
                };
        menuAction.setToolTipText(UIConstants.PREFERENCE_TOOLTIP);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        menuAction =
                new MenuAction("Color Legends ...", null, KeyEvent.VK_H) {

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        (new LegendDialog(IGVMainFrame.this, false)).setVisible(true);
                    }
                };
        menuAction.setToolTipText(UIConstants.SHOW_HEATMAP_LEGEND_TOOLTIP);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        menuItems.add(new JSeparator());

        // Hide or Show the attribute panels
        boolean isShow = PreferenceManager.getInstance().getShowAttributeView();

        menuAction =
                new MenuAction("Show Attribute Display", null, KeyEvent.VK_A) {

                    @Override
                    public void actionPerformed(ActionEvent e) {

                        JCheckBoxMenuItem menuItem =
                                (JCheckBoxMenuItem) e.getSource();
                        doShowAttributeDisplay(menuItem.getState());
                    }
                };
        menuAction.setToolTipText(UIConstants.SHOW_ATTRIBUTE_DISPLAY_TOOLTIP);
        showAttributeMenuItem =
                MenuAndToolbarUtils.createMenuItem(menuAction, isShow);
        menuItems.add(showAttributeMenuItem);
        doShowAttributeDisplay(isShow);

        menuAction =
                new MenuAction("Select Attributes to Show...", null, KeyEvent.VK_S) {

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        doSelectDisplayableAttribute();
                    }
                };
        menuAction.setToolTipText(UIConstants.SELECT_DISPLAYABLE_ATTRIBUTES_TOOLTIP);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));


        // Hide or Show the Regions of Interest
        /*boolean isShowROI = PreferenceManager.getInstance().getShowRegionTool();
        menuAction =
        new MenuAction("Enable Regions of Interest", null, KeyEvent.VK_E) {
        @Override
        public void actionPerformed(ActionEvent e) {
        JCheckBoxMenuItem menuItem =
        (JCheckBoxMenuItem) e.getSource();
        doEnableRegionsOfInterest(menuItem.getState());
        }
        };
        menuAction.setToolTipText(UIConstants.ENABLE_REGIONS_OF_INTEREST_TOOLTIP);
        regionsOfInterestMenuItem =
        MenuAndToolbarUtils.createMenuItem(menuAction, isShowROI);
        menuItems.add(regionsOfInterestMenuItem);
        doEnableRegionsOfInterest(isShowROI);
         * */

        menuItems.add(new JSeparator());
        menuAction =
                new MenuAction("Refresh", null, KeyEvent.VK_R) {

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        doRefreshAndImageCacheClear();
                    }
                };
        menuAction.setToolTipText(UIConstants.REFRESH_TOOLTIP);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));



        /* SAVE THIS CODE FOR DEBUGGING
        menuAction =
        new MenuAction("Use image caching", null, KeyEvent.VK_C) {
        @Override
        public void actionPerformed(ActionEvent e) {
        JCheckBoxMenuItem menuItem =
        (JCheckBoxMenuItem) e.getSource();
        boolean isSelected = menuItem.getState();
        ((DataPanel) dataTrackPanel).setCaching(isSelected);
        doRefresh();
        }
        };
        boolean linkedSorting = ((DataPanel) dataTrackPanel).linkedSorting();
        menuAction.setToolTipText(UIConstants.USE_IMAGE_CACHING_TOOLTIP);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction, linkedSorting));
         */

        // Add to View menu
        MenuAction dataMenuAction = new MenuAction("View", null, KeyEvent.VK_V);
        return MenuAndToolbarUtils.createMenu(menuItems, dataMenuAction);
    }

    private JMenu createToolMenu() {

        List<JComponent> menuItems = new ArrayList<JComponent>();

        MenuAction menuAction = null;

        menuAction =
                new ResetPreferencesAction("Reset Preferences", this);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));



        //menuAction = new LaunchGeneEAction("Launch Gene-E", this);
        //menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        //menuAction = new MacroSnapshotAction("Macro Snapshots ... ", this);
        //menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        // Direct Draw Disabled - Windows Only
        if (!IS_MAC) {

            // Separator
            menuItems.add(new JSeparator());

            // Is Direct Draw Disabled
            PreferenceManager manager = PreferenceManager.getInstance();
            boolean isDirectDrawDisabled = manager.getDirectDrawDisabled();
            menuAction =
                    new DisableDirectDrawAction("Disable Direct Draw", KeyEvent.VK_D, this);
            menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction, isDirectDrawDisabled));
            if (isDirectDrawDisabled) {
                System.setProperty("sun.java2d.noddraw", "true");
            }

        }


        menuAction =
                new MenuAction("Use Linked Sorting", null, KeyEvent.VK_C) {

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        JCheckBoxMenuItem menuItem =
                                (JCheckBoxMenuItem) e.getSource();
                        boolean isSelected = menuItem.getState();
                        PreferenceManager.getInstance().setLinkedSortingEnabled(isSelected);
                    }
                };
        boolean linkedSorting = PreferenceManager.getInstance().isLinkedSortingEnabled();
        menuAction.setToolTipText("Enable linked sorting");
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction, linkedSorting));

        // ROC curves
        menuAction =
                new MenuAction("Enable ROC", null, KeyEvent.VK_C) {

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        JCheckBoxMenuItem menuItem =
                                (JCheckBoxMenuItem) e.getSource();
                        boolean isSelected = menuItem.getState();
                        ROC.ENABLED = isSelected;
                    }
                };
        boolean rocEnabled = ROC.ENABLED;
        menuAction.setToolTipText("Enable ROC curve for gene searching.");
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction, rocEnabled));


        menuItems.add(new JSeparator());

        // Load genome
        menuAction =
                new MenuAction(LOAD_GENOME_LIST_MENU_ITEM, null, KeyEvent.VK_I) {

                    @Override
                    public void actionPerformed(ActionEvent event) {

                        SwingWorker worker = new SwingWorker() {

                            public Object doInBackground() {

                                ProgressMonitor monitor = null;
                                if (isIGVIntialized()) {
                                    monitor = new ProgressMonitor();
                                }

                                doLoadGenome(monitor);
                                return null;
                            }
                        };
                        worker.execute();
                    }
                };

        menuAction.setToolTipText(UIConstants.LOAD_GENOME_TOOLTIP);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        // Label y axis charts
        final JCheckBoxMenuItem cbMenuItem = new JCheckBoxMenuItem("Label Y Axis on Charts");
        PreferenceManager.ChartPreferences cp = PreferenceManager.getInstance().getChartPreferences();
        cbMenuItem.setSelected(cp.isDrawAxis());
        cbMenuItem.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent arg0) {
                PreferenceManager.getInstance().put(
                        PreferenceManager.ChartPreferences.Y_AXIS,
                        String.valueOf(cbMenuItem.isSelected()));
                repaint();

            }
        });
        menuItems.add(cbMenuItem);


        MenuAction toolMenuAction = new MenuAction("Tools");
        JMenu menu = MenuAndToolbarUtils.createMenu(menuItems, toolMenuAction);
        menu.setVisible(false);



        return menu;
    }

    private JMenu createHelpMenu() {

        List<JComponent> menuItems = new ArrayList<JComponent>();

        MenuAction menuAction = null;

        menuAction =
                new MenuAction("Help ... ") {

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        try {
                            BrowserLauncher.openURL(SERVER_BASE_URL + "igv/doc/userguide.html");
                        } catch (IOException ex) {
                            log.error("Error opening browser", ex);
                        }

                    }
                };
        menuAction.setToolTipText(UIConstants.HELP_TOOLTIP);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));


        menuAction =
                new MenuAction("Tutorial ... ") {

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        try {
                            BrowserLauncher.openURL(SERVER_BASE_URL + "igv/doc/quickstart.html");
                        } catch (IOException ex) {
                            log.error("Error opening browser", ex);
                        }

                    }
                };
        menuAction.setToolTipText(UIConstants.TUTORIAL_TOOLTIP);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        menuAction =
                new MenuAction("About IGV ") {

                    @Override
                    public void actionPerformed(ActionEvent e) {
                        (new AboutDialog(IGVMainFrame.this, true)).setVisible(true);
                    }
                };
        menuAction.setToolTipText(UIConstants.ABOUT_TOOLTIP);
        menuItems.add(MenuAndToolbarUtils.createMenuItem(menuAction));

        MenuAction toolMenuAction = new MenuAction("Help");
        return MenuAndToolbarUtils.createMenu(menuItems, toolMenuAction);
    }

    public void enableRemoveGenomes() {
        if (removeImportedGenomeAction != null) {
            removeImportedGenomeAction.setEnabled(true);
        }

    }

    /**
     * TODO  Remove this method and replace all references to with
     * a direct call to clearImageCacheWithNoRepaint();
     * @deprecated
     */
    public void fireViewChangedEvent() {
        clearImageCacheWithNoRepaint();
    }

    /**
     * Select a genome
     */
    final public void doChooseGenome(GenomeDescriptor genomeType) {

        CursorToken token = null;
        try {

            token = WaitCursorManager.showWaitCursor();

            if (genomeType != null) {

                final String genomeId = genomeType.getId();
                setCurrentGenome(genomeId);
                PreferenceManager.getInstance().setDefaultGenome(genomeId);
            }

        } finally {
            WaitCursorManager.removeWaitCursor(token);
        }

    }

    private void setCurrentGenome(final String genomeId) {

        String currentGenomeId = IGVModel.getInstance().getViewContext().getGenomeId();
        if (currentGenomeId != null && genomeId.equalsIgnoreCase(currentGenomeId)) {
            // Nothing to do if genome already loaded
            return;
        }

        GuiUtilities.invokeOnEventThread(new Runnable() {

            public void run() {

                CursorToken token = null;
                try {
                    token = WaitCursorManager.showWaitCursor();

                    String currentGenomeId = IGVModel.getInstance().getViewContext().getGenomeId();
                    if (currentGenomeId != null && genomeId.equalsIgnoreCase(currentGenomeId)) {
                        // Nothing to do if genome already loaded
                        return;
                    }

                    setGenomeId(genomeId);
                    IGVMainFrame.theInstance.doRefresh();
                } finally {
                    WaitCursorManager.removeWaitCursor(token);
                }

            }
        });
    }

    /**
     * Open the user preferences dialog
     */
    final public void doViewPreferences() {

        GuiUtilities.invokeOnEventThread(new Runnable() {

            public void run() {

                boolean originalSingleTrackValue = PreferenceManager.getInstance().getShowSingleTrackPane();
                boolean originalShowAttributeValue = PreferenceManager.getInstance().getShowAttributeView();

                PreferencesEditor dialog = new PreferencesEditor(IGVMainFrame.this, true);
                dialog.setVisible(true);


                if (dialog.isCanceled()) {
                    resetStatusMessage();
                    return;

                }


                try {

                    // Should missing data be shown on track panel
                    boolean value =
                            PreferenceManager.getInstance().getShowMissingData();
                    MiscStuff.setShowMissingDataEnabled(value);

                    //Should data and feature panels be combined ?
                    boolean singlePanel =
                            PreferenceManager.getInstance().getShowSingleTrackPane();
                    if (originalSingleTrackValue != singlePanel) {
                        JOptionPane.showMessageDialog(IGVMainFrame.this, "Panel option change will take affect after restart.");
                    }

                    if (originalShowAttributeValue !=
                            PreferenceManager.getInstance().getShowAttributeView()) {
                        packViews();
                    }

                } finally {

                    // Update the state of the current tracks for drawing purposes
                    updateTrackState();
                    resetStatusMessage();

                }



            }
        });
    }

    final public void doExitApplication() {

        try {

            ((ApplicationStatusBar) statusBar).setMessage("Exiting...");

            // Store recent sessions
            if (!recentSessionList.isEmpty()) {

                int size = recentSessionList.size();
                if (size > IGVConstants.NUMBER_OF_RECENT_SESSIONS_TO_LIST) {
                    size = IGVConstants.NUMBER_OF_RECENT_SESSIONS_TO_LIST;
                }

                String recentSessions = "";
                for (int i = 0; i <
                        size; i++) {
                    recentSessions += recentSessionList.get(i);

                    if (i < (size - 1)) {
                        recentSessions += ";";
                    }

                }
                PreferenceManager.getInstance().remove(PreferenceManager.RECENT_SESSION_KEY);
                PreferenceManager.getInstance().setRecentSessions(recentSessions);
            }

// Save application location and size
            PreferenceManager.getInstance().setApplicationFrameBounds(getBounds());

            // Hide and close the application
            setVisible(false);
        } finally {
            System.exit(0);
        }

    }

    final public void doShowAttributeDisplay(boolean enableAttributeView) {

        boolean oldState =
                PreferenceManager.getInstance().getShowAttributeView();

        // First store the newly requested state
        PreferenceManager.getInstance().setShowAttributeView(enableAttributeView);

        showAttributeMenuItem.setSelected(enableAttributeView);

        // Now, if the state has actually change we
        // need to refresh everything
        if (oldState != enableAttributeView) {
            packViews();
            doRefresh();

        }








    }

    final public void doRefreshAndImageCacheClear() {
        clearImageCacheForTrackPanels();
        doRefresh();

    }

    final public void doRefresh() {

        GuiUtilities.invokeOnEventThread(new Runnable() {

            public void run() {
                try {
                    doResizeTrackPanels();
                    packViews();

                    getContentPane().validate();
                    getContentPane().repaint();
                } finally {
                    resetStatusMessage();
                }

            }
        });
    }

    final public void refreshFeatureTrackView() {

        GuiUtilities.invokeOnEventThread(new Runnable() {

            public void run() {
                try {
                    packViews();
                    for (TrackSetScrollPane sp : trackSetScrollPanes.values()) {
                        sp.getTrackSetView().validate();
                    }

                } finally {
                    resetStatusMessage();
                }

            }
        });

    }

    final public void doResizeTrackPanels() {
        clearImageCacheWithNoRepaint();
        GuiUtilities.invokeOnEventThread(new Runnable() {

            public void run() {

                for (TrackSetScrollPane tsp : trackSetScrollPanes.values()) {
                    DataPanel dp = tsp.getDataPanel();
                    dp.doResize();
                    dp.validate();
                    dp.repaint();
                }

            }
        });
    }

// TODO -- move all of this attribute stuf out of IGVMainFrame,  perhaps to
// some Attribute helper class.
    final public void doSelectDisplayableAttribute() {

        GuiUtilities.invokeOnEventThread(new Runnable() {

            public void run() {

                try {


                    ((ApplicationStatusBar) statusBar).setMessage("Select Attributes to Show...");

                    attributeCheckList.update();
                    attributeCheckList.setEditing(true);
                    int status =
                            JOptionPane.showConfirmDialog(
                            IGVMainFrame.this,
                            attributeCheckList,
                            "Select Attributes to Show",
                            JOptionPane.OK_CANCEL_OPTION,
                            JOptionPane.PLAIN_MESSAGE,
                            null);

                    if (status == JOptionPane.CANCEL_OPTION ||
                            status == JOptionPane.CLOSED_OPTION) {

                        attributeCheckList.cancelChanges();
                        attributeCheckList.setEditing(false);
                        return;

                    }






                    attributeCheckList.setEditing(false);
                    attributeCheckList.update();

                    // Save the user's choices
                    HashSet<String> attributeNames =
                            attributeCheckList.getSelectedAttributes();

                    PreferenceManager.getInstance().setDisplayableAttributes(
                            attributeNames);

                    AttributeManager.getInstance().
                            firePropertyChange(
                            AttributeManager.getInstance(),
                            AttributeManager.ATTRIBUTES_NARROWED_PROPERTY,
                            null,
                            null);

                } finally {

                    attributeCheckList.setEditing(false);

                    // Refresh view
                    packViews();

                    doRefresh();

                    resetStatusMessage();

                }






            }
        });
    }

    private void setupDisplayableAttributePreferences(List<String> attributeNames) {

        String displayableAttributes =
                PreferenceManager.getInstance().getDisplayableAttributes();
        setupDisplayableAttributePreferences(attributeNames, displayableAttributes);
    }

    public void setupDisplayableAttributePreferences(List<String> attributeNames,
            String displayableAttributes) {

        // If first time a preference was set default attributes to dispayable
        Boolean defaultCheckState =
                PreferenceManager.getInstance().isDisplayableAttributesNotSet();

        setupDisplayableAttributePreferences(attributeNames,
                displayableAttributes, defaultCheckState, false);
    }

    public void setupDisplayableAttributePreferences(List<String> attributeNames,
            String displayableAttributes, Boolean defaultCheckState, boolean clearList) {

        if (clearList) {
            attributeCheckList.clear();
            attributeCheckList =
                    null;
        }

// Load requested preferences first
        if (attributeCheckList == null) {

            attributeCheckList = new AttributeCheckList(defaultCheckState);

            if (displayableAttributes != null) {
                String[] attributes = displayableAttributes.split(",");
                List<String> names = new ArrayList<String>();

                for (String attribute : attributes) {
                    names.add(attribute);
                }

                attributeCheckList.addItems(names, true);
            }

        }

        // Any items not saved in the preferences as displayable
        // are handled here
        if (attributeNames != null) {
            attributeCheckList.addItems(attributeNames, defaultCheckState);
        }

        attributeCheckList.sort();
    }

    /**
     * Return the subset of attribute keys that shoould be displayed in the
     * attribute panel.  By default this returns all keys not beggining with
     * a # sign, but the user can select a subset.
     *
     * @return
     */
    public List<String> getDisplayableAttributes() {

        if (attributeCheckList == null) {
            setupDisplayableAttributePreferences(null);
        }

        List<String> attributesToShow = new ArrayList<String>();
        HashSet<String> selectedAttributes = attributeCheckList.getSelectedAttributes();
        if (!selectedAttributes.isEmpty()) {
            List<String> attributeKeys =
                    AttributeManager.getInstance().getAttributeKeys();

            for (String attribute : attributeKeys) {
                if (!attribute.startsWith("#") && selectedAttributes.contains(attribute.trim())) {
                    attributesToShow.add(attribute);
                }

            }
        }

        Collections.sort(attributesToShow,
                AttributeManager.getInstance().getAttributeComparator());
        return attributesToShow;
    }

    final public void doApplicationSnapshot() {

        GuiUtilities.invokeOnEventThread(new Runnable() {

            public void run() {

                ((ApplicationStatusBar) statusBar).setMessage("Creating snapshot...");
                File defaultFile = new File("igv_snapshot.png");
                try {
                    isExportingSnapshot = true;
                    createSnapshot(centerPanel, defaultFile);
                } finally {
                    isExportingSnapshot = false;
                }

                resetStatusMessage();
            }
        });
    }

    public boolean isExportingSnapshot() {
        return isExportingSnapshot;
    }

    final public void createSnapshot(final Component target,
            final File defaultFile) {

        LongRunningTask.submit(new Runnable() {

            public void run() {
                try {
                    ((ApplicationStatusBar) statusBar).setMessage("Creating snapshot " +
                            defaultFile.getAbsolutePath());
                    File file = selectSnapshotFile(defaultFile);
                    if (file == null) {
                        return;
                    }

                    createSnapshotNonInteractive(target, file);
                } finally {
                    resetStatusMessage();
                }

            }
        });

    }

    public void createSnapshotNonInteractive(File file) {
        createSnapshotNonInteractive(centerPanel, file);
    }

    protected void createSnapshotNonInteractive(Component target, File file) {

        String extension =
                GenericUtilities.getFileExtension(file.getAbsolutePath());

        // Use default extension if file has none
        if (extension == null) {

            FileFilter filter = (FileFilter) snapshotFileChooser.getFileFilter();

            // Figure out the proper extension
            if (!(filter instanceof SnapshotFileFilter)) {
                extension = SnapshotFileType.PNG.getExtension();
            } else {
                extension = ((SnapshotFileFilter) filter).getExtension();
            }

            file = new File((file.getAbsolutePath() + extension));
        }

        SnapshotFileType type = GenericUtilities.getSnapshotFileType(extension);

        // If valid extension
        if (type != SnapshotFileType.NULL) {
            // Take the snapshot
            try {
                RepaintManager.currentManager(
                        getContentPane()).setDoubleBufferingEnabled(false);
                doComponentSnapshot(target, file, type);

            } finally {
                if (!IGVConstants.IS_MAC) {
                    RepaintManager.currentManager(
                            getContentPane()).setDoubleBufferingEnabled(true);
                }

            }
        }
    }

    public File selectSnapshotFile(
            File defaultFile) {

        SnapshotFileFilter snapshotFileFilter = null;
        if (defaultFile != null) {

            String fileExtension =
                    GenericUtilities.getFileExtension(defaultFile.getAbsolutePath());
            snapshotFileFilter =
                    GenericUtilities.getSnapshotFileFilterForType(
                    GenericUtilities.getSnapshotFileType(fileExtension));
        }

        snapshotFileChooser.setFileFilter(snapshotFileFilter);
        snapshotFileChooser.setSelectedFile(defaultFile);

        // Display the dialog
        snapshotFileChooser.showSaveDialog(this);

        resetStatusMessage();

        File file = snapshotFileChooser.getSelectedFile();

        // If a file selection was made
        if (file != null) {

            File directory = snapshotFileChooser.getCurrentDirectory();
            if (directory != null) {
                PreferenceManager.getInstance().setLastSnapshotDirectory(
                        directory);
            }

        }

        return file;
    }

// TODO -- MOVE ALL THESE GENOME METHODS OUT OF IGV MAIN FRAME !!!
    final public GenomeDescriptor chooseGenomeType(
            boolean forceSelection) {

        GenomeDescriptor type = null;
        final GenomeSelectionDialog genomeDialog =
                new GenomeSelectionDialog(this, true);

        Map<String, GenomeDescriptor> map =
                GenomeManager.getInstance().getGenomeDescriptorMap();
        List<GenomeDescriptor> descriptors = new ArrayList(map.values());
        GenomeManager.getInstance().sort(descriptors);
        Object[] typeArray = descriptors.toArray();

        genomeDialog.setModel(new DefaultComboBoxModel(typeArray));
        genomeDialog.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
        genomeDialog.setCancelingEnabled(!forceSelection);

        String genome = PreferenceManager.getInstance().getDefaultGenome();
        genomeDialog.setDefaultGenome((GenomeDescriptor) map.get(genome));


        genomeDialog.setVisible(true);

        CursorToken token = null;
        try {
            token = WaitCursorManager.showWaitCursor();
            type =
                    genomeDialog.getSelectedItem();
        } finally {
            WaitCursorManager.removeWaitCursor(token);
            resetStatusMessage();

        }







        return type;
    }

    private void intializeDefaultGenome()
            throws FileNotFoundException, NoRouteToHostException {
        igvCommandBar.intializeDefaultGenome(null);
    }

    public void setGenomeId(String id) {
        //synchronized(this) {

        String currentGenomeId = IGVModel.getInstance().getViewContext().getGenomeId();
        if (currentGenomeId != null && id.equalsIgnoreCase(currentGenomeId)) {
            // Nothing to do if genome already loaded
            return;
        }

        model.getViewContext().setGenomeId(id);
        loadGeneTrack(id);
        FeatureDB.clearFeatures();
        //}

        if (igvCommandBar != null) {
            igvCommandBar.updateChromosomeDropdown();
        }

        PreferenceManager.getInstance().setDefaultGenome(id);
    }

    private void createZoomCursors() throws HeadlessException, IndexOutOfBoundsException {
        if (zoomInCursor == null || zoomOutCursor == null) {
            final Image zoomInImage =
                    ((ImageIcon) IconFactory.getInstance().getIcon(
                    IconFactory.IconID.ZOOM_IN)).getImage();
            final Image zoomOutImage =
                    ((ImageIcon) IconFactory.getInstance().getIcon(
                    IconFactory.IconID.ZOOM_OUT)).getImage();
            final Point hotspot = new Point(10, 10);
            zoomInCursor =
                    getToolkit().createCustomCursor(
                    zoomInImage, hotspot, "Zoom in");
            zoomOutCursor =
                    getToolkit().createCustomCursor(
                    zoomOutImage, hotspot, "Zoom out");

        }

    }

    private void createHandCursor() throws HeadlessException, IndexOutOfBoundsException {
        if (handCursor == null) {
            BufferedImage handImage = new BufferedImage(32, 32, BufferedImage.TYPE_INT_ARGB);

            // Make backgroun transparent
            Graphics2D g = handImage.createGraphics();
            g.setComposite(AlphaComposite.getInstance(
                    AlphaComposite.CLEAR, 0.0f));
            Rectangle2D.Double rect = new Rectangle2D.Double(0, 0, 32, 32);
            g.fill(rect);

            // Draw hand image in middle
            g =
                    handImage.createGraphics();
            g.drawImage(IconFactory.getInstance().getIcon(
                    IconFactory.IconID.OPEN_HAND).getImage(), 0, 0, null);
            handCursor =
                    getToolkit().createCustomCursor(
                    handImage, new Point(8, 6), "Move");
        }

        if (fistCursor == null) {
            BufferedImage handImage = new BufferedImage(32, 32, BufferedImage.TYPE_INT_ARGB);

            // Make backgroun transparent
            Graphics2D g = handImage.createGraphics();
            g.setComposite(AlphaComposite.getInstance(
                    AlphaComposite.CLEAR, 0.0f));
            Rectangle2D.Double rect = new Rectangle2D.Double(0, 0, 32, 32);
            g.fill(rect);

            // Draw hand image in middle
            g =
                    handImage.createGraphics();
            g.drawImage(IconFactory.getInstance().getIcon(
                    IconFactory.IconID.FIST).getImage(), 0, 0, null);
            fistCursor =
                    getToolkit().createCustomCursor(
                    handImage, new Point(8, 6), "Move");
        }

    }

    private void createDragAndDropCursor()
            throws HeadlessException, IndexOutOfBoundsException {

        if (dragNDropCursor == null) {
            ImageIcon icon =
                    IconFactory.getInstance().getIcon(
                    IconFactory.IconID.DRAG_AND_DROP);

            int width = icon.getIconWidth();
            int height = icon.getIconHeight();

            BufferedImage dragNDropImage =
                    new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);

            // Make background transparent
            Graphics2D g = dragNDropImage.createGraphics();
            g.setComposite(AlphaComposite.getInstance(
                    AlphaComposite.CLEAR, 0.0f));
            Rectangle2D.Double rect = new Rectangle2D.Double(0, 0, width, height);
            g.fill(rect);

            // Draw DND image
            g =
                    dragNDropImage.createGraphics();
            Image image = icon.getImage();
            g.drawImage(image, 0, 0, null);
            dragNDropCursor =
                    getToolkit().createCustomCursor(
                    dragNDropImage, new Point(0, 0), "Drag and Drop");
        }

    }

    public void packViews() {
        ((View) applicationHeaderView).packView();
        for (TrackSetScrollPane tsv : trackSetScrollPanes.values()) {
            tsv.getTrackSetView().packView();
        }

    }

    public void reset() {
        reset(true);
    }

    public void reset(boolean initializeGenome) {

        attributeCheckList.clear();
        setCurrentSessionFilePath(null);
        setTitle(IGVConstants.APPLICATION_NAME);

        if (filterTracksAction != null) {
            filterTracksAction.resetTrackFilter();
        }

        AttributeManager.getInstance().clearAllAttributes();
        IGVModel.getInstance().clearRegionsOfInterest();
        TrackManager.getInstance().resetApplication(!initializeGenome);

        // Local values that need to be reset
        loadedTrackFiles.clear();

        // Reload the Default Genome
        if (initializeGenome) {

            try {
                igvCommandBar.initializeGenomeList();
                intializeDefaultGenome();

            } catch (FileNotFoundException e) {
                log.error("File not found while intializing genome!", e);
            } catch (NoRouteToHostException e) {
                log.error("Error while intializing genome!", e);
            } catch (Exception e) {
                log.error("Error while intializing genome!", e);
            }

        }

        // Remove user added panels
        for (TrackSetScrollPane tsp : trackSetScrollPanes.values()) {
            if (tsp == dataTrackScrollPane || tsp == featureTrackScrollPane) {
                continue;
            }

            centerSplitPane.remove(tsp);
        }

        trackSetScrollPanes.clear();
        trackSetScrollPanes.put(DATA_PANEL_NAME, (TrackSetScrollPane) dataTrackScrollPane);
        if (!TrackManager.getInstance().isSinglePanel()) {
            trackSetScrollPanes.put(FEATURE_PANEL_NAME, (TrackSetScrollPane) featureTrackScrollPane);
        }


// Re-Layout the track panel
        doResizeTrackPanels();

// Refresh Entire view
        doRefresh();

    }

    /**
     * Set the status bar message.  If the message equals "Done." intercept
     * and reset to the default "quite" message,  currently the number of tracks
     * loaded.
     * @param message
     */
    public void setStatusBarMessage(String message) {
        if (message.equals("Done.")) {
            resetStatusMessage();
        }

        ((ApplicationStatusBar) statusBar).setMessage(message);
    }

    /**
     * Resets factory settings. this is not the same as reset user defaults
     * DO NOT DELETE used when debugging
     */
    public void resetToFactorySettings() {

        try {
            PreferenceManager.getInstance().clear();
            boolean isShow =
                    PreferenceManager.getInstance().getShowAttributeView();
            boolean isShowROI =
                    PreferenceManager.getInstance().getShowRegionTool();

            doShowAttributeDisplay(isShow);

            doRefresh();

        } catch (Exception e) {
            String message = "Failure while resetting preferences!";
            showAndLogErrorMessage(IGVMainFrame.theInstance, message, log, e);
        }

    }

    public void updateTrackState() {
        clearImageCacheWithNoRepaint();
        for (TrackSetScrollPane sp : trackSetScrollPanes.values()) {
            sp.getDataPanel().doResize();
        }

        doRefresh();

    }

    public TrackFilter getTrackFilter() {

        if (filterTracksAction == null) {
            return null;
        } else {
            return filterTracksAction.getTrackFilter();
        }

    }

    public void setTrackFilter(TrackFilter trackFilter) {

        if (filterTracksAction != null) {
            // TODO -- HOLDING THIS STATE ON THE ACTION OBJECT SEEMS BAD
            filterTracksAction.setTrackFilter(trackFilter);
            // Update the state of the current tracks
            updateTrackState();

        }






    }

    public void setFilterMatchAll(boolean value) {
        if (filterTracksAction != null) {
            filterTracksAction.setFilterMatchAll(value);
        }

    }

    public boolean isFilterMatchAll() {
        if (filterTracksAction != null) {
            return filterTracksAction.isFilterMatchAll();
        }

        return false;
    }

    public void setFilterShowAllTracks(boolean value) {
        if (filterTracksAction != null) {
            filterTracksAction.setFilterShowAllTracks(value);
        }

    }

    public boolean isFilterShowAllTracks() {
        if (filterTracksAction != null) {
            return filterTracksAction.getShowAllTracksFilterCheckBox().isSelected();
        }

        return false;
    }

    public void setCurrentLocus(final String locusString) {

        if (locusString == null || locusString.trim().length() < 1) {
            return;
        }

        igvCommandBar.searchByLocus(locusString);
    }

    /**
     * Add a new data panel set
     */
    public void addDataPanel(String name) {
        TrackSetView tsv = new TrackSetView(name);
        final TrackSetScrollPane sp = new TrackSetScrollPane();
        sp.setViewportView(tsv);
        sp.setPreferredSize(new Dimension(700, 300));
        AttributeManager.getInstance().addPropertyChangeListener(sp.getAttributePanel());

        for (TrackSetScrollPane tsp : trackSetScrollPanes.values()) {
            tsp.minimizeHeight();
        }

        trackSetScrollPanes.put(name, sp);
        TrackManager.getInstance().initGroupsForPanel(name);

        GuiUtilities.invokeOnEventThread(new Runnable() {

            public void run() {
                // TODO Resize the data panel to make as much space as possible
                //DataPanel dp = ((TrackSetScrollPane) dataTrackScrollPane).getDataPanel();
                //dp.doResize();


                // Insert the new panel just before the feature panel, or at the end if there is no feature panel.
                int featurePaneIdx = centerSplitPane.indexOfPane(featureTrackScrollPane);
                if (featurePaneIdx > 0) {
                    centerSplitPane.insertPane(sp, featurePaneIdx);
                } else {
                    centerSplitPane.add(sp);
                }

// Resize data panel, unless single pane
                if (!PreferenceManager.getInstance().isShowSingleTrackPane()) {
                    if (TrackManager.getInstance().getTracksForPanel(IGVMainFrame.DATA_PANEL_NAME).size() == 0) {
                        centerSplitPane.setDividerLocation(0, 3);
                    } else {
                        // TrackSetScrollPane dsp = trackSetScrollPanes.get(IGVMainFrame.DATA_PANEL_NAME);
                        // int dh = Math.max(50, dsp.getDataPanel().getHeight());
                        // int dph = dsp.getTrackSetView().getPreferredPanelHeight();
                        // int nh = Math.min(dh, Math.max(50, dph));
                    }
                }

                packViews();
            }
        });
    }

    public void removeDataPanel(String name) {
        TrackSetScrollPane sp = trackSetScrollPanes.get(name);
        if (sp != null) {
            centerSplitPane.remove(sp);
            AttributeManager.getInstance().removePropertyChangeListener(sp.getAttributePanel());
            trackSetScrollPanes.remove(name);
        }

    }

    public boolean scrollToTrack(String trackName) {
        for (TrackSetScrollPane sp : trackSetScrollPanes.values()) {
            if (sp.getNamePanel().scrollTo(trackName)) {
                return true;
            }

        }
        return false;
    }

    private void initComponents() {

        setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);

        infoPanel =
                new InfoPanel();
        attributeHeaderPanel =
                new AttributeHeaderPanel();
        cytobandPanel =
                new CytobandPanel();
        rulerPanel =
                new RulerPanel();
        regionOfInterestPane =
                new RegionOfInterestPane();

        headerPanel =
                new HeaderPanel();
        headerPanel.setBackground(new java.awt.Color(255, 255, 255));
        headerPanel.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        headerPanel.setDebugGraphicsOptions(javax.swing.DebugGraphics.NONE_OPTION);
        headerPanel.setMinimumSize(new java.awt.Dimension(700, 0));
        headerPanel.setPreferredSize(new java.awt.Dimension(0, 0));
        headerPanel.setLayout(new java.awt.BorderLayout());

        headerScrollPane =
                new JideScrollPane();
        headerScrollPane.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(204, 204, 204), 2));
        headerScrollPane.setForeground(new java.awt.Color(102, 102, 102));
        headerScrollPane.setHorizontalScrollBarPolicy(javax.swing.ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
        headerScrollPane.setVerticalScrollBarPolicy(javax.swing.ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
        headerScrollPane.setDoubleBuffered(true);
        headerScrollPane.setMinimumSize(new java.awt.Dimension(0, 0));
        headerScrollPane.setPreferredSize(new java.awt.Dimension(1019, 130));

        cytobandPanel.setBackground(new java.awt.Color(255, 255, 255));
        cytobandPanel.setAlignmentX(0.0F);
        cytobandPanel.setAlignmentY(0.0F);
        cytobandPanel.setDebugGraphicsOptions(javax.swing.DebugGraphics.NONE_OPTION);
        cytobandPanel.setOpaque(false);
        cytobandPanel.setPreferredSize(new java.awt.Dimension(0, 50));
        cytobandPanel.setRequestFocusEnabled(false);
        cytobandPanel.setLayout(null);
        headerPanel.add(cytobandPanel, java.awt.BorderLayout.NORTH);

        rulerPanel.setBackground(new java.awt.Color(255, 255, 255));
        rulerPanel.setDebugGraphicsOptions(javax.swing.DebugGraphics.NONE_OPTION);
        rulerPanel.setMinimumSize(new java.awt.Dimension(100, 500));
        rulerPanel.setOpaque(false);
        rulerPanel.setPreferredSize(new java.awt.Dimension(100, 500));
        rulerPanel.setLayout(null);
        headerPanel.add(rulerPanel, java.awt.BorderLayout.CENTER);


        applicationHeaderView =
                new View();
        applicationHeaderView.setDebugGraphicsOptions(javax.swing.DebugGraphics.NONE_OPTION);
        applicationHeaderView.setMaximumSize(new java.awt.Dimension(2147483647, 110));
        applicationHeaderView.setMinimumSize(new java.awt.Dimension(1000, 135));
        applicationHeaderView.setPreferredSize(new java.awt.Dimension(1000, 110));
        applicationHeaderView.setLayout(new org.netbeans.lib.awtextra.AbsoluteLayout());

        attributeHeaderPanel.setBackground(new java.awt.Color(255, 255, 255));
        attributeHeaderPanel.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        attributeHeaderPanel.setDebugGraphicsOptions(javax.swing.DebugGraphics.NONE_OPTION);
        attributeHeaderPanel.setMinimumSize(new java.awt.Dimension(0, 0));
        attributeHeaderPanel.setPreferredSize(new java.awt.Dimension(0, 0));
        attributeHeaderPanel.setLayout(new java.awt.GridLayout(1, 3));

        infoPanel.setBackground(new java.awt.Color(255, 255, 255));
        infoPanel.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(0, 0, 0)));
        infoPanel.setDebugGraphicsOptions(javax.swing.DebugGraphics.NONE_OPTION);
        infoPanel.setMinimumSize(new java.awt.Dimension(0, 0));
        infoPanel.setPreferredSize(new java.awt.Dimension(0, 0));
        infoPanel.setLayout(null);

        applicationHeaderView.add(infoPanel, new org.netbeans.lib.awtextra.AbsoluteConstraints(0, 0, 150, 125));
        applicationHeaderView.add(attributeHeaderPanel, new org.netbeans.lib.awtextra.AbsoluteConstraints(160, 0, 120, 125));

        regionOfInterestPane.setBackground(new java.awt.Color(255, 255, 255));
        regionOfInterestPane.setDebugGraphicsOptions(javax.swing.DebugGraphics.NONE_OPTION);
        regionOfInterestPane.setMinimumSize(new java.awt.Dimension(0, 13));
        regionOfInterestPane.setOpaque(false);
        GroupLayout regionOfInterestPaneLayout = new org.jdesktop.layout.GroupLayout(regionOfInterestPane);
        regionOfInterestPane.setLayout(regionOfInterestPaneLayout);
        regionOfInterestPaneLayout.setHorizontalGroup(
                regionOfInterestPaneLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING).add(0, 708, Short.MAX_VALUE));
        regionOfInterestPaneLayout.setVerticalGroup(
                regionOfInterestPaneLayout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING).add(0, 10, Short.MAX_VALUE));

        headerPanel.add(regionOfInterestPane, java.awt.BorderLayout.SOUTH);

        applicationHeaderView.add(headerPanel, new org.netbeans.lib.awtextra.AbsoluteConstraints(290, 0, 710, 125));
        headerScrollPane.setViewportView(applicationHeaderView);

        centerPanel =
                new javax.swing.JPanel();
        centerPanel.setBackground(new java.awt.Color(204, 204, 204));
        centerPanel.setPreferredSize(new java.awt.Dimension(1021, 510));
        centerPanel.setLayout(new java.awt.BorderLayout());
        centerPanel.add(headerScrollPane, java.awt.BorderLayout.NORTH);

        dataTrackScrollPane =
                new TrackSetScrollPane();
        JPanel dataTrackView = new TrackSetView(DATA_PANEL_NAME);


        dataTrackScrollPane.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(102, 102, 102)));
        dataTrackScrollPane.setForeground(new java.awt.Color(153, 153, 153));
        dataTrackScrollPane.setHorizontalScrollBarPolicy(javax.swing.ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
        dataTrackScrollPane.setVerticalScrollBarPolicy(javax.swing.ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
        dataTrackScrollPane.setDebugGraphicsOptions(javax.swing.DebugGraphics.NONE_OPTION);
        dataTrackScrollPane.setDoubleBuffered(true);
        dataTrackScrollPane.setMinimumSize(new java.awt.Dimension(0, 0));
        dataTrackScrollPane.setPreferredSize(new java.awt.Dimension(1021, 349));

        dataTrackView.setMinimumSize(new java.awt.Dimension(1000, 0));
        dataTrackView.setPreferredSize(new java.awt.Dimension(1000, 345));
        dataTrackView.setLayout(null);
        dataTrackScrollPane.setViewportView(dataTrackView);

        centerPanel.add(dataTrackScrollPane, java.awt.BorderLayout.CENTER);

        if (!PreferenceManager.getInstance().isShowSingleTrackPane()) {
            featureTrackScrollPane = new TrackSetScrollPane();
            JPanel featureTrackView = new TrackSetView(FEATURE_PANEL_NAME);
            featureTrackScrollPane.setBorder(javax.swing.BorderFactory.createLineBorder(new java.awt.Color(102, 102, 102)));
            featureTrackScrollPane.setHorizontalScrollBarPolicy(javax.swing.ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
            featureTrackScrollPane.setVerticalScrollBarPolicy(javax.swing.ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
            featureTrackScrollPane.setDebugGraphicsOptions(javax.swing.DebugGraphics.NONE_OPTION);
            featureTrackScrollPane.setDoubleBuffered(true);
            featureTrackScrollPane.setFocusTraversalPolicyProvider(true);
            featureTrackScrollPane.setMinimumSize(new java.awt.Dimension(0, 0));
            featureTrackScrollPane.setOpaque(false);
            featureTrackScrollPane.setPreferredSize(new java.awt.Dimension(1021, 50));

            featureTrackView.setDebugGraphicsOptions(javax.swing.DebugGraphics.NONE_OPTION);
            featureTrackView.setMinimumSize(new java.awt.Dimension(1000, 0));
            featureTrackView.setPreferredSize(new java.awt.Dimension(1000, 115));
            featureTrackView.setLayout(null);
            featureTrackScrollPane.setViewportView(featureTrackView);

            centerPanel.add(featureTrackScrollPane, java.awt.BorderLayout.SOUTH);
        }

        getContentPane().add(centerPanel, java.awt.BorderLayout.CENTER);

        statusBar =
                new ApplicationStatusBar();
        statusBar.setDebugGraphicsOptions(javax.swing.DebugGraphics.NONE_OPTION);
        getContentPane().add(statusBar, java.awt.BorderLayout.SOUTH);

        pack();

    }

    public String getCurrentSessionFilePath() {
        return currentSessionFilePath;

    }

    public void setCurrentSessionFilePath(String sessionFilePath) {
        if (sessionFilePath == null) {
            currentSessionFilePath = null;
            setTitle(IGVConstants.APPLICATION_NAME);
        } else {
            currentSessionFilePath = sessionFilePath.trim();
            setTitle(IGVConstants.APPLICATION_NAME + " - Session: " + currentSessionFilePath);
        }

    }

    final public void doRestoreSession(final File sessionFile,
            final String locus) {
        doRestoreSession(sessionFile, locus, true);

    }

    final public void doRestoreSession(final File sessionFile,
            final String locus,
            boolean doInBackgrond) {

        String filePath = "";
        if (sessionFile != null) {

            Runnable runnable = new Runnable() {

                public void run() {
                    InputStream inputStream = null;
                    try {
                        inputStream = new BufferedInputStream(new FileInputStream(sessionFile));
                        doRestoreSession(inputStream, sessionFile.toString(), locus);
                        String sessionFilePath = sessionFile.getAbsolutePath();
                        setCurrentSessionFilePath(sessionFilePath);
                        if (!recentSessionList.contains(sessionFilePath)) {
                            recentSessionList.addFirst(sessionFilePath);
                        }

                    } catch (Exception e) {
                        String message = "Failed to load session! : " + sessionFile.getAbsolutePath();
                        showAndLogErrorMessage(IGVMainFrame.this, message, log, e);
                    } finally {
                        if (inputStream != null) {
                            try {
                                inputStream.close();
                            } catch (IOException iOException) {
                                log.error("Error closing session stream", iOException);
                            }

                        }
                    }
                }
            };

            if (doInBackgrond) {
                LongRunningTask.submit(runnable);
            } else {
                runnable.run();
            }

        } else {
            String message = "Session file does not exist! : " + filePath;
            showAndLogErrorMessage(IGVMainFrame.this, message, log);
        }

    }

    /**
     * TODO -- this is nearly an exact copy of the doRestoreSession(File sessionFile)
     * method.  Refactor to combine these using streams.
     *
     * @param sessionURL
     */
    final public void doRestoreSession(final URL sessionURL,
            final String locus) {


        if (sessionURL != null) {
            InputStream inputStream = null;
            try {
                inputStream = new BufferedInputStream(sessionURL.openStream());
                doRestoreSession(inputStream, URLDecoder.decode(sessionURL.getFile(), "UTF-8"),
                        locus);
            } catch (Exception e) {
                String message = "Failed to load session! : " + sessionURL;
                showAndLogErrorMessage(IGVMainFrame.this, message, log, e);
            } finally {

                if (inputStream != null) {
                    try {
                        inputStream.close();
                    } catch (IOException iOException) {
                        log.error("Error closing session stream", iOException);
                    }

                }
            }


        } else {
            String message = "Session file does not exist! : ";
            try {
                message += URLDecoder.decode(sessionURL.getFile(), "UTF-8");
            } catch (UnsupportedEncodingException ex) {
                message += sessionURL.getFile();
            }

            showAndLogErrorMessage(IGVMainFrame.this, message, log);
        }

    }

    final public void doRestoreSession(final InputStream inputStream,
            final String sessionName,
            final String locus) {

        try {
            setStatusBarMessage("Opening session...");


            reset(false); // Clear everything but the genome
            Session session = new Session(sessionName);


            SessionManager.getInstance().loadSession(inputStream, session, sessionName);

            String searchText = locus == null ? session.getLocus() : locus;
            if (searchText != null && searchText.trim().length() > 0) {
                igvCommandBar.searchByLocus(searchText);
            }

            setTitle(IGVConstants.APPLICATION_NAME + " - Session: " + sessionName);

        } catch (Exception e) {
            String message = "Failed to load session! : " + sessionName;
            showAndLogErrorMessage(IGVMainFrame.this, message, log, e);
        } finally {


            resetStatusMessage();
        }

    }

    /**
     * Reset the default status message, which is the number of tracks loaded.
     */
    public void resetStatusMessage() {
        ((ApplicationStatusBar) statusBar).setMessage("" +
                TrackManager.getInstance().getTrackCount() + " tracks loaded");

    }

    public String getDisplayedLocusString() {
        return igvCommandBar.getSearchText();
    }

    public void rebuildGenomeDropdownList(Set excludedArchivesUrls) {
        igvCommandBar.rebuildGenomeItemList(excludedArchivesUrls);
    }

    public void showLoadedTrackCount() {
        ((ApplicationStatusBar) statusBar).setMessage("" +
                TrackManager.getInstance().getTrackCount() +
                " track(s) currently loaded");
    }

    /**
     * Disable direct draw for windows,  and quartz for macs.  These technologies don't work
     * well with java.
     */
    private void disableGraphicAccelerators() {

        // Disable quartz for mac, direct draw for windows
        if (IS_MAC) {
            System.setProperty("apple.awt.graphics.UseQuartz", "false");
        } else {
            System.setProperty("sun.java2d.noddraw", "true");
        }

    }

    public static void main(final String args[]) {

        Thread.setDefaultUncaughtExceptionHandler(new DefaultExceptionHandler());


        java.awt.EventQueue.invokeLater(new Runnable() {

            public void run() {
                com.jidesoft.utils.Lm.verifyLicense("The Broad Institute, MIT", "Gene Pattern",
                        "D.DQSR7z9m6fxL1IqWZ6svQFmE6vj3Q");

                if (IS_LINUX) {
                    try {
                        UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
                        UIManager.put("JideSplitPane.dividerSize", 5);
                        UIManager.put("JideSplitPaneDivider.background", Color.darkGray);


                    } catch (Exception exception) {
                        exception.printStackTrace();
                    }

                } else {
                    LookAndFeelFactory.installDefaultLookAndFeel();
                }

                LookAndFeelFactory.installJideExtension();

                IGVMainFrame frame = null;
                JWindow splashScreen = null;
                try {
                    /*if (args.length > 0)
                    { // Remove this if line when ready always show splash
                    // Need to show before the frame is ready
                    splashScreen = new JWindow();
                    splashScreen.setSize(400, 400);
                    splashScreen.setVisible(true);
                    splashScreen.setLocation(500, 500);
                    }
                     * */
                    frame = new IGVMainFrame();

                    frame.setupInitialGenome(args);


                    KeyboardFocusManager.getCurrentKeyboardFocusManager().addKeyEventDispatcher(new GlobalKeyDispatcher());
                //GlobalHotkeyManager.getInstance().configureIGV();



                } catch (Exception e) {

                    log.error("Fatal application error!", e);
                    System.exit(-1);
                } finally {
                    if (splashScreen != null) {
                        splashScreen.setVisible(false);
                    }

                }


            }
        });
    }

    /**
     * @return the trackSetScrollPanes
     */
    public Collection<TrackSetScrollPane> getTrackSetScrollPanes() {
        return trackSetScrollPanes.values();
    }
}
