/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.ui;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.Collection;
import java.util.HashSet;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileFilter;
import org.apache.log4j.Logger;
import org.broad.igv.IGVConstants;
import org.broad.igv.PreferenceManager;
import org.broad.igv.feature.CytoBandFileParser;
import org.broad.igv.feature.GeneManager;
import org.broad.igv.feature.GenomeImporter;
import org.broad.igv.feature.GenomeManager;
import org.broad.igv.ui.util.FileChooser;

import org.broad.igv.ui.WaitCursorManager.CursorToken;
import org.broad.igv.util.AsciiLineReader;
import static org.broad.igv.IGVConstants.*;

/**
 *
 * @author  eflakes
 */
public class GenomeBuilderPane extends javax.swing.JPanel {

    private static Logger logger = Logger.getLogger(GenomeBuilderPane.class);
    private String genomeArchiveLocation;
    private String genomeFilename;
    GenomeImporter importer;

    /** Creates new form GenomeBuilder */
    public GenomeBuilderPane() {
        initComponents();
        importer = new GenomeImporter();
    }

    public String getCytobandFileName() {
        String cytobandFile = cytobandFileTextField.getText();
        if (cytobandFile != null && cytobandFile.trim().equals(""))
        {
            cytobandFile = null;
        }
        return cytobandFile;
    }

    public String getRefFlatFileName() {
        String refFlatFile = refFlatFileTextField.getText();
        if (refFlatFile != null && refFlatFile.trim().equals(""))
        {
            refFlatFile = null;
        }
        return refFlatFile;
    }

    public String getFastaFileName() {
        String fastaFile = fastaFileTextField.getText();
        if (fastaFile != null && fastaFile.trim().equals(""))
        {
            fastaFile = null;
        }
        return fastaFile;
    }

    public String getGenomeId() {
        String genomeName = getGenomeDisplayName();
        String name = GenomeManager.getInstance().generateGenomeKeyFromText(genomeName);
        return name;
    }

    public String getGenomeDisplayName() {
        String name = genomeDisplayNameTextField.getText();
        if (name != null && name.trim().equals(""))
        {
            name = null;
        } else
        {
            name = name.trim();
        }
        return name;
    }

    public String getGenomeArchiveLocation() {
        if (genomeArchiveLocation != null && genomeArchiveLocation.trim().equals(""))
        {
            genomeArchiveLocation = null;
        }
        return genomeArchiveLocation;
    }

    public String getArchiveFileName() {

        if (genomeFilename == null)
        {
            genomeFilename = getGenomeId() + IGVConstants.GENOME_FILE_EXTENSION;
        }
        return genomeFilename;
    }

    /**
     * Show Cytoband file chooser dialog.
     */
    private void showCytobandFileChooser() {

        // File Filters
        FileFilter[] fileFilters = null;

        final FileChooser cytobandFileChooser =
            org.broad.igv.ui.util.UIUtilities.getFileChooser(
            null, null, fileFilters, JFileChooser.CUSTOM_DIALOG);
        cytobandFileChooser.setFileSelectionMode(FileChooser.FILES_ONLY);
        cytobandFileChooser.setDialogTitle("Choose Cytoband File");

        File directory = PreferenceManager.getInstance().getDefineGenomeInputDirectory();
        cytobandFileChooser.setCurrentDirectory(directory);
        cytobandFileChooser.setSelectedFile(null);

        cytobandFileChooser.addPropertyChangeListener(
            new PropertyChangeListener() {

                public void propertyChange(PropertyChangeEvent e) {

                    File oldFile = null;
                    String property = e.getPropertyName();
                    if (JFileChooser.SELECTED_FILE_CHANGED_PROPERTY.equals(property))
                    {
                        oldFile = (File) e.getOldValue();
                        cytobandFileChooser.setPreviousFile(oldFile);
                    }
                }
            });
        cytobandFileChooser.showOpenDialog(this);

        File file = cytobandFileChooser.getSelectedFile();
        if (file == null)
        {
            cytobandFileTextField.setText(null);
        } else
        {

            PreferenceManager.getInstance().setDefineGenomeInputDirectory(file.getParentFile());
            String errorText = (file.getAbsolutePath() +
                " is not a valid Cytoband file - " +
                "Please choose another file!");

            AsciiLineReader reader = null;
            try
            {

                reader = new AsciiLineReader(new FileInputStream(file));
                boolean isValid = CytoBandFileParser.isValid(reader, file.getName());

                if (isValid)
                {
                    cytobandFileTextField.setText(file.getAbsolutePath());
                } else
                {
                    cytobandFileTextField.setText(null);
                    JOptionPane.showMessageDialog(this, errorText);
                }
            } catch (FileNotFoundException e)
            {
                cytobandFileTextField.setText(null);
                JOptionPane.showMessageDialog(this, errorText);
                logger.error(errorText, e);
            } finally
            {

                if (reader != null)
                {
                    reader.close();
                }
            }
        }
    }

    /**
     * Show Gene file chooser dialog.
     */
    private void showGeneFileChooser() {

        // File Filters
        FileFilter[] fileFilters = null;

        final FileChooser refFlatFileChooser =
            org.broad.igv.ui.util.UIUtilities.getFileChooser(
            null, null, fileFilters, JFileChooser.CUSTOM_DIALOG);
        refFlatFileChooser.setFileSelectionMode(FileChooser.FILES_ONLY);
        refFlatFileChooser.setDialogTitle("Choose Gene File");

        File directory = PreferenceManager.getInstance().getDefineGenomeInputDirectory();
        refFlatFileChooser.setCurrentDirectory(directory);
        refFlatFileChooser.setSelectedFile(null);

        refFlatFileChooser.addPropertyChangeListener(
            new PropertyChangeListener() {

                public void propertyChange(PropertyChangeEvent e) {

                    File oldFile = null;
                    String property = e.getPropertyName();
                    if (JFileChooser.SELECTED_FILE_CHANGED_PROPERTY.equals(property))
                    {
                        oldFile = (File) e.getOldValue();
                        refFlatFileChooser.setPreviousFile(oldFile);
                    }
                }
            });
        refFlatFileChooser.showOpenDialog(this);

        File file = refFlatFileChooser.getSelectedFile();
        if (file == null)
        {
            refFlatFileTextField.setText(null);
        } else
        {

            PreferenceManager.getInstance().setDefineGenomeInputDirectory(file.getParentFile());
            String errorText = (file.getAbsolutePath() +
                " is not a valid Gene file - " +
                "Please choose another file!");

            AsciiLineReader reader = null;
            try
            {

                reader = new AsciiLineReader(new FileInputStream(file));
                boolean isValid = GeneManager.isValid(reader, file.getName());

                if (isValid)
                {
                    refFlatFileTextField.setText(file.getAbsolutePath());
                } else
                {
                    refFlatFileTextField.setText(null);
                    JOptionPane.showMessageDialog(this, errorText);
                }
            } catch (FileNotFoundException e)
            {

                refFlatFileTextField.setText(null);
                JOptionPane.showMessageDialog(this, errorText);
                logger.error(errorText, e);
            } finally
            {

                if (reader != null)
                {

                    try
                    {
                        reader.close();
                    } catch (Exception e)
                    {
                        logger.error("Error closing Gene stream", e);
                    }
                }
            }

        }
    }

    /**
     * Show FASTA file chooser dialog.
     */
    private void showFastaFileChooser() {

        CursorToken token = null;
        try
        {

            // File Filters
            FileFilter[] fileFilters = null;

            final FileChooser fastaFileChooser =
                org.broad.igv.ui.util.UIUtilities.getFileChooser(
                null, null, fileFilters, JFileChooser.CUSTOM_DIALOG);
            fastaFileChooser.setFileSelectionMode(FileChooser.FILES_AND_DIRECTORIES);
            fastaFileChooser.setDialogTitle("Choose Sequence File or Directory");

            File directory = PreferenceManager.getInstance().getDefineGenomeInputDirectory();
            fastaFileChooser.setCurrentDirectory(directory);
            fastaFileChooser.setSelectedFile(null);

            fastaFileChooser.addPropertyChangeListener(
                new PropertyChangeListener() {

                    public void propertyChange(PropertyChangeEvent e) {

                        File oldFile = null;
                        String property = e.getPropertyName();
                        if (JFileChooser.SELECTED_FILE_CHANGED_PROPERTY.equals(property))
                        {
                            oldFile = (File) e.getOldValue();
                            fastaFileChooser.setPreviousFile(oldFile);
                        }
                    }
                });
            fastaFileChooser.showOpenDialog(this);

            token = WaitCursorManager.showWaitCursor();

            File file = fastaFileChooser.getSelectedFile();
            if (file == null)
            {
                fastaFileTextField.setText(null);
            } else
            {

                PreferenceManager.getInstance().setDefineGenomeInputDirectory(file.getParentFile());
                boolean isOk = false;
                if (file.isDirectory())
                {

                    HashSet<String> returnedInvalidFastaFiles = new HashSet();
                    isOk = importer.checkFastaFilesInDirectory(
                        file, returnedInvalidFastaFiles);

                    if (!isOk)
                    {
                        StringBuffer buffer = new StringBuffer();
                        buffer.append("The following sequence files were formatted incorrectly\n\n" +
                            "in the selected directory :\n");
                        for (String filePath : returnedInvalidFastaFiles)
                        {
                            buffer.append("\t");
                            buffer.append(filePath);
                            buffer.append("\n");
                        }
                        JOptionPane.showMessageDialog(this, buffer.toString());
                    }
                } else if (file.getName().toLowerCase().endsWith(FASTA_FILE_SET_EXTENSION))
                {

                    // It's a multi-file FASTA archive so validate it
                    HashSet<String> returnedInvalidZipEntries = new HashSet();
                    isOk = importer.checkFastaFileSetFormat(
                        file, returnedInvalidZipEntries);

                    if (!isOk)
                    {
                        StringBuffer buffer = new StringBuffer();
                        buffer.append("The following sequence files were formatted incorrectly\n\n" +
                            "in the supplied file set:\n");
                        for (String entryName : returnedInvalidZipEntries)
                        {
                            buffer.append("\t");
                            buffer.append(entryName);
                            buffer.append("\n");
                        }
                        JOptionPane.showMessageDialog(this, buffer.toString());
                    }
                } else
                { // Single FASTA file
                    isOk = importer.checkFastaFileFormat(file);
                }

                if (isOk)
                {
                    fastaFileTextField.setText(file.getAbsolutePath());
                } else
                {
                    fastaFileTextField.setText(null);
                }
            }
        } finally
        {
            if (token != null)
            {
                WaitCursorManager.removeWaitCursor(token);
            }
        }
    }

    protected File showGenomeArchiveDirectoryChooser() {

        File directory =
            PreferenceManager.getInstance().getLastGenomeImportDirectory();

        directory = PreferenceManager.getInstance().getLastGenomeImportDirectory();
        File newArchiveFile = new File(directory, getArchiveFileName());

        String extensions[] = {IGVConstants.GENOME_FILE_EXTENSION};
        File file = null;
        boolean isOk = false;
        if (!isOk)
        {

            file =
                MacCompatibleFileChooser.showSaveDialog(
                IGVMainFrame.getInstance(), directory, newArchiveFile,
                extensions, "Genome Archive");

            if (file == null)
            {
                //JOptionPane.showMessageDialog(this, "No genome file was selected!");
                isOk = false;
            } else
            {

                genomeFilename = file.getName();
                if (genomeFilename != null)
                {
                    if (!genomeFilename.endsWith(IGVConstants.GENOME_FILE_EXTENSION))
                    {
                        genomeFilename += IGVConstants.GENOME_FILE_EXTENSION;
                    }
                    genomeArchiveLocation = file.getParentFile().getAbsolutePath();
                    isOk = true;
                }
            }
        }

        return file;
    }

    public boolean validateSelection() {

        /*
         * Validity has 3 states: false, true or null;
         * A null Boolean means data was not supplied. This is not the same
         * as being invalid.
         */
        try
        {

            // Check properties
            /*
            Boolean isIdValid = isGenomeIdValid();
            if(isIdValid != null && !isIdValid) {
            JOptionPane.showMessageDialog(this, "Invalid Id!");
            return false;
            }
             */
            Boolean isNameValid = isGenomeDisplayNameValid();
            if (isNameValid != null && !isNameValid)
            {
                genomeDisplayNameTextField.setText(null);
                return false;
            }
            /*
            Boolean isSequenceUrlValid = isSequenceURLValid();
            if(isSequenceUrlValid != null && !isSequenceUrlValid) {
            JOptionPane.showMessageDialog(this, "Invalid Sequence Location!");
            return false;
            }
             */
            // Check files
            /*
            Boolean isRefFlatValid = isRefFlatFileValid();
            if(isRefFlatValid != null && !isRefFlatValid) {
            JOptionPane.showMessageDialog(this, "Invalid refFlat File!");
            return false;
            }
            Boolean isCytobandValid = isCytobandFileValid();
            if(isCytobandValid != null && !isCytobandValid) {
            JOptionPane.showMessageDialog(this, "Invalid Cytoband File!");
            return false;
            }            
            Boolean isFastaValid = isFASTAFileValid();
            if(isFastaValid != null && !isFastaValid) {
            JOptionPane.showMessageDialog(this, "Invalid FASTA File!");
            return false;
            }
             */

            /*
            if(isFastaValid != null && isCtyobandValid != null) {
            JOptionPane.showMessageDialog(this, "Invalid to set both a " +
            "FASTA file and a Cytoband file!");
            return false;
            }
             */

            Boolean isFastaValid = isFASTAFileValid();
            if (isFastaValid == null)
            {
                JOptionPane.showMessageDialog(this, "A Sequence file is required!");
                return false;
            }
        /*
        else if(isRefFlatValid != null) {
        JOptionPane.showMessageDialog(this, "If FASTA file is  "+
        "supplied you cannot also supply a refFlat file!");
        return false;
        }
         */
        /*
        if(isRefFlatValid == null) {
        if(isFastaValid == null) {
        JOptionPane.showMessageDialog(this, "If no refFlat and "+
        "Cytoband file is supplied you must supply a FASTA "+
        "file!");
        return false;
        }
        }
         */
        } catch (Exception e)
        {
            logger.error("Error during Genome Builder validation!", e);
            return false;
        }
        return true;
    }

    private Boolean isFASTAFileValid() {

        String file = fastaFileTextField.getText();
        if (file == null)
        {
            return null;
        }
        if (file.trim().equals(""))
        {
            return null;
        }
        return true;
    }

    private Boolean isGenomeDisplayNameValid() {

        String displayName = getGenomeDisplayName();
        if (displayName == null)
        {
            JOptionPane.showMessageDialog(this, "No display name supplied!");
            return false;
        }

        displayName = displayName.trim();

        if (displayName.equals(""))
        {
            JOptionPane.showMessageDialog(this, "No display name supplied!");
            return false;
        }

        Collection<String> inUseDisplayNames =
            IGVMainFrame.getInstance().getGenomeDisplayNames();

        if (inUseDisplayNames.contains(displayName))
        {
            JOptionPane.showMessageDialog(this,
                "The selected genome name is already in use - please select another!");
            return false;
        }
        return true;
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jLabel1 = new javax.swing.JLabel();
        jLabel2 = new javax.swing.JLabel();
        jPanel1 = new javax.swing.JPanel();
        cytobandFileTextField = new javax.swing.JTextField();
        cytobandFileButton = new javax.swing.JButton();
        refFlatFileLabel = new javax.swing.JLabel();
        refFlatFileButton = new javax.swing.JButton();
        genomeDisplayNameTextField = new javax.swing.JTextField();
        fastaFileLabel = new javax.swing.JLabel();
        fastaFileButton = new javax.swing.JButton();
        cytobandFileLabel = new javax.swing.JLabel();
        fastaFileTextField = new javax.swing.JTextField();
        refFlatFileTextField = new javax.swing.JTextField();
        genomeDisplayNameLabel = new javax.swing.JLabel();

        setToolTipText("Used to completely define all the items associated with a genome");
        setFont(new java.awt.Font("Tahoma", 2, 12)); // NOI18N
        setMaximumSize(new java.awt.Dimension(700, 500));
        setMinimumSize(new java.awt.Dimension(400, 300));
        setPreferredSize(new java.awt.Dimension(600, 400));

        jLabel1.setFont(new java.awt.Font("Lucida Sans Unicode", 2, 12)); // NOI18N
        jLabel1.setText("<html>The sequence file (required) can be a FASTA file, a directory of FASTA files, or a zip of FASTA files. Optionally, specify a cytoband file to display the chromosome ideogram and an annotation file to display the gene track. See the documentation for descriptions of supported annotation formats.</html>");
        jLabel1.setVerticalAlignment(javax.swing.SwingConstants.TOP);

        jLabel2.setText("* Required");

        cytobandFileTextField.setToolTipText("A cytoband data file");
        cytobandFileTextField.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cytobandFileTextFieldActionPerformed(evt);
            }
        });

        cytobandFileButton.setLabel("...");
        cytobandFileButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cytobandFileButtonActionPerformed(evt);
            }
        });
        cytobandFileButton.addHierarchyBoundsListener(new java.awt.event.HierarchyBoundsListener() {
            public void ancestorMoved(java.awt.event.HierarchyEvent evt) {
                cytobandFileButtonAncestorMoved(evt);
            }
            public void ancestorResized(java.awt.event.HierarchyEvent evt) {
            }
        });

        refFlatFileLabel.setText("Gene File");

        refFlatFileButton.setLabel("...");
        refFlatFileButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                refFlatFileButtonActionPerformed(evt);
            }
        });

        genomeDisplayNameTextField.setToolTipText("The user-readable name of the genome");
        genomeDisplayNameTextField.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                genomeDisplayNameTextFieldActionPerformed(evt);
            }
        });

        fastaFileLabel.setText("Sequence File *");

        fastaFileButton.setLabel("...");
        fastaFileButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                fastaFileButtonActionPerformed(evt);
            }
        });

        cytobandFileLabel.setText("Cytoband File");

        fastaFileTextField.setToolTipText("A FASTA data file");

        refFlatFileTextField.setToolTipText("A refFlat gene file");
        refFlatFileTextField.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                refFlatFileTextFieldActionPerformed(evt);
            }
        });

        genomeDisplayNameLabel.setText("Name");

        org.jdesktop.layout.GroupLayout jPanel1Layout = new org.jdesktop.layout.GroupLayout(jPanel1);
        jPanel1.setLayout(jPanel1Layout);
        jPanel1Layout.setHorizontalGroup(
            jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(jPanel1Layout.createSequentialGroup()
                        .add(1, 1, 1)
                        .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.TRAILING)
                            .add(fastaFileLabel)
                            .add(genomeDisplayNameLabel)))
                    .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.TRAILING)
                        .add(cytobandFileLabel)
                        .add(refFlatFileLabel)))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(refFlatFileTextField, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 295, Short.MAX_VALUE)
                    .add(cytobandFileTextField, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 295, Short.MAX_VALUE)
                    .add(fastaFileTextField, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 295, Short.MAX_VALUE)
                    .add(genomeDisplayNameTextField, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 295, Short.MAX_VALUE))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(cytobandFileButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 35, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(refFlatFileButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 35, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(fastaFileButton, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 35, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                .addContainerGap())
        );
        jPanel1Layout.setVerticalGroup(
            jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jPanel1Layout.createSequentialGroup()
                .addContainerGap()
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(genomeDisplayNameLabel)
                    .add(genomeDisplayNameTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(fastaFileLabel)
                    .add(fastaFileTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(fastaFileButton))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(cytobandFileLabel, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 14, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(cytobandFileTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(cytobandFileButton))
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.RELATED)
                .add(jPanel1Layout.createParallelGroup(org.jdesktop.layout.GroupLayout.BASELINE)
                    .add(refFlatFileLabel)
                    .add(refFlatFileTextField, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                    .add(refFlatFileButton))
                .add(29, 29, 29))
        );

        org.jdesktop.layout.GroupLayout layout = new org.jdesktop.layout.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(layout.createSequentialGroup()
                .add(layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
                    .add(layout.createSequentialGroup()
                        .add(51, 51, 51)
                        .add(jLabel2, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 82, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                    .add(layout.createSequentialGroup()
                        .add(25, 25, 25)
                        .add(jLabel1, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 371, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE))
                    .add(layout.createSequentialGroup()
                        .addContainerGap()
                        .add(jPanel1, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)))
                .addContainerGap(51, Short.MAX_VALUE))
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(org.jdesktop.layout.GroupLayout.TRAILING, layout.createSequentialGroup()
                .add(38, 38, 38)
                .add(jPanel1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                .addPreferredGap(org.jdesktop.layout.LayoutStyle.UNRELATED)
                .add(jLabel2)
                .add(18, 18, 18)
                .add(jLabel1, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE, 123, org.jdesktop.layout.GroupLayout.PREFERRED_SIZE)
                .addContainerGap())
        );
    }// </editor-fold>//GEN-END:initComponents

private void cytobandFileTextFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cytobandFileTextFieldActionPerformed
// TODO add your handling code here:
}//GEN-LAST:event_cytobandFileTextFieldActionPerformed

private void cytobandFileButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cytobandFileButtonActionPerformed
    showCytobandFileChooser();
}//GEN-LAST:event_cytobandFileButtonActionPerformed

private void cytobandFileButtonAncestorMoved(java.awt.event.HierarchyEvent evt) {//GEN-FIRST:event_cytobandFileButtonAncestorMoved
// TODO add your handling code here:
}//GEN-LAST:event_cytobandFileButtonAncestorMoved

private void refFlatFileButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_refFlatFileButtonActionPerformed
    showGeneFileChooser();
}//GEN-LAST:event_refFlatFileButtonActionPerformed

private void fastaFileButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_fastaFileButtonActionPerformed
    showFastaFileChooser();
}//GEN-LAST:event_fastaFileButtonActionPerformed

private void refFlatFileTextFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_refFlatFileTextFieldActionPerformed
// TODO add your handling code here:
}//GEN-LAST:event_refFlatFileTextFieldActionPerformed

private void genomeDisplayNameTextFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_genomeDisplayNameTextFieldActionPerformed
// TODO add your handling code here:
}//GEN-LAST:event_genomeDisplayNameTextFieldActionPerformed


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton cytobandFileButton;
    private javax.swing.JLabel cytobandFileLabel;
    private javax.swing.JTextField cytobandFileTextField;
    private javax.swing.JButton fastaFileButton;
    private javax.swing.JLabel fastaFileLabel;
    private javax.swing.JTextField fastaFileTextField;
    private javax.swing.JLabel genomeDisplayNameLabel;
    private javax.swing.JTextField genomeDisplayNameTextField;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JButton refFlatFileButton;
    private javax.swing.JLabel refFlatFileLabel;
    private javax.swing.JTextField refFlatFileTextField;
    // End of variables declaration//GEN-END:variables

}
