/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.track;

import com.jidesoft.swing.JidePopupMenu;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import org.apache.commons.math.stat.StatUtils;
import org.apache.log4j.Logger;
import org.broad.igv.PreferenceManager;
import org.broad.igv.renderer.BarChartRenderer;
import org.broad.igv.renderer.DataRange;
import org.broad.igv.renderer.HeatmapRenderer;
import org.broad.igv.renderer.LineplotRenderer;
import org.broad.igv.renderer.RNAiBarChartRenderer;
import org.broad.igv.renderer.ScatterplotRenderer;
import org.broad.igv.ui.DataRangeDialog;
import org.broad.igv.ui.GuiUtilities;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.util.UIUtilities;

/**
 *
 * @author jrobinso
 */
public class TrackMenuUtils {

    static Logger log = Logger.getLogger(TrackMenuUtils.class);

    final static String LEADING_HEADING_SPACER = "  ";

    /**
     * Return a popup menu with items applicable to the collection of tracks.
     * @param tracks
     * @return
     */
    public static JPopupMenu getPopupMenu(Collection<Track> tracks, int x, int y, String title) {

        if (log.isDebugEnabled()) {
            log.debug("enter getPopupMenu");
        }

        boolean hasDataTracks = false;
        boolean hasFeatureTracks = false;
        for (Track track : tracks) {

            // This is a messy test. A "mutation track" is implemented as a
            // FeatureTrack, but the "DataTrack" menu applies.  Refactor later
            // to fix this,  for now live with the messy test.
            if ((track instanceof DataTrack) || (track instanceof HDFDataTrack) || ((track instanceof FeatureTrack) && ((FeatureTrack) track).isMutationTrack())) {
                hasDataTracks = true;
            } else if (track instanceof FeatureTrack) {
                hasFeatureTracks = true;
            }
            if (hasDataTracks && hasFeatureTracks) {
                return getSharedPopupMenu();
            }
        }

        if (hasDataTracks) {
            return getDataPopupMenu(tracks, x, y, title);
        } else {
            return getFeaturePopupMenu(tracks, x, y, title);
        }

    }

    /**
     * Return popup menu with items applicable to data tracks
     * @return
     */
    public static JPopupMenu getDataPopupMenu(final Collection<Track> tracks, int x, int y, String title) {

        if (log.isDebugEnabled()) {
            log.debug("enter getDataPopupMenu");
        }
        
        JidePopupMenu dataPopupMenu = new JidePopupMenu();

        final String[] labels = {"Heatmap", "Bar Chart", "Scatterplot", "Line Plot"};
        final Class[] renderers = {HeatmapRenderer.class, BarChartRenderer.class,
            ScatterplotRenderer.class, LineplotRenderer.class
        };

        JLabel popupTitle = new JLabel(LEADING_HEADING_SPACER + title,
                JLabel.CENTER);

        Font newFont = dataPopupMenu.getFont().deriveFont(Font.BOLD, 12);
        popupTitle.setFont(newFont);
        if (popupTitle != null) {
            dataPopupMenu.add(popupTitle);
            dataPopupMenu.addSeparator();
        }

        JLabel rendererHeading = new JLabel(
                LEADING_HEADING_SPACER + "Type of Graph", JLabel.LEFT);
        rendererHeading.setFont(newFont);

        dataPopupMenu.add(rendererHeading);

        // Get existing selections
        Set<Class> currentRenderers = new HashSet<Class>();
        for (Track track : tracks) {
            if (track.getRenderer() != null) {
                currentRenderers.add(track.getRenderer().getClass());
            }
        }

        // Create and add a menu item
        for (int i = 0; i < labels.length; i++) {
            JCheckBoxMenuItem item = new JCheckBoxMenuItem(labels[i]);
            final Class rendererClass = renderers[i];
            if (currentRenderers.contains(rendererClass)) {
                item.setSelected(true);
            }
            item.addActionListener(new ActionListener() {

                public void actionPerformed(ActionEvent e) {
                    changeRenderer(rendererClass);
                }
            });
            dataPopupMenu.add(item);
        }
        dataPopupMenu.addSeparator();


        // Get union of all valid window functions for selected tracks
        Set<WindowFunction> avaibleWindowFunctions = new HashSet();
        for (Track track : tracks) {
            avaibleWindowFunctions.addAll(track.getAvailableWindowFunctions());
        }
        final WindowFunction[] orderedWindowFunctions = {
            // WindowFunction.min,
            WindowFunction.percentile10, WindowFunction.median, WindowFunction.mean,
            WindowFunction.percentile90, WindowFunction.max
        };


        // dataPopupMenu.addSeparator();
        // Collection all window functions for selected tracks
        Set<WindowFunction> currentWindowFunctions = new HashSet<WindowFunction>();
        for (Track track : tracks) {
            if (track.getWindowFunction() != null) {
                currentWindowFunctions.add(track.getWindowFunction());
            }
        }

        if (!avaibleWindowFunctions.isEmpty() || !currentWindowFunctions.isEmpty()) {
            JLabel statisticsHeading = new JLabel(
                    LEADING_HEADING_SPACER + "Windowing Function",
                    JLabel.LEFT);
            statisticsHeading.setFont(newFont);

            dataPopupMenu.add(statisticsHeading);

            for (final WindowFunction wf : orderedWindowFunctions) {
                JCheckBoxMenuItem item = new JCheckBoxMenuItem(
                        wf.getDisplayName());
                if (avaibleWindowFunctions.contains(wf) || currentWindowFunctions.contains(
                        wf)) {
                    if (currentWindowFunctions.contains(wf)) {
                        item.setSelected(true);
                    }
                    item.addActionListener(new ActionListener() {

                        public void actionPerformed(ActionEvent e) {
                            changeStatType(wf.toString());
                        }
                    });
                    dataPopupMenu.add(item);
                }
            }
            dataPopupMenu.addSeparator();
        }


        JLabel trackSettingsHeading = new JLabel(
                LEADING_HEADING_SPACER + "Track Settings",
                JLabel.LEFT);
        trackSettingsHeading.setFont(newFont);

        dataPopupMenu.add(trackSettingsHeading);

        addTrackRenameItem(dataPopupMenu);

        JMenuItem maxValItem = new JMenuItem("Set Data Range");

        maxValItem.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                final Collection<Track> selectedTracks = tracks;
                if (selectedTracks.size() > 0) {
                    DataRange prevAxisDefinition = selectedTracks.iterator().next().getAxisDefinition();
                    DataRangeDialog dlg = new DataRangeDialog(
                            IGVMainFrame.getInstance(),
                            prevAxisDefinition);
                    dlg.setVisible(true);
                    if (!dlg.isCanceled()) {
                        DataRange axisDefinition = new DataRange(dlg.getMin(),
                                dlg.getBase(),
                                dlg.getMax());

                        // dlg.isFlipAxis());
                        for (Track track : selectedTracks) {
                            track.setDataRange(axisDefinition);
                        }
                        IGVMainFrame.getInstance().clearImageCacheWithNoRepaint();
                        IGVMainFrame.getInstance().repaint();
                    }

                }

            }
        });
        dataPopupMenu.add(maxValItem);

        addTrackSettingsMenuItems(dataPopupMenu);

        return dataPopupMenu;
    }

    /**
     * Return popup menu with items applicable to feature tracks
     * @return
     */
    public static JPopupMenu getFeaturePopupMenu(Collection<Track> tracks, int x, int y, String title) {

        JPopupMenu featurePopupMenu = new JidePopupMenu();

        JLabel popupTitle = new JLabel(LEADING_HEADING_SPACER + title,
                JLabel.CENTER);

        Font newFont = featurePopupMenu.getFont().deriveFont(Font.BOLD, 12);
        popupTitle.setFont(newFont);
        if (popupTitle != null) {
            featurePopupMenu.add(popupTitle);
            featurePopupMenu.addSeparator();
        }


        JLabel trackSettingsHeading = new JLabel(
                LEADING_HEADING_SPACER + "Track Settings",
                JLabel.LEFT);
        trackSettingsHeading.setFont(newFont);

        featurePopupMenu.add(trackSettingsHeading);

        addTrackRenameItem(featurePopupMenu);

        addExpandCollapseItem(featurePopupMenu);

        addTrackSettingsMenuItems(featurePopupMenu, true);


        return featurePopupMenu;
    }

    /**
     * Popup menu with items applicable to both feature and data tracks
     * @return
     */
    public static JPopupMenu getSharedPopupMenu() {

        JPopupMenu sharedPopupMenu = new JidePopupMenu();

        JLabel trackSettingsHeading = new JLabel(
                LEADING_HEADING_SPACER + "Track Settings",
                JLabel.LEFT);
        sharedPopupMenu.add(trackSettingsHeading);

        addTrackSettingsMenuItems(sharedPopupMenu);


        return sharedPopupMenu;
    }

    public static void addTrackSettingsMenuItems(JPopupMenu menu) {
        addTrackSettingsMenuItems(menu, false);
    }

    public static void addTrackSettingsMenuItems(JPopupMenu menu,
            boolean featureTracksOnly) {




        // Change track height by attribute
        JMenuItem item = new JMenuItem("Change Track Height");
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {
                        changeTrackHeight();
                    }
                });
            }
        });
        menu.add(item);

        // Change track color by attribute
        String colorLabel = featureTracksOnly
                ? "Change Track Color" : "Change Track Color (Positive Values)";
        item = new JMenuItem(colorLabel);
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {
                        changeTrackColor();
                    }
                });
            }
        });
        menu.add(item);

        if (!featureTracksOnly) {

            // Change track color by attribute
            item = new JMenuItem("Change Track Color (Negative Values)");
            item.setToolTipText(
                    "Change the alternate track color.  This color is used when graphing negative values");
            item.addActionListener(new ActionListener() {

                public void actionPerformed(ActionEvent e) {
                    GuiUtilities.invokeOnEventThread(new Runnable() {

                        public void run() {
                            changeAltTrackColor();
                        }
                    });
                }
            });
            menu.add(item);
        }

        // Remove tracks by attribute
        item = new JMenuItem("Remove Tracks");
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {

                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {
                        removeTracks();
                    }
                });
            }
        });
        menu.add(item);
    }

    public static void changeStatType(String statType) {
        for (Track track : getSelectedTracks()) {
            track.setStatType(WindowFunction.valueOf(statType));
        }
        refresh();
    }

    public static void addTrackRenameItem(JPopupMenu menu) {
        // Change track height by attribute
        JMenuItem item = new JMenuItem("Rename Track");
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {
                        renameTrack();
                    }
                });
            }
        });
        menu.add(item);
        if (getSelectedTracks().size() > 1) {
            item.setEnabled(false);
        }

    }

    public static void addExpandCollapseItem(JPopupMenu featurePopupMenu) {

        Collection<Track> tracks = getSelectedTracks();

        // If any tracks are expanded show the "Collapse" option, otherwise expand
        boolean expanded = false;
        for (Track track : tracks) {
            if (track.isExpanded()) {
                expanded = true;
                break;
            }
        }

        // Show Multi-Level Features
        String text = expanded ? "Collapse Track" : "Expand Track";
        if (tracks.size() > 1) {
            text += "s";
        }
        JMenuItem item = new JMenuItem(text);
        item.addActionListener(new ActionListener() {

            public void actionPerformed(final ActionEvent e) {

                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {
                        doToggleSelectedMultiTrackFeatures();
                    }
                });
            }
        });
        featurePopupMenu.add(item);
    }

    /**
     * Toggle selected multi-level tracks
     */
    public static void doToggleSelectedMultiTrackFeatures() {

        for (Track track : getSelectedTracks()) {
            boolean isMultiLevel = track.isExpanded();
            track.setExpanded(!isMultiLevel);
        }

        IGVMainFrame.getInstance().clearImageCacheWithNoRepaint();
        IGVMainFrame.getInstance().doResizeTrackPanels();
    //IGVMainFrame.getInstance().refreshFeatureTrackView();
    }

    public static void changeRenderer(Class rendererClass) {
        for (Track track : getSelectedTracks()) {

            // TODO -- a temporary hack to facilitate RNAi development
            if (track.getTrackType() == TrackType.RNAI) {
                if (rendererClass == BarChartRenderer.class) {
                    rendererClass = RNAiBarChartRenderer.class;
                }

            }
            track.setRendererClass(rendererClass);
        }
        refresh();
    }

    public static void addTrackRenameItem(JPopupMenu menu, Collection<Track> selectedTracks) {
        // Change track height by attribute
        JMenuItem item = new JMenuItem("Rename Track");
        item.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                GuiUtilities.invokeOnEventThread(new Runnable() {

                    public void run() {
                        renameTrack();
                    }
                });
            }
        });
        menu.add(item);
        if (selectedTracks.size() > 1) {
            item.setEnabled(false);
        }

    }

    public static void renameTrack() {

        Collection<Track> selectedTracks = getSelectedTracks();
        if (selectedTracks.isEmpty()) {
            return;
        }

        Track t = getSelectedTracks().iterator().next();


        String newName = JOptionPane.showInputDialog(
                IGVMainFrame.getInstance(), "Enter new name: ", t.getDisplayName());

        if (newName == null || newName.trim() == "") {
            return;
        }

        t.setDisplayName(newName);

        refresh();
    }

    public static void changeTrackHeight() {


        HashSet<Track> selectedTracks = new HashSet<Track>(getSelectedTracks());

        if (selectedTracks.isEmpty()) {
            return;
        }

        int intHeight = getRepresentativeTrackHeight(selectedTracks);

        while (true) {

            String height = JOptionPane.showInputDialog(
                    IGVMainFrame.getInstance(), "New Track: ",
                    String.valueOf(intHeight));

            if ((height == null) || height.trim().equals("")) {
                return;
            }

            try {
                intHeight = Integer.parseInt(height);
                break;
            } catch (NumberFormatException numberFormatException) {
                JOptionPane.showMessageDialog(IGVMainFrame.getInstance(),
                        "Track height must be an integer number.");
            }
        }

        for (Track track : selectedTracks) {
            track.setHeight((int) intHeight);
        }

        refresh();
    }

    public static void removeTracks() {

        HashSet<Track> selectedTracks = new HashSet<Track>(getSelectedTracks());
        if (selectedTracks.isEmpty()) {
            return;
        }

        StringBuffer buffer = new StringBuffer();
        for (Track track : selectedTracks) {
            buffer.append("\n\t");
            buffer.append(track.getDisplayName());
        }
        String deleteItems = buffer.toString();

        JTextArea textArea = new JTextArea();
        textArea.setEditable(false);
        JScrollPane scrollPane = new JScrollPane(textArea);
        textArea.setText(deleteItems);

        JOptionPane optionPane = new JOptionPane(scrollPane,
                JOptionPane.PLAIN_MESSAGE,
                JOptionPane.YES_NO_OPTION);
        optionPane.setPreferredSize(new Dimension(550, 500));
        JDialog dialog = optionPane.createDialog(IGVMainFrame.getInstance(),
                "Remove The Following Tracks");
        dialog.setVisible(true);

        Object choice = optionPane.getValue();
        if ((choice == null) || (JOptionPane.YES_OPTION != ((Integer) choice).intValue())) {
            return;
        }

        TrackManager.getInstance().removeTracks(selectedTracks);

        IGVMainFrame.getInstance().doResizeTrackPanels();

        refresh();
    }

    public static void changeTrackColor() {

        HashSet<Track> selectedTracks = new HashSet<Track>(getSelectedTracks());

        if (selectedTracks.isEmpty()) {
            return;
        }

        Color currentSelection = selectedTracks.iterator().next().getColor();

        Color color = UIUtilities.showColorChooserDialog(
                "Select Track Color (Positive Values)",
                currentSelection);

        if (color != null) {
            for (Track track : selectedTracks) {
                track.setColor(color);
            }

            refresh();
        }

    }

    public static void changeAltTrackColor() {

        HashSet<Track> selectedTracks = new HashSet<Track>(getSelectedTracks());

        if (selectedTracks.isEmpty()) {
            return;
        }

        Color currentSelection = selectedTracks.iterator().next().getColor();

        Color color = UIUtilities.showColorChooserDialog(
                "Select Track Color (Negative Values)",
                currentSelection);

        if (color == null) {
            return;
        }

        for (Track track : selectedTracks) {

            track.setAltColor(color);
        }

        refresh();

    }

    static public Collection<Track> getSelectedTracks() {
        return TrackManager.getInstance().getSelectedTracks();
    }

    /**
     * Return a representative track height to use as the default.  For now
     * using the median track height.
     * @return
     */
    public static int getRepresentativeTrackHeight(Collection<Track> tracks) {

        double[] heights = new double[tracks.size()];
        int i = 0;
        for (Track track : tracks) {
            heights[i] = track.getHeight();
            i++;
        }
        int medianTrackHeight = (int) Math.round(StatUtils.percentile(heights,
                50));
        if (medianTrackHeight > 0) {
            return medianTrackHeight;
        }

        return PreferenceManager.getInstance().getDefaultTrackHeight();

    }

    public static void refresh() {
        GuiUtilities.invokeOnEventThread(new Runnable() {

            public void run() {

                IGVMainFrame.getInstance().showLoadedTrackCount();
                IGVMainFrame.getInstance().clearImageCacheWithNoRepaint();
                IGVMainFrame.getInstance().getContentPane().repaint();
            }
        });
    }
}

