/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.track;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;

import org.broad.igv.IGVConstants;


import org.broad.igv.PreferenceManager;
import org.broad.igv.data.*;
import org.broad.igv.data.rnai.RNAIDataSource;
import org.broad.igv.data.rnai.RNAIGeneScoreParser;
import org.broad.igv.data.rnai.RNAIHairpinParser;
import org.broad.igv.data.rnai.RNAIGCTDatasetParser;
import org.broad.igv.feature.AbstractFeatureParser;
import org.broad.igv.feature.Feature;
import org.broad.igv.feature.FeatureParser;
import org.broad.igv.feature.GeneManager;
import org.broad.igv.feature.Genome;
import org.broad.igv.feature.GisticFileParser;
import org.broad.igv.feature.MutationParser;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.maf.MAFTrack;
import org.broad.igv.renderer.DataRange;
import org.broad.igv.renderer.GeneTrackRenderer;
import org.broad.igv.renderer.HeatmapRenderer;
import org.broad.igv.renderer.MutationRenderer;
import org.broad.igv.synteny.Mapping;
import org.broad.igv.synteny.MappingParser;
import org.broad.igv.ui.IGVModel;
import org.broad.igv.ui.MessageCollection;
import org.broad.igv.ui.RegionOfInterest;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;

import java.io.File;
import java.io.FileNotFoundException;



import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.swing.JOptionPane;
import org.broad.igv.ui.util.UIUtilities;
import org.broad.igv.maf.conservation.OmegaDataSource;
import org.broad.igv.maf.conservation.OmegaTrack;
import org.broad.igv.sam.AlignmentTrack;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.panel.DragEventManager;
import org.broad.igv.ui.panel.DragListener;

/**
 *  Misc globally visible parameters.  These need a home.
 *
 */
/**
 *  Placeholder for miscelleneous stuff that hasn't found a home yet.
 */
public class TrackManager {

    public static final int GENE_TRACK_HEIGHT = 35;
    public static final int GISTIC_TRACK_HEIGHT = 50;
    private static Logger log = Logger.getLogger(TrackManager.class);
    private static TrackManager theInstance = null;
    static List emptyList = new ArrayList();
    final public static String GENE_TRACK_NAME = "Gene";
    private String groupByAttribute = null;
    /**
     * The gene track.  Always present, rendered in the FeaturePanel
     */
    private Track geneTrack;
    /** 
     * A map of data panel name -> track group 
     */
    Map<String, TrackGroup> panelTrackGroups;
    /** 
     * A map of data panel name -> [map of group name -> track group] 
     */
    Map<String, Map<String, TrackGroup>> trackGroups;
    // Hack for overlay experiment
    public static String OVERLAY_DATASET = "";
    Map<String, List<Track>> overlayTracksMap = new HashMap();
    /**
     * Map of id -> track. 
     * 
     * Note: assumes ids are unique, but there is nothing that enforces this.
     */
    Map<String, Track> trackDictionary = new Hashtable();

    static public enum MatchTrack {

        EXACT, CONTAINS;
    }
    private GeneManager geneData;

    /**
     * Object representing the complete set of genes for the selected genomeId
     *
     * @return
     */
    public static TrackManager getInstance() {
        if (theInstance == null) {
            theInstance = new TrackManager();
        }

        return theInstance;
    }

    private TrackManager() {

        // Create default groups for each  panel
        panelTrackGroups = new Hashtable();
        trackGroups = new Hashtable();
        initGroupsForPanel(IGVMainFrame.DATA_PANEL_NAME);
        initGroupsForPanel(IGVMainFrame.FEATURE_PANEL_NAME);
    }

    public void initGroupsForPanel(String panelName) {
        TrackGroup allDataTracks = new TrackGroup();
        allDataTracks.setDrawBorder(false);
        panelTrackGroups.put(panelName, allDataTracks);

        trackGroups.put(panelName, new LinkedHashMap());
    }

    public Track getTrackById(String id) {
        return trackDictionary.get(id);
    }

    public List<Track> getTracksForPanel(String panelName) {
        return panelTrackGroups.containsKey(panelName) ? panelTrackGroups.get(panelName).getTracks() : emptyList;
    }

    public Collection<String> getPanelNames() {
        return panelTrackGroups.keySet();
    }

    /**
     * Method description
     *
     *
     * @param newGeneTrack
     */
    public void setGeneTrack(Track newGeneTrack) {

        boolean foundGeneTrack = false;
        for (TrackGroup g : panelTrackGroups.values()) {
            if (g.contains(geneTrack)) {
                int geneTrackIndex = g.indexOf(geneTrack);
                g.remove(geneTrack);
                geneTrackIndex = Math.min(g.size(), geneTrackIndex);
                g.add(geneTrackIndex, newGeneTrack);
                foundGeneTrack = true;
                break;
            }
        }

        if (!foundGeneTrack) {
            if (PreferenceManager.getInstance().isShowSingleTrackPane()) {
                panelTrackGroups.get(IGVMainFrame.DATA_PANEL_NAME).add(newGeneTrack);
            } else {
                panelTrackGroups.get(IGVMainFrame.FEATURE_PANEL_NAME).add(newGeneTrack);
            }
        }

        // Keep a reference to this track so it can be removed
        geneTrack = newGeneTrack;

        groupTracksByAttribute();
    }

    /**
     * Return the track groups.
     *
     * @return
     */
    public Collection<TrackGroup> getTrackGroups(String panelName) {
        Collection<TrackGroup> group = trackGroups.get(panelName).values();
        return group;
    }

    /**
     * Method description
     *
     *
     * @param attributeName
     */
    public void setGroupByAttribute(String attributeName) {
        this.groupByAttribute = attributeName;
        groupTracksByAttribute();
    }

    /**
     * Method description
     *
     */
    public void groupTracksByAttribute() {

        for (Map.Entry<String, Map<String, TrackGroup>> entry : trackGroups.entrySet()) {
            String panelName = entry.getKey();
            Map<String, TrackGroup> groupsMap = entry.getValue();
            groupsMap.clear();

            if (getGroupByAttribute() == null) {
                groupsMap.put("", panelTrackGroups.get(panelName));
            } else {
                for (Track track : panelTrackGroups.get(panelName).getTracks()) {
                    String attributeValue = track.getAttributeValue(
                            getGroupByAttribute());

                    if (attributeValue == null) {
                        attributeValue = "";
                    }

                    TrackGroup group = groupsMap.get(attributeValue);

                    if (group == null) {
                        group = new TrackGroup(getGroupByAttribute(), attributeValue);
                        groupsMap.put(attributeValue, group);
                    }

                    group.add(track);
                }
            }
        }
    }

    /**
     *@deprecated -- keep until Session is refactored to break dependence on panels
     *
     *
     * @return
     */
    public List<Track> getDataTracks() {
        return panelTrackGroups.get(IGVMainFrame.DATA_PANEL_NAME).getTracks();
    }

    /**
     * Method description
     *
     *
     * @param ignoreGeneTrack
     */
    public void resetApplication(boolean ignoreGeneTrack) {

        // Remove Tracks
        clearTracks(ignoreGeneTrack);

        // Remove loaded Attributes
        AttributeManager.getInstance().clearAllAttributes();
    }

    /**
     * Method description
     *
     *
     * @param ignoreGeneTrack
     */
    public void clearTracks(boolean ignoreGeneTrack) {

        setGroupByAttribute(null);
        for (TrackGroup tg : panelTrackGroups.values()) {
            tg.clear();
        }
        setGeneTrack(geneTrack);

    }

    /**
     * A (hopefully) temporary solution to force SAM track reloads,  until we have a better
     * dependency scheme
     */
    public void reloadSAMTracks() {
        for (Track t : getAllTracks(false)) {
            if (t instanceof AlignmentTrack) {
                ((AlignmentTrack) t).reloadData();
            }
        }
    }

    public boolean isSinglePanel() {
        return PreferenceManager.getInstance().isShowSingleTrackPane();
    }

    public boolean loadResources(Collection<ResourceLocator> locators) {
        boolean tracksWereLoaded = false;
        try {
            log.info("Loading tracks " + locators.size() + " tracks.");
            // Get Data Track Files
            MessageCollection messages = loadResources_(locators);
            tracksWereLoaded = true;

            trackDictionary = new Hashtable();
            for (Track t : this.getAllTracks(true)) {
                if (t != null) {
                    trackDictionary.put(t.getId(), t);
                }
            }

            // Need this because TrackManager loads some attributes
            // internally. So we need to pick them up.
            IGVMainFrame.getInstance().updateAttributePanel();
            // Show and log error messages
            if (!messages.isEmpty()) {
                UIUtilities.showAndLogErrorMessage(IGVMainFrame.getInstance(),
                        messages.getFormattedMessage(),
                        log);
            }
        } catch (FileNotFoundException ex) {
            String message = "Some files could not be loaded!";
            UIUtilities.showAndLogErrorMessage(IGVMainFrame.getInstance(), message, log, ex);
        }
        return tracksWereLoaded;
    }

    private boolean isSamFile(String path) {
        return path.endsWith("sam") || path.endsWith("sam.group") ||
                path.endsWith("bam");
    }

    /**
     *
     */
    private MessageCollection loadResources_(
            Collection<ResourceLocator> resourceLocators)
            throws FileNotFoundException {

        MessageCollection messages = new MessageCollection();

        // StringBuffer messages = new StringBuffer();
        // boolean foundInvalidTracks = false;

        try {


            // Sort the resource list so that SAM/BAM files are loaded last.
            List<ResourceLocator> sortedLocators = new ArrayList(resourceLocators);
            Collections.sort(sortedLocators, new Comparator<ResourceLocator>() {

                public int compare(ResourceLocator arg0, ResourceLocator arg1) {
                    if (isSamFile(arg0.getPath()) && !isSamFile(arg1.getPath())) {
                        return 1;
                    } else if (isSamFile(arg1.getPath()) && !isSamFile(arg0.getPath())) {
                        return -1;
                    } else {
                        return 0;
                    }
                }
            });

            LinkedHashSet<Attribute> attributeList = new LinkedHashSet();
            for (ResourceLocator locator : sortedLocators) {

                if (locator.isLocal()) {
                    File trackSetFile = new File(locator.getPath());

                    if (!trackSetFile.exists()) {
                        messages.append("File not found: " + locator.getPath());
                        continue;
                    }
                }
                TrackGroup group = getTrackGroupFor(locator);
                load(locator, group, attributeList, messages);
            }
            AttributeManager.getInstance().addAttributes(attributeList);
            groupTracksByAttribute();
            resetOverlayTracks();
            for (Track t : this.getAllTracks(true)) {
                if (t != null) {
                    t.updateProperties();
                }
            }
        } catch (Exception ex) {
            messages.append("<html> Error loading data: " + ex.getMessage());
            log.error("Error loading tracks", ex);
        } finally {

            AttributeManager.getInstance().firePropertyChange(
                    AttributeManager.getInstance(),
                    AttributeManager.ATTRIBUTES_LOADED_PROPERTY, null, null);

        }
        return messages;
    }

    /**
     * Load the data file into the specified group.
     *
     * @param file
     * @param panelId
     * @return
     */
    public MessageCollection load(File file, String panelId) {
        MessageCollection messages = new MessageCollection();
        LinkedHashSet<Attribute> attributeList = new LinkedHashSet();

        TrackGroup group = panelTrackGroups.get(panelId);
        ResourceLocator locator = new ResourceLocator(file.getAbsolutePath());
        load(locator, group, attributeList, messages);

        AttributeManager.getInstance().addAttributes(attributeList);
        return messages;

    }

    /**
     * Reset the overlay tracks collection.  Currently the only overlayable track
     * type is Mutation.  This method finds all mutation tracks and builds a map
     * of key -> mutatinon track,  where the key is the specified attribute value
     * for linking tracks for overlay.
     *
     * The method also resets all tracks overlay properties from the user
     * user preference value.  That should probably be done in another method.
     */
    public void resetOverlayTracks() {
        overlayTracksMap.clear();

        if (isDoOverlays()) {
            String overlayAttribute = getOverlayAttribute();

            if (overlayAttribute != null) {
                for (Track track : getAllTracks(false)) {
                    if (track != null) {
                        if (track.getTrackType() == IGVConstants.overlayTrackType) {
                            String value = track.getAttributeValue(overlayAttribute);

                            if (value != null) {
                                List<Track> trackList = overlayTracksMap.get(value);

                                if (trackList == null) {
                                    trackList = new ArrayList();
                                    overlayTracksMap.put(value, trackList);
                                }

                                trackList.add(track);
                            }
                        }
                    }
                }
            }
        }

        boolean displayOverlays = PreferenceManager.getInstance().getDiplayOverlayTracks();

        for (Track track : getAllTracks(false)) {
            if (track != null) {
                if (track.getTrackType() == IGVConstants.overlayTrackType) {
                    track.setOverlayVisible(displayOverlays);
                }
            }

        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean hasOverlays() {
        return !overlayTracksMap.isEmpty();
    }

    /**
     * @deprecated
     * Keep until Session is refactored
     *
     * @return
     */
    public List<Track> getFeatureTracks() {
        return panelTrackGroups.get(IGVMainFrame.FEATURE_PANEL_NAME).getTracks();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getTrackCount() {
        return getAllTracks(true).size();
    }

    /**
     * Method description
     *
     *
     * @param includeGeneTrack
     *
     * @return
     */
    public List<Track> getAllTracks(boolean includeGeneTrack) {
        List<Track> allTracks = new ArrayList<Track>();

        for (TrackGroup tg : panelTrackGroups.values()) {
            allTracks.addAll(tg.getTracks());
        }
        if ((geneTrack != null) && !includeGeneTrack) {
            allTracks.remove(geneTrack);
        }

        return allTracks;
    }

    public void clearTrackSelections() {
        if (log.isDebugEnabled()) {
            log.debug("enter clearTrackSelections");
        }

        for (TrackGroup tg : panelTrackGroups.values()) {
            for (Track t : tg.getTracks()) {
                t.setSelected(false);
            }
        }
    }

    public void setTrackSelections(Set<Track> selectedTracks) {
        for (Track t : getAllTracks(true)) {
            if (selectedTracks.contains(t)) {
                t.setSelected(true);
            }
        }
    }

    public void shiftSelectTracks(Track track) {
        List<Track> allTracks = getAllTracks(true);
        int clickedTrackIndex = allTracks.indexOf(track);
        // Find another track that is already selected.  The semantics of this
        // are not well defined, so any track will do
        int otherIndex = clickedTrackIndex;
        for (int i = 0; i < allTracks.size(); i++) {
            if (allTracks.get(i).isSelected() && i != clickedTrackIndex) {
                otherIndex = i;
                break;
            }
        }

        int left = Math.min(otherIndex, clickedTrackIndex);
        int right = Math.max(otherIndex, clickedTrackIndex);
        for (int i = left; i <= right; i++) {
            allTracks.get(i).setSelected(true);
        }
    }

    public void toggleTrackSelections(Set<Track> selectedTracks) {
        for (Track t : getAllTracks(true)) {
            if (selectedTracks.contains(t)) {
                t.setSelected(!t.isSelected());
            }
        }
    }

    public Collection<Track> getSelectedTracks() {
        List<Track> selectedTracks = new ArrayList();
        for (Track t : getAllTracks(true)) {
            if (t.isSelected()) {
                selectedTracks.add(t);
            }
        }
        return selectedTracks;

    }

    /**
     * Return the complete set of unique DataResourcLocators currently loaded
     * @return
     */
    public Set<ResourceLocator> getDataResourceLocators() {
        HashSet<ResourceLocator> locators = new HashSet();

        for (Track track : getAllTracks(false)) {
            ResourceLocator locator = track.getDataResourceLocator();

            if (locator != null) {
                locators.add(locator);
            }
        }

        return locators;

    }

    /**
     * Method description
     *
     *
     * @param newHeight
     */
    public void setAllTrackHeights(int newHeight) {
        for (Track track : this.getAllTracks(false)) {
            track.setHeight((int) newHeight);
        }

    }

    /**
     * Method description
     *
     *
     * @param tracksToRemove
     */
    public void removeTracks(Collection<Track> tracksToRemove) {
        for (TrackGroup tg : panelTrackGroups.values()) {
            tg.removeAll(new HashSet<Track>(tracksToRemove));
        }

        for (Track t : tracksToRemove) {
            if (t instanceof DragListener) {
                DragEventManager.getInstance().removeDragListener((DragListener) t);
            }

        }
        groupTracksByAttribute();

        // If the last track of a user defined panel is removed, remove the panel.
        List<String> panelsToRemove = new ArrayList();
        for (Map.Entry<String, TrackGroup> entry : panelTrackGroups.entrySet()) {
            if (entry.getValue().getTracks().isEmpty() &&
                    !entry.getKey().equals(IGVMainFrame.DATA_PANEL_NAME) &&
                    !entry.getKey().equals(IGVMainFrame.FEATURE_PANEL_NAME)) {
                panelsToRemove.add(entry.getKey());
            }
        }

        for (String panelName : panelsToRemove) {
            panelTrackGroups.remove(panelName);
            trackGroups.remove(panelName);
            IGVMainFrame.getInstance().removeDataPanel(panelName);
        }

    }

    /**
     * Sort all groups (data and feature) by attribute value(s).  Tracks are
     * sorted within groups.
     *
     *
     * @param attributeNames
     * @param ascending
     */
    public void sortAllTracksByAttributes(final String attributeNames[],
            final boolean[] ascending) {
        assert attributeNames.length == ascending.length;

        for (TrackGroup tg : panelTrackGroups.values()) {
            sortByAttributes(tg.getTracks(), attributeNames, ascending);
        }
        for (Map<String, TrackGroup> panelGroups : trackGroups.values()) {
            for (TrackGroup tg : panelGroups.values()) {
                sortByAttributes(tg.getTracks(), attributeNames, ascending);
            }
        }
    }

    public synchronized void moveSelectedTracksTo(Collection<Track> selectedTracks,
            String target, Track targetTrack, boolean before) {

        if (selectedTracks.isEmpty()) {
            return;
        }

        // The targetPanelTracks is the track collection for the panel we are dropping to
        TrackGroup targetTrackGroup = panelTrackGroups.get(target);
        List<Track> targetPanelTracks = targetTrackGroup.getTracks();


        // A targetTrack of null => drop at end of track
        int index = targetTrack == null ? targetPanelTracks.size() : targetPanelTracks.indexOf(
                targetTrack);
        if (!before) {
            index = index + 1;
        }

        // 1. Divdide the target list up into 2 parts, one before the index and one after
        List<Track> beforeList = new ArrayList(targetPanelTracks.subList(0,
                index));
        List<Track> afterList = new ArrayList(targetPanelTracks.subList(
                index, targetPanelTracks.size()));

        // 2.  Remove the selected tracks from anywhere they occur
        beforeList.removeAll(selectedTracks);
        afterList.removeAll(selectedTracks);
        for (TrackGroup tg : panelTrackGroups.values()) {
            if (tg != targetTrackGroup) {
                tg.getTracks().removeAll(selectedTracks);
            }
        }

        // 3. Now insert the selected tracks
        targetPanelTracks.clear();
        targetPanelTracks.addAll(beforeList);
        targetPanelTracks.addAll(selectedTracks);
        targetPanelTracks.addAll(afterList);


    // 3.  Clear selections ?
    // clearTrackSelections();


    }
    static int panelNumber = 1;

    private synchronized int getNextPanelNumber() {
        return panelNumber++;
    }

    protected TrackGroup getTrackGroupFor(ResourceLocator locator) {
        String ext = locator.getPath().toLowerCase();
        if (PreferenceManager.getInstance().isShowSingleTrackPane()) {
            return panelTrackGroups.get(IGVMainFrame.DATA_PANEL_NAME);
        } else if (ext.endsWith(".sam") || ext.endsWith(".bam") ||
                ext.endsWith(".sam.group") || ext.endsWith(".maf") ||
                ext.endsWith(".aligned") || ext.endsWith(".sorted.txt")) {
            TrackGroup dataGroup = panelTrackGroups.get(IGVMainFrame.DATA_PANEL_NAME);
            //if (dataGroup != null && dataGroup.getTracks().isEmpty()) {
            //    return dataGroup;
            //} else {
            String newPanelName = "Panel" + getNextPanelNumber();
            IGVMainFrame.getInstance().addDataPanel(newPanelName);
            return panelTrackGroups.get(newPanelName);
        //}

        } else {
            return getDefaultGroup(locator.getPath());
        }
    }

    /**
     * Load a resource (track or sample attribute file) into the specified
     * track group.
     * @param locator
     * @param group  ignored if loading a samaple info file
     * @param attributeList
     * @param messages
     */
    private void load(ResourceLocator locator, TrackGroup group,
            LinkedHashSet<Attribute> attributeList, MessageCollection messages) {

        try {
            String filename = locator.getPath().toLowerCase();

            if (filename.endsWith(".txt") || filename.endsWith(
                    ".xls") || filename.endsWith(".gz")) {
                filename = filename.substring(0, filename.lastIndexOf("."));
            }

            List<Track> newTracks = new ArrayList();

            if (filename.endsWith("h5") || filename.endsWith("hbin")) {
                loadH5File(locator, newTracks, messages);
            } else if (filename.endsWith("rnai.gct")) {
                loadRnaiGctFile(locator, newTracks, messages);
            } else if (filename.endsWith("gct") || filename.endsWith("res") || filename.endsWith(
                    "tab")) {
                loadGctFile(locator, newTracks, messages);
            } else if (filename.endsWith("cn") || filename.endsWith("xcn") || filename.endsWith(
                    "snp") || filename.endsWith("igv") || filename.endsWith("loh")) {
                loadIGVFile(locator, newTracks, messages);
            } else if (filename.endsWith("mut")) {
                loadMutFile(locator, newTracks, messages);
            } else if (filename.endsWith("cbs") || filename.endsWith("seg") || filename.endsWith(
                    "glad")) {
                loadSegFile(locator, newTracks, messages);
            } else if (filename.endsWith("seg.zip")) {
                loadBinarySegFile(locator, newTracks, messages);
            } else if (filename.endsWith("gistic")) {
                loadGisticFile(locator, newTracks, messages);
            } else if (filename.endsWith(".gs")) {
                loadRNAiGeneScoreFile(locator, newTracks, messages,
                        RNAIGeneScoreParser.Type.GENE_SCORE);
            } else if (filename.endsWith("riger")) {
                loadRNAiGeneScoreFile(locator, newTracks, messages,
                        RNAIGeneScoreParser.Type.POOLED);
            } else if (filename.endsWith(".hp")) {
                loadRNAiHPScoreFile(locator, newTracks, messages);
            } else if (filename.endsWith("gene")) {
                loadGeneFile(locator, newTracks, messages);
            } else if (filename.contains("tabblastn") || filename.endsWith(
                    "orthologs")) {
                loadSyntentyMapping(locator, newTracks, attributeList, messages);
            } else if (filename.endsWith(".sam") || filename.endsWith(".bam") ||
                    filename.endsWith(".sam.group") || filename.endsWith("_sorted") ||
                    filename.endsWith(".aligned") || filename.endsWith(".sai")) {
                loadAlignmentsTrack(locator, newTracks, messages);
            } else if (filename.endsWith(".maf")) {
                loadMAFTrack(locator, newTracks, messages);
            } else if (filename.endsWith("omega")) {
                loadOmegaTrack(locator, newTracks, messages);
            } else if (filename.endsWith("wig")) {
                loadWigFile(locator, newTracks, messages);
            } else if (filename.endsWith("list")) {
                loadListFile(locator, newTracks, messages);
            } else if (filename.endsWith("list")) {
                loadListFile(locator, newTracks, messages);
            } else {
                loadFeatureOrAttributeFile(locator, newTracks, attributeList,
                        messages);
            }

            String trackSetName = locator.getName();

            if (trackSetName == null) {
                trackSetName = new File(locator.getPath()).getName();
            }


            for (Track track : newTracks) {
                track.setSourceFile(locator.getPath());
                group.add(track);
                regsiterAttributes(track, trackSetName);
            }

        } catch (Exception e) {
            messages.append("\t");

            messages.append(
                    "There were errors reading file: " + locator.getPath());

            // TODO -- the "hack" below is used to print message details when there is
            // anything non-trivial in the message file.   We should give the user more
            // meaningful messages than the generic ones picked up here,  perhaps by catching
            // them in the parser classes and explicitly setting the message fields there before
            // rethrowing.
            if ((e.getMessage() != null) && (e.getMessage().length() > 10)) {
                messages.append("\t");
                messages.append("<br>Details: " + e.getMessage());
            }

            log.error("Error reading file: " + locator.getName(), e);
        }

    }

    private TrackGroup getDefaultGroup(String path) {

        String filename = path.toLowerCase();

        if (filename.endsWith(".txt") || filename.endsWith(".tab") || filename.endsWith(
                ".xls") || filename.endsWith(".gz")) {
            filename = filename.substring(0, filename.lastIndexOf("."));
        }



        if (filename.contains("refflat") || filename.contains("ucscgene") ||
                filename.contains("genepred") || filename.contains("ensgene") ||
                filename.contains("refgene") ||
                filename.endsWith("gff") || filename.endsWith("gtf") ||
                filename.endsWith("gff3") || filename.endsWith("embl") ||
                filename.endsWith("bed") || filename.endsWith("gistic")) {
            return panelTrackGroups.get(IGVMainFrame.FEATURE_PANEL_NAME);
        } /*
        if (filename.endsWith("h5") || filename.endsWith("h5.list") ||
        filename.endsWith("hbin") || filename.endsWith("gct") ||
        filename.endsWith("res") || filename.endsWith("dchip") ||
        filename.endsWith("cn") || filename.endsWith("xcn") ||
        filename.endsWith("snp") || filename.endsWith("igv") ||
        filename.endsWith("wig") || filename.endsWith("mut") ||
        filename.endsWith("cbs") || filename.endsWith("seg") ||
        filename.endsWith("glad") || filename.endsWith("seg.zip") ||
        filename.endsWith("dzip") ||
        filename.endsWith(".gs") || filename.endsWith("riger") ||
        filename.endsWith(".hp") || filename.endsWith(".sam") ||
        filename.endsWith(".bam") || filename.endsWith(".sam.group")) {
        return panelTrackGroups.get(IGVMainFrame.DATA_PANEL_NAME);
        } */ else {
            return panelTrackGroups.get(IGVMainFrame.DATA_PANEL_NAME);
        }
    }

    /**
     * Load the input file as a BED or Attribute (Sample Info) file.  First assume
     * it is a BED file,  if no features are found load as an attribute file.
     *
     * @param file
     * @param featureGroup
     * @param attributeList
     * @param messages
     */
    private void loadGeneFile(ResourceLocator locator, List<Track> newTracks,
            MessageCollection messages) {

        String fn = locator.getPath().toUpperCase();

        FeatureParser featureParser = AbstractFeatureParser.getInstanceFor(
                locator.getPath());
        List<FeatureTrack> tracks = featureParser.loadTracks(
                locator);
        newTracks.addAll(tracks);

    }

    private void loadSyntentyMapping(ResourceLocator locator,
            List<Track> newTracks,
            LinkedHashSet<Attribute> attributeList,
            MessageCollection messages) {

        List<Mapping> mappings = (new MappingParser()).parse(locator.getPath());
        List<Feature> features = new ArrayList(mappings.size());
        features.addAll(mappings);
        String fName = locator.getDisplayName();

        Track track = new FeatureTrack(locator, fName, features);

        // track.setRendererClass(AlignmentBlockRenderer.class);
        newTracks.add(track);
    }

    /**
     * Load the input file as a feature or attribute (Sample Info) file.  First assume
     * it is a BED file,  if no features are found load as an attribute file.
     *
     * @param file
     * @param featureGroup
     * @param attributeList
     * @param messages
     */
    private void loadFeatureOrAttributeFile(ResourceLocator locator,
            List<Track> newTracks,
            LinkedHashSet<Attribute> attributeList, MessageCollection messages) {

        FeatureParser featureParser = AbstractFeatureParser.getInstanceFor(
                locator.getPath());
        if (GCTDatasetParser.parsableMAGE_TAB(locator)) {
            locator.setDescription("MAGE_TAB");
            loadGctFile(locator, newTracks, messages);
        } else if (IGVDatasetParser.parsableMAGE_TAB(locator)) {
            locator.setDescription("MAGE_TAB");
            loadIGVFile(locator, newTracks, messages);
        } else if (featureParser.isFeatureFile(locator)) {
            List<FeatureTrack> tracks = featureParser.loadTracks(locator);
            newTracks.addAll(tracks);
        } else if (WiggleParser.isWiggle(locator)) {
            loadWigFile(locator, newTracks, messages);
        } else {
            loadAttributeFile(locator, messages);
            for (Track track : this.getAllTracks(true)) {
            }
        }
    }

    /**
     * Load the input file as a BED or Attribute (Sample Info) file.  First assume
     * it is a BED file,  if no features are found load as an attribute file.
     *
     * @param file
     * @param featureGroup
     * @param attributeList
     * @param messages
     */
    private void loadAttributeFile(ResourceLocator locator,
            MessageCollection messages) {

        String msg = AttributeManager.getInstance().loadAttributes(locator);

        if ((msg != null) && (msg.length() > 0)) {

            // TODO -- temporary until "name" property is added to locator
            String fName = locator.getDisplayName();

            messages.append("Could not determine format of file: " + fName);
        }

    }

    private void loadRnaiGctFile(ResourceLocator locator, List<Track> newTracks,
            MessageCollection messages) {

        RNAIGCTDatasetParser parser = new RNAIGCTDatasetParser(locator);

        Collection<RNAIDataSource> dataSources = parser.parse();
        if (dataSources != null) {
            for (RNAIDataSource ds : dataSources) {
                Track track = new DataSourceTrack(locator, ds.getDisplayName(), ds);

                // Set attributes.  This "hack" is neccessary to register these attributes with the
                // attribute manager to get displayed.
                track.setAttributeValue("SCREEN", ds.getScreen());

                // Set RNAi data scale -- TODO -- remove hardcoded values or make constants.
                track.setHeight(80);
                track.setDataRange(new DataRange(-3, 0, 3));
                newTracks.add(track);
            }
        }
    }

    // Hack for demo TODO -- remove
    public GCTDataset gctDataset;

    private void loadGctFile(ResourceLocator locator, List<Track> newTracks,
            MessageCollection messages) {

        GCTDatasetParser parser = null;
        GCTDataset ds = null;

        String fName = locator.getDisplayName();

        // TODO -- handle remote resource
        try {
            parser = new GCTDatasetParser(locator, null,
                    IGVModel.getInstance().getViewContext().getGenomeId());
            ds = parser.parse();
            gctDataset = ds;
        } catch (Exception e) {
            log.error("Error loading gct file ", e);
            messages.append(
                    "Unrecoverable error reading file " + fName + " [" + e.getMessage() + "]");
            return;
        }

        if ((ds == null) || ds.isEmpty()) {
            String genome = IGVModel.getInstance().getViewContext().getGenome().getName();
            messages.append("Probes in the gct file \"" + fName +
                    "\" could not be mapped to locations on the current genome (" + genome + ").");
            return;
        }

        ds.setName(fName);
        ds.setNormalized(true);
        ds.setLogValues(true);

        /*
         * File outputFile = new File(IGVMainFrame.DEFAULT_USER_DIRECTORY, file.getName() + ".h5");
         * OverlappingProcessor proc = new OverlappingProcessor(ds);
         * proc.setZoomMax(0);
         * proc.process(outputFile.getAbsolutePath());
         * loadH5File(outputFile, messages, attributeList, group);
         */

        // Counter for generating ID
        TrackProperties trackProperties = ds.getTrackProperties();
        for (String trackName : ds.getDataHeadings()) {
            Genome currentGenome = IGVModel.getInstance().getViewContext().getGenome();
            DatasetDataSource dataSource = new DatasetDataSource(currentGenome,
                    trackName, ds);

            Track track = new DataSourceTrack(locator, trackName, dataSource);

            track.setRendererClass(HeatmapRenderer.class);

            track.setTrackProperties(trackProperties);

            newTracks.add(track);
        }
    }

    private void loadIGVFile(ResourceLocator locator, List<Track> newTracks,
            MessageCollection messages) {

        String dsName = locator.getDisplayName();

        IGVDataset ds = null;

        String currentGenomeId = IGVModel.getInstance().getViewContext().getGenomeId();
        ds = new IGVDataset(currentGenomeId, locator);

        ds.setName(dsName);

        TrackProperties trackProperties = ds.getTrackProperties();
        for (String trackName : ds.getDataHeadings()) {

            Genome currentGenome = IGVModel.getInstance().getViewContext().getGenome();
            DatasetDataSource dataSource = new DatasetDataSource(currentGenome,
                    trackName, ds);
            DataSourceTrack track = new DataSourceTrack(locator, trackName,
                    dataSource);

            // track.setRendererClass(HeatmapRenderer.class);
            track.setTrackType(ds.getType());

            track.setTrackProperties(trackProperties);

            // A hack until we have a better way to set defaults
            if (track.getTrackType() == TrackType.CHIP_CHIP) {
                track.setHeight(60);
                track.setColor(Color.BLACK);
                track.setStatType(WindowFunction.max);
            }
            newTracks.add(track);
        }

    }

    private void loadWigFile(ResourceLocator locator, List<Track> newTracks,
            MessageCollection messages) {

        String genome = IGVModel.getInstance().getViewContext().getGenomeId();

        WiggleDataset ds = (new WiggleParser(locator, genome)).parse();
        TrackProperties trackProperties = ds.getTrackProperties();

        for (String heading : ds.getDataHeadings()) {
            String trackName = heading;
            if (trackProperties.getName() != null) {
                trackName = trackProperties.getName();
            }

            Genome currentGenome = IGVModel.getInstance().getViewContext().getGenome();
            DatasetDataSource dataSource = new DatasetDataSource(currentGenome,
                    trackName, ds);
            Track track = new DataSourceTrack(locator, trackName, dataSource);
            track.setTrackProperties(trackProperties);

            track.setTrackType(ds.getType());
            newTracks.add(track);
        }


    // OverlappingProcessor proc = new OverlappingProcessor(ds);

    // proc.setZoomMax(0);
    // proc.process(outputFile.getAbsolutePath());

    // loadH5File(locator, newTracks, messages);


    }

    private void loadListFile(ResourceLocator locator, List<Track> newTracks,
            MessageCollection messages) {
        try {
            // Hack for testing -- TODO remove this
            if (locator.getPath().endsWith("h5.list")) {
                Track track = new HDFListTrack(locator, locator.getDisplayName());
                newTracks.add(track);
            } else {
                Track track = new FeatureDirTrack(locator, locator.getDisplayName());
                newTracks.add(track);
            }

        } catch (Exception ex) {
            log.error("Error loading " + locator.getDisplayName(), ex);
            messages.append("Error loading " + locator.getDisplayName() + ": " + ex.getMessage());
        }



    // OverlappingProcessor proc = new OverlappingProcessor(ds);

    // proc.setZoomMax(0);
    // proc.process(outputFile.getAbsolutePath());

    // loadH5File(locator, newTracks, messages);


    }

    private void loadGisticFile(ResourceLocator locator, List<Track> newTracks,
            MessageCollection messages) {

        GisticTrack track = GisticFileParser.loadData(locator);

        if (track == null) {
            messages.append(locator.getPath());
        }

        track.setHeight(GISTIC_TRACK_HEIGHT);
        newTracks.add(track);
    }

    /**
     * Load the data from an HDF5 file and add resulting tracks to the supplied TrackGroup.
     * Error messages are appended to the MessageCollection
     *
     * @param locator
     * @param group
     * @param messages
     */
    private void loadH5File(ResourceLocator locator, List<Track> newTracks,
            MessageCollection messages) {

        TrackSet trackSet = null;

        // TODO -- temporary until "name" property is added to locator
        String fName = null;

        try {

            // TODO -- temporary until "name" property is added to locator
            fName = locator.getDisplayName();

            HDFDataManager dataManager = HDFDataManagerFactory.getDataManager(
                    locator);
            TrackProperties trackProperties = dataManager.getTrackProperties();
            String[] trackNames = dataManager.getTrackNames();

            List<Track> tracks = new ArrayList();

            for (int trackNumber = 0; trackNumber < trackNames.length; trackNumber++) {
                String name = trackNames.length == 1 ? fName : trackNames[trackNumber];
                Track track = new HDFDataTrack(dataManager, locator, name, trackNumber);
                if (trackProperties != null) {
                    track.setTrackProperties(trackProperties);
                }
                tracks.add(track);
            }

            trackSet = new TrackSet(tracks);

            if (trackSet.isEmpty()) {
                messages.append("No data found in file " + fName);
            }

            newTracks.addAll(trackSet.getTracks());

        } catch (Exception e) {
            log.error("Error loading resource! ", e);

            String detailMessage = e.getMessage();

            messages.append(
                    "Error loading " + locator.getDisplayName() + "   <i>" + detailMessage + "</i>");
        }
    }

    /**
     * Load a rnai gene score file and create a datasource and track.
     *
     * // TODO -- change parser to use resource locator rather than path.
     *
     * @param locator
     * @param newTracks
     * @param messages
     */
    private void loadRNAiGeneScoreFile(ResourceLocator locator,
            List<Track> newTracks,
            MessageCollection messages, RNAIGeneScoreParser.Type type) {

        String dsName = locator.getDisplayName();

        String genomeId = IGVModel.getInstance().getViewContext().getGenomeId();
        RNAIGeneScoreParser parser = new RNAIGeneScoreParser(locator.getPath(),
                genomeId, type);

        Collection<RNAIDataSource> dataSources = parser.parse();
        for (RNAIDataSource ds : dataSources) {
            Track track = new DataSourceTrack(locator, ds.getDisplayName(), ds);

            // Set attributes.  This "hack" is neccessary to register these attributes with the
            // attribute manager to get displayed.
            track.setAttributeValue("SCREEN", ds.getScreen());
            if ((ds.getCondition() != null) && (ds.getCondition().length() > 0)) {
                track.setAttributeValue("CONDITION", ds.getCondition());
            }


            // Set RNAi data scale -- TODO -- remove hardcoded values or make constants.
            track.setHeight(80);
            track.setDataRange(new DataRange(-3, 0, 3));
            newTracks.add(track);
        }
    }

    /**
     * Load a RNAi haripin score file.  The results of this action are hairpin scores
     * added to the RNAIDataManager.  Currently no tracks are created for hairpin
     * scores, although this could change.
     *
     * // TODO -- change parser to use resource locator rather than path.
     *
     * @param locator
     * @param newTracks
     * @param messages
     */
    private void loadRNAiHPScoreFile(ResourceLocator locator,
            List<Track> newTracks,
            MessageCollection messages) {

        String dsName = locator.getDisplayName();

        (new RNAIHairpinParser(locator.getPath())).parse();
    }

    private void loadMAFTrack(ResourceLocator locator, List<Track> newTracks,
            MessageCollection messages) {

        MAFTrack t = new MAFTrack(locator, "Multiple Alignments");

        newTracks.add(t);
    }

    private void loadOmegaTrack(ResourceLocator locator, List<Track> newTracks,
            MessageCollection messages) {
        OmegaDataSource ds = new OmegaDataSource();
        Track track = new OmegaTrack(locator, "Conservation (Omega)", ds);

        // Set RNAi data scale -- TODO -- remove hardcoded values or make constants.
        track.setHeight(40);
        //track.setDataRange(new DataRange(-3, 0, 3));
        newTracks.add(track);
    }

    /**
     * Load a rnai gene score file and create a datasource and track.
     *
     * // TODO -- change parser to use resource locator rather than path.
     *
     * @param locator
     * @param newTracks
     * @param messages
     */
    private void loadAlignmentsTrack(ResourceLocator locator, List<Track> newTracks,
            MessageCollection messages) {

        try {
            String dsName = locator.getDisplayName();

            // If the user tried to load the index,  look for the file (this is a common mistake)
            if (locator.getPath().endsWith(".sai")) {
                String alignmentFileName = locator.getPath().substring(0, locator.getPath().length() - 4);
                locator = new ResourceLocator(locator.getServerURL(), alignmentFileName);
            }


            AlignmentTrack track = new AlignmentTrack(locator, dsName);    // parser.loadTrack(locator, dsName);

            // Search for an h5 count file
            if (locator.isLocal()) {
                // TODO -- verify index here and don't load track if missing (or can't be created)
            }
            track.preloadData();
            newTracks.add(track);
        } catch (Exception e) {
            JOptionPane.showMessageDialog(IGVMainFrame.getInstance(),
                    "Error loading sam track (" + e.getMessage() + ")");
            log.error("Error loading sam track", e);
        }

    }

    /**
     * Load a ".mut" file (muation file) and create tracks.
     * @param locator
     * @param newTracks
     * @param messages
     */
    private void loadMutFile(ResourceLocator locator, List<Track> newTracks,
            MessageCollection messages) {

        String dsName = locator.getDisplayName();

        List<Track> mutationTracks = MutationParser.loadMutationTracks(locator);

        if (mutationTracks.isEmpty()) {
            messages.append(dsName);
        }

        for (Track track : mutationTracks) {
            track.setTrackType(TrackType.MUTATION);
            track.setRendererClass(MutationRenderer.class);
            newTracks.add(track);
        }
    }

    private void loadSegFile(ResourceLocator locator, List<Track> newTracks,
            MessageCollection messages) {

        // TODO - -handle remote resource
        File trackSetFile = new File(locator.getPath());

        SegmentedAsciiDataSet ds = null;

        try {
            String currentGenome = IGVModel.getInstance().getViewContext().getGenomeId();

            ds = new SegmentedAsciiDataSet(locator);
        } catch (Exception e) {
            messages.append("\n\t");
            messages.append("Error reading file " + locator.getDisplayName());
            messages.append("\n\tError message: " + e.getMessage());
            log.error("Error reading file: " + locator.getDisplayName(), e);
        }

        for (String trackName : ds.getDataHeadings()) {

            SegmentedDataSource dataSource = new SegmentedDataSource(trackName, ds);
            Track track = new DataSourceTrack(locator, trackName, dataSource);

            track.setRendererClass(HeatmapRenderer.class);
            track.setTrackType(ds.getType());
            newTracks.add(track);
        }
    }

    private void loadBinarySegFile(ResourceLocator locator,
            List<Track> newTracks,
            MessageCollection messages) {

        SegmentedBinaryDataSet ds = null;

        try {
            ds = new SegmentedBinaryDataSet(locator);
        } catch (Exception e) {
            messages.append("\n\t");
            messages.append("Error reading file " + locator.getDisplayName());
            messages.append("\n\tError message: " + e.getMessage());
            log.error("Error reading file: " + locator.getDisplayName(), e);
        }

        for (String trackName : ds.getSampleNames()) {

            SegmentedDataSource dataSource = new SegmentedDataSource(
                    trackName, ds);
            Track track = new DataSourceTrack(locator, trackName, dataSource);

            track.setRendererClass(HeatmapRenderer.class);

            track.setTrackType(ds.getType());
            newTracks.add(track);
        }
    }

    private void regsiterAttributes(Track track, String trackSetName) {
        track.setAttributeValue("DATA FILE", trackSetName);
        track.setAttributeValue("DATA TYPE", track.getTrackType().toString());
        track.setAttributeValue("NAME", track.getId());
    }

    private void sortByAttributes(List<Track> tracks,
            final String attributeNames[],
            final boolean[] ascending) {
        if ((tracks != null) && !tracks.isEmpty()) {
            Comparator comparator = new Comparator() {

                public int compare(Object arg0, Object arg1) {
                    Track t1 = (Track) arg0;
                    Track t2 = (Track) arg1;

                    for (int i = 0; i < attributeNames.length; i++) {
                        String attName = attributeNames[i];

                        if (attName != null) {
                            String value1 = t1.getAttributeValue(attName);

                            if (value1 == null) {
                                value1 = "";
                            }

                            String value2 = t2.getAttributeValue(attName);

                            if (value2 == null) {
                                value2 = "";
                            }

                            int c = 0;
                            int k = 0;
                            if (value1.matches("[^0-9]") || value2.matches("[^0-9]")) {
                                c = value1.compareTo(value2);
                            } else {
                                boolean numeric = false;
                                while (k < value1.length() && k < value2.length()) {
                                    while (k < value1.length() && k < value2.length() && Character.isDigit(value1.charAt(
                                            k)) && Character.isDigit(value2.charAt(k))) {
                                        int num1 = Character.getNumericValue(value1.charAt(k));
                                        int num2 = Character.getNumericValue(value2.charAt(k));

                                        if (c == 0) {
                                            if (num1 < num2) {
                                                c = -1;
                                            }
                                            if (num1 > num2) {
                                                c = 1;
                                            }
                                        }
                                        k++;
                                        numeric = true;
                                    }

                                    if (numeric && k < value1.length() && Character.isDigit(value1.charAt(
                                            k))) {
                                        c = 1;
                                        numeric = false;
                                    } else if (numeric && k < value2.length() && Character.isDigit(value2.charAt(
                                            k))) {
                                        c = -1;
                                        numeric = false;
                                    }

                                    if (k < value1.length() && k < value2.length() && c == 0) {
                                        c = Character.valueOf(value1.charAt(k)).compareTo(value2.charAt(
                                                k));
                                    }

                                    if (c != 0) {
                                        break;
                                    }

                                    k++;
                                }
                            }

                            if (c == 0 && k < value1.length()) {
                                c = 1;
                            }

                            if (c == 0 && k < value2.length()) {
                                c = -1;
                            }

                            if (c != 0) {
                                return (ascending[i] ? c : -c);
                            }
                        }
                    }

                    // All compares are equal
                    return 0;
                }
            };

            Collections.sort(tracks, comparator);
        }

    }

    /**
     * Sort all groups (data and feature) by a computed score over a region.  The
     * sort is done twice (1) groups are sorted with the featureGroup, and (2) the
     * groups themselves are sorted.
     *
     * @param region
     * @param type
     */
    public void sortGroup(final RegionOfInterest region,
            final RegionScoreType type) {

        boolean useLinkedSorting = PreferenceManager.getInstance().isLinkedSortingEnabled();
        String linkingAtt = PreferenceManager.getInstance().getOverlayAttribute();

        for (Map.Entry<String, Map<String, TrackGroup>> entry : trackGroups.entrySet()) {
            String dataPanelName = entry.getKey();
            Map<String, TrackGroup> groupsMap = entry.getValue();

            if (groupsMap.size() <= 1) {
                if ((linkingAtt == null) || !useLinkedSorting) {
                    sortByRegionScore(panelTrackGroups.get(dataPanelName).getTracks(), region, type);
                } else {
                    sortGroup(panelTrackGroups.get(dataPanelName), region, linkingAtt, type);
                }
            } else {
                List<TrackGroup> dataGroups = new ArrayList();
                dataGroups.addAll(groupsMap.values());
                sortGroupsByRegionScore(dataGroups, region, type);
                groupsMap.clear();

                for (TrackGroup group : dataGroups) {
                    groupsMap.put(group.getAttributeValue(), group);

                    // If there is a non-null linking attribute
                    // Segregate tracks into 2 groups, those matching the score type and those that do not
                    if ((linkingAtt == null) || !useLinkedSorting) {
                        sortByRegionScore(group.getTracks(), region, type);
                    } else {
                        sortGroup(group, region, linkingAtt, type);
                    }
                }
            }
        }
    }

    private void sortByRegionScore(List<Track> tracks,
            final RegionOfInterest region,
            final RegionScoreType type) {
        if ((tracks != null) && (region != null) && !tracks.isEmpty()) {
            final int zoom = Math.max(0,
                    IGVModel.getInstance().getViewContext().getZoom());
            final String chr = region.getChromosomeName();
            final int start = region.getStart();
            final int end = region.getEnd();

            Comparator<Track> c = new Comparator<Track>() {

                public int compare(Track t1, Track t2) {
                    float s1 = t1.getRegionScore(chr, start, end, zoom, type);
                    float s2 = t2.getRegionScore(chr, start, end, zoom, type);

                    if (s2 > s1) {
                        return 1;
                    } else if (s1 < s2) {
                        return -1;
                    } else {
                        return 0;
                    }

                }
            };

            Collections.sort(tracks, c);

        }

    }

    private void sortGroup(TrackGroup group, final RegionOfInterest region,
            String linkingAtt,
            final RegionScoreType type) {
        List<Track> tracksWithScore = new ArrayList(group.getTracks().size());
        List<Track> otherTracks = new ArrayList(group.getTracks().size());
        for (Track t : group.getTracks()) {
            if (t instanceof DataTrack) {
                if (((DataTrack) t).isRegionScoreType(type)) {
                    tracksWithScore.add(t);
                } else {
                    otherTracks.add(t);
                }
            } else {
                otherTracks.add(t);
            }
        }

        sortByRegionScore(tracksWithScore, region, type);
        List<String> sortedAttributes = new ArrayList();
        for (Track t : tracksWithScore) {
            String att = t.getAttributeValue(linkingAtt);
            if (att != null) {
                sortedAttributes.add(att);
            }
        }
        sortByAttributeOrder(otherTracks, sortedAttributes, linkingAtt);

        group.clear();
        group.addAll(tracksWithScore);
        group.addAll(otherTracks);
    }

    // TODO -- refactor to elimate this copied method, use interface for
    // things with region scores (tracks & track groups).
    private void sortGroupsByRegionScore(List<TrackGroup> groups,
            final RegionOfInterest region,
            final RegionScoreType type) {
        if ((groups != null) && (region != null) && !groups.isEmpty()) {
            final int zoom = Math.max(0,
                    IGVModel.getInstance().getViewContext().getZoom());
            final String chr = region.getChromosomeName();
            final int start = region.getStart();
            final int end = region.getEnd();
            Comparator<TrackGroup> c = new Comparator<TrackGroup>() {

                public int compare(TrackGroup t1, TrackGroup t2) {
                    float s1 = t1.getRegionScore(chr, start, end, zoom, type);
                    float s2 = t2.getRegionScore(chr, start, end, zoom, type);

                    if (s2 > s1) {
                        return 1;
                    } else if (s1 < s2) {
                        return -1;
                    } else {
                        return 0;
                    }

                }
            };

            Collections.sort(groups, c);
        }
    }

    /**
     *
     * @param tracks
     * @param region
     * @param type
     */
    private void sortByAttributeOrder(List<Track> tracks,
            List<String> sortedAttributes,
            final String attributeId) {
        if ((tracks != null) && (sortedAttributes != null) && !tracks.isEmpty()) {

            // Create a rank hash.  Loop backwards so that the lowest index for an attribute
            final HashMap<String, Integer> rankMap = new HashMap(
                    sortedAttributes.size() * 2);
            for (int i = sortedAttributes.size() - 1; i >= 0; i--) {
                rankMap.put(sortedAttributes.get(i), i);
            }

            // Comparator for sorting in ascending order
            Comparator<Track> c = new Comparator<Track>() {

                public int compare(Track t1, Track t2) {
                    String a1 = t1.getAttributeValue(attributeId);
                    String a2 = t2.getAttributeValue(attributeId);
                    Integer r1 = ((a1 == null) ? null : rankMap.get(a1));
                    Integer r2 = ((a2 == null) ? null : rankMap.get(a2));
                    if ((r1 == null) && (r2 == null)) {
                        return 0;
                    } else if (r1 == null) {
                        return 1;
                    } else if (r2 == null) {
                        return -1;
                    } else {
                        return r1.intValue() - r2.intValue();
                    }

                }
            };

            Collections.sort(tracks, c);

        }

    }

    /**
     * Sort all groups (data and feature) by a computed score over a region.  The
     * sort is done twice (1) groups are sorted with the featureGroup, and (2) the
     * groups themselves are sorted.
     *
     * @param region
     * @param type
     */
    public void doROC(final RegionOfInterest region, final RegionScoreType type) {
    }

    /*
     *     class LinkedSampleComparator<Track> implements Comparator<Track> {
     * RegionScoreType scoreType;
     * String linkingAttribute;
     * public int compare(Track t1, Track t2) {
     * }
     * }
     */
    /**
     * Method description
     *
     *
     * @return
     */
    public String getGroupByAttribute() {
        return groupByAttribute;
    }

//  To support mutation track overlay.  Generalize later.  Cancer specific option.
    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isDoOverlays() {
        return PreferenceManager.getInstance().getOverlayTracks();
    }

    /**
     * Return the track type
     * @return
     */
    public TrackType getOverlayTrackType() {
        return IGVConstants.overlayTrackType;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getOverlayAttribute() {
        return PreferenceManager.getInstance().getOverlayAttribute();
    }

    /**
     * Method description
     *
     *
     * @param value
     *
     * @return
     */
    public List<Track> getOverlayTracks(String value) {
        return overlayTracksMap.get(value);
    }

    /**
     * Method description
     *
     *
     * @param track
     *
     * @return
     */
    public List<Track> getOverlayTracks(Track track) {
        String overlayAttribute = PreferenceManager.getInstance().getOverlayAttribute();
        String value = track.getAttributeValue(overlayAttribute);

        return getOverlayTracks(value);

    }

    /**
     * Method description
     *
     *
     * @param genomeId
     */
    public void loadGeneTrack(String genomeId) {

        // Load genes
        geneData = GeneManager.getGeneManager(genomeId);


        /*
         * if (geneData == null)
         * {
         *   return;
         * }
         */

        Map<String, List<Feature>> featureMap = null;
        if (geneData != null) {
            featureMap = geneData.getChromsomeGeneMap();
        } else {
            featureMap = new HashMap<String, List<Feature>>();
        }

        FeatureTrack geneFeatureTrack = new FeatureTrack(null, GENE_TRACK_NAME,
                featureMap);

        geneFeatureTrack.setMinimumHeight(35);
        geneFeatureTrack.setHeight(45);
        geneFeatureTrack.setRendererClass(GeneTrackRenderer.class);
        geneFeatureTrack.setColor(Color.BLUE.darker());

        SequenceTrack seqTrack = new SequenceTrack("Reference");

        if (geneData != null) {
            String geneTrackName = geneData.getGeneTrackName();

            GeneTrack gt = new GeneTrack(geneFeatureTrack, seqTrack);
            gt.setDisplayName(geneTrackName);

            setGeneTrack(gt);
        }
        else {
            setGeneTrack(seqTrack);
        }

    }

    /**
     * Method description
     *
     *
     * @return
     */
    public GeneManager getGeneManager() {
        return geneData;
    }

    /**
     * Method description
     *
     *
     * @param id
     * @param match
     *
     * @return
     */
    public synchronized List<Track> findTracksById(String id, MatchTrack match) {

        List<Track> tracks = new ArrayList();

        if (id != null) {
            for (TrackGroup tg : panelTrackGroups.values()) {
                List<Track> dataTracks = tg.getTracks();

                for (Track track : dataTracks) {
                    if (track != null) {
                        if (MatchTrack.EXACT.equals(match)) {
                            if (track.getId().equalsIgnoreCase(id)) {
                                tracks.add(track);
                            }

                        } else if (MatchTrack.CONTAINS.equals(match)) {
                            if (track.getId().toLowerCase().indexOf(id.toLowerCase()) != -1) {
                                tracks.add(track);
                            }

                        }
                    }
                }
            }
        }

        return tracks;
    }

    /**
     * Method description
     *
     */
    public void refreshTrackData() {
        long timestamp = System.currentTimeMillis();

        for (Track track : this.getAllTracks(false)) {
            track.refreshData(timestamp);
        }

    }

    /**
     * Return the set of DataResourceLocators for all loaded tracks..
     * Used to test if a track has been previously loaded.
     * This set could be maintained as tracks are loaded and removed to avoid
     * the need to compute it on the fly, but this could be error prone.
     * @return
     */
    public Set<ResourceLocator> getLocatorSet() {
        HashSet<ResourceLocator> locators = new HashSet();

        for (Track track : getAllTracks(false)) {
            locators.add(track.getDataResourceLocator());
        }

        return locators;
    }

    /*
    public void updateTrackPositions(String panelName) {
    int regionY = 0;
    Collection<TrackGroup> groups = trackGroups.get(panelName).values();
    for (TrackGroup group : groups) {
    if (group.isVisible()) {
    if (groups.size() > 1) {
    regionY += IGVConstants.groupGap;
    }
    int previousRegion = -1;
    for (Track track : group.getTracks()) {
    int trackHeight = track.getHeight();
    if (track.isVisible()) {
    if (isDragging) {
    Color currentColor = g.getColor();
    if (dragY >= previousRegion && dragY <= regionY) {
    g.setColor(Color.GRAY);
    g.drawLine(0, regionY, getWidth(), regionY);
    regionY++;
    g.setColor(currentColor);
    dropTarget = track;
    } else if (dragY >= (regionY + trackHeight)) {
    g.setColor(Color.GRAY);
    g.drawLine(0, regionY + trackHeight,
    getWidth(), regionY + trackHeight);
    trackHeight--;
    g.setColor(currentColor);
    dropTarget = null;
    }
    }
    previousRegion = regionY;
    if (regionY + trackHeight >= visibleRect.y) {
    int width = getWidth();
    int height = track.getHeight();
    Rectangle region = new Rectangle(regionX,
    regionY, width, height);
    addMousableRegion(new MouseableRegion(region, track));
    draw(graphics2D, track, regionX, regionY, width,
    height, visibleRect);
    }
    regionY += trackHeight;
    }
    }
    if (group.isDrawBorder()) {
    g.drawLine(0, regionY, getWidth(), regionY);
    }
    }
    }
    }
     * */
    /**
     * Method description
     *
     */
    /*
    public void dumpData() {
    PrintWriter pw = null;
    try {
    String chr = IGVModel.getInstance().getViewContext().getChromosome().getName();
    double x = IGVModel.getInstance().getViewContext().getOrigin();
    pw = new PrintWriter(new FileWriter("DataDump.tab"));
    pw.println("Sample\tCopy Number\tExpression\tMethylation");
    for (String name : groupsMap.keySet()) {
    pw.print(name);
    float cn = 0;
    float exp = 0;
    float meth = 0;
    int nCn = 0;
    int nExp = 0;
    int nMeth = 0;
    for (Track t : groupsMap.get(name).getTracks()) {
    LocusScore score = ((DataTrack) t).getLocusScoreAt(chr, x);
    if ((score != null) && !Float.isNaN(score.getScore())) {
    if (t.getTrackType() == TrackType.COPY_NUMBER) {
    nCn++;
    cn += score.getScore();
    } else if (t.getTrackType() == TrackType.GENE_EXPRESSION) {
    nExp++;
    exp += score.getScore();
    } else if (t.getTrackType() == TrackType.DNA_METHYLATION) {
    nMeth++;
    meth += score.getScore();
    }
    }
    }
    pw.print("\t");
    if (nCn > 0) {
    pw.print(cn / nCn);
    }
    pw.print("\t");
    if (nExp > 0) {
    pw.print(exp / nExp);
    }
    pw.print("\t");
    if (nMeth > 0) {
    pw.print(cn / meth);
    }
    pw.println();
    }
    for (Track track : getTracksForPanel(false)) {
    if (track instanceof DataTrack) {
    LocusScore score = ((DataTrack) track).getLocusScoreAt(chr,
    x);
    if (score != null) {
    pw.println(
    track.getDisplayName() + "\t" + track.getTrackType() + "\t" + score.getScore());
    }
    }
    }
    } catch (IOException ex) {
    ex.printStackTrace();
    } finally {
    pw.close();
    }
    }     * */
    /**
     * Utility class for creating tracks from and HDF5 file.
     *
     */
    static class HDF5TrackReader {

        ResourceLocator locator;
        HDFDataManager dataManager;

        /**
         * Constructs ...
         *
         *
         * @param locator
         */
        public HDF5TrackReader(ResourceLocator locator) {
            this.locator = locator;
            dataManager = HDFDataManagerFactory.getDataManager(locator);
        }
    }
}
