/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
* To change this template, choose Tools | Templates
* and open the template in the editor.
 */
package org.broad.igv.track;

//~--- JDK imports ------------------------------------------------------------

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;

/**
 * Container for a group of tracks.  Behaves as a single unit when sorting
 * by region score.
 *
 * @author jrobinso
 */
public class TrackGroup {

    /**
     * Key used to group tracks (e.g. SAMPLE_ID).
     */
    String attributeKey;
    private String attributeValue;
    private boolean drawBorder = true;
    List<Track> tracks;

    /**
     * Constructs ...
     *
     */
    public TrackGroup() {
        this("", "");
    }

    /**
     * Constructs ...
     *
     *
     * @param attributeKey
     * @param attributeValue
     */
    public TrackGroup(String attributeKey, String attributeValue) {
        this.attributeKey = attributeKey;
        this.attributeValue = attributeValue;
        tracks = new ArrayList<Track>();
    }

    /**
     * Method description
     *
     *
     * @param track
     *
     * @return
     */
    public boolean contains(Track track) {
        return tracks.contains(track);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public  int getPreferredHeight() {
        int height = 0;
        {
            for (Track track : tracks)
            {
                if (track.isVisible())
                {
                    height += track.getPreferredHeight();
                }
            }
        }
        return height;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public List<Track> getTracks() {
        return tracks;
    }

    /**
     * Method description
     *
     *
     * @param track
     *
     * @return
     */
    public int indexOf(Track track) {
        return tracks.indexOf(track);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int size() {
        return tracks.size();
    }

    /**
     * Method description
     *
     *
     * @param track
     */
    public void add(Track track) {
        this.tracks.add(track);
    }

    /**
     * Method description
     *
     *
     * @param pos
     * @param track
     */
    public void add(int pos, Track track) {
        this.tracks.add(pos, track);
    }

    /**
     * 
     */
    public void addAll(Collection<Track> trackList) {
        tracks.addAll(trackList);
    }

    public void addAll(int index, Collection<Track> trackList) {
        tracks.addAll(index, trackList);
    }

    /**
     * Remove all tracks in supplied list. The list is rebuilt for
     * effeciency, 1 pass is needed rather than 2 and the membership
     * test on a set is very fast.
     * 
     * @param tracksToRemove
     */
    public void removeAll(Collection<Track> tracksToRemove) {
        tracks.removeAll(tracksToRemove);

    }

    /**
     * Method description
     *
     *
     * @param track
     */
    public void remove(Track track) {
        tracks.remove(track);
    }


    /**
     * Method description
     *
     */
    public void clear() {
        tracks.clear();
    }

    /**
     * Returns a representative value for an attribute.  Its not clear what
     * the rule for groups should be, for now just takes the first non-null
     * value from a track.
     *
     * @param attributeKey
     *
     * @return
     */
    public String getAttributeValue(String attributeKey) {

        for (Track track : tracks)
        {
            String value = track.getAttributeValue(attributeKey);
            if ((value != null) &&!value.equals(attributeKey))
            {
                return value;
            }
        }
        return "";
    }

    /**
     * Return a composite score for the entire group.  For now use the maximum
     * score.   Note that scores for tracks not appropriate to the score type will
     * return -Float.MAX, so they are effectively ignored.
     *
     * @param chr
     * @param start
     * @param end
     * @param zoom
     * @param type
     * @return
     */
    public float getRegionScore(String chr, int start, int end, int zoom, RegionScoreType type) {
        float score = -Float.MAX_VALUE;
        for (Track track : tracks)
        {
            score = Math.max(score, track.getRegionScore(chr, start, end, zoom, type));
        }
        return score;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getAttributeValue() {
        return attributeValue;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isDrawBorder() {
        return drawBorder;
    }

    /**
     * Method description
     *
     *
     * @param drawBorder
     */
    public void setDrawBorder(boolean drawBorder) {
        this.drawBorder = drawBorder;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isVisible() {
        for (Track t : tracks)
        {
            if ((t != null) && t.isVisible())
            {
                return true;
            }
        }
        return false;
    }
}
