/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.track;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.log4j.Logger;
import org.broad.igv.data.DatasetParserUtils;
import org.broad.igv.data.HDFDataManager;
import org.broad.igv.data.HDFDataManagerFactory;
import org.broad.igv.feature.Feature;
import org.broad.igv.feature.LocusScore;
import org.broad.igv.feature.ParsingUtils;
import org.broad.igv.util.AsciiLineReader;
import org.broad.igv.util.ObjectCache;
import org.broad.igv.util.ResourceLocator;

/**
 *
 * @author jrobinso
 *
 * TODO -- generalize this
 */
public class HDFListTrack extends DataTrack {

    static Logger log = Logger.getLogger(HDFListTrack.class);
    ObjectCache<String, List<Feature>> featureCache;
    List<Feature> wholeGenomeFeatures = new ArrayList();
    Map<String, ResourceLocator> locatorMap;
    Map<String, HDFDataTrack> trackMap = new HashMap();
    File rootDir;
    ResourceLocator rootLocator;

    public HDFListTrack(ResourceLocator locator, String name) throws IOException {
        super(locator, name);
        featureCache = new ObjectCache(3);
        rootLocator = locator;
        rootDir = (new File(locator.getPath())).getParentFile();

        init(locator);
    }

    //  chr    path   server
    private void init(ResourceLocator locator) {
        locatorMap = new HashMap();
        AsciiLineReader reader = null;
        try {
            reader = ParsingUtils.openAsciiReader(locator);
            String nextLine;
            while ((nextLine = reader.readLine()) != null) {
                if (nextLine.startsWith("track")) {
                    TrackProperties tp = new TrackProperties();
                    DatasetParserUtils.parseTrackLine(nextLine, tp);
                    setTrackProperties(tp);

                }
                String[] tokens = nextLine.split("\t");
                if (tokens.length < 2) {
                 //   log.info("Unexpected # of tokens in file: " + locator.getPath() + " => " + nextLine);
                } else {
                    String chr = tokens[0].trim();
                    String path = tokens[1].trim();
                    String server = null;
                    if (tokens.length > 2) {
                        server = tokens[2].trim();
                    }
                    locatorMap.put(chr, new ResourceLocator(server, path));
                }
            }

        } catch (IOException ex) {
            log.error("Error parsing sam file group file", ex);
            throw new RuntimeException("Error parsing file: " + locator.getPath() + " (" +
                    ex.getMessage());
        } finally {
            if (reader != null) {
                reader.close();
            }
        }

    }

    private HDFDataTrack loadTrack(String chr) throws FileNotFoundException {

        ResourceLocator locator = locatorMap.get(chr);
        if (locator == null) {
            return null;
        }

        HDFDataManager dataManager = HDFDataManagerFactory.getDataManager(locator);
        TrackProperties trackProperties = dataManager.getTrackProperties();

        HDFDataTrack track = new HDFDataTrack(dataManager, locator, this.getDisplayName(), 0);
        if (trackProperties != null) {
            track.setTrackProperties(trackProperties);
        }
        return track;
    }
    // TODO -- get window functions dynamically from data
    static List<WindowFunction> wfs = new ArrayList();


    static {
        wfs.add(WindowFunction.percentile10);
        wfs.add(WindowFunction.median);
        wfs.add(WindowFunction.mean);
        wfs.add(WindowFunction.percentile90);
        wfs.add(WindowFunction.max);

    }

    @Override
    public Collection<WindowFunction> getAvailableWindowFunctions() {
        return wfs;
    }

    @Override
    public List<LocusScore> getSummaryScores(String chr, int startLocation, int endLocation, int zoom) {

        HDFDataTrack t = trackMap.get(chr);
        if (t == null) {
            try {
                t = loadTrack(chr);
                trackMap.put(chr, t);
            } catch (FileNotFoundException fileNotFoundException) {
                log.error("Error loading track", fileNotFoundException);
            }
        }
        return t == null ? new ArrayList() : t.getSummaryScores(chr, startLocation, endLocation, zoom);
    }

    public void setStatType(WindowFunction type) {
        for (HDFDataTrack t : trackMap.values()) {
            t.setStatType(type);
        }
    }

    public WindowFunction getWindowFunction() {
        return trackMap.values().iterator().next().getWindowFunction();
    }

    public boolean isLogNormalized() {
        return trackMap.values().iterator().next().isLogNormalized();
    }

    public float getRegionScore(String chr, int start, int end, int zoom, RegionScoreType type) {
        HDFDataTrack t = trackMap.get(chr);
        return t == null ? 0.0f : t.getRegionScore(chr, start, end, zoom, type);
    }
}

