/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
/*
 * FeatureTrackH5.java
 *
 * Created on November 12, 2007, 8:22 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */
package org.broad.igv.track;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;


import org.broad.igv.feature.LocusScore;

import org.broad.igv.util.ResourceLocator;
import org.broad.igv.renderer.DataRenderer;
import org.broad.igv.ui.IGVModel;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;

import java.awt.Rectangle;

import java.util.List;
import org.broad.igv.feature.FeatureUtils;

/**
 * Represents a feature track backed by an HDF-5 file.
 *
 * @author jrobinso
 */
public abstract class DataTrack extends AbstractTrack {

    private static Logger log = Logger.getLogger(DataTrack.class);
    private DataRenderer renderer;

    /**
     * Constructs ...
     *
     *
     * @param locator
     * @param name
     */
    public DataTrack(ResourceLocator locator, String name) {
        super(locator, name);

    }

    public void preloadData(String chrName, int start, int end, int zoom) {
        getSummaryScores(chrName, start, end, zoom);
    }

    /**
     * Method description
     *
     *
     * @param context
     * @param rect
     */
    public void render(RenderContext context, Rectangle rect) {
        List<LocusScore> scores = getSummaryScores(context.getChr(),
                (int) context.getStartLocation(),
                (int) context.getEndLocation() + 1, context.getZoom());

        getRenderer().render(this, scores, context, rect);
    }

    /**
     * Method description
     *
     *
     * @param rc
     */
    public void setRendererClass(Class rc) {
        try {
            renderer = (DataRenderer) rc.newInstance();
        } catch (Exception ex) {
            log.error("Error instatiating renderer ", ex);
        }
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public DataRenderer getRenderer() {
        if (renderer == null) {
            setRendererClass(getDefaultRendererClass());
        }
        return renderer;
    }

    /**
     * Method description
     *
     *
     * @param chr
     * @param position
     * @param y
     *
     * @return
     */
    public String getValueStringAt(String chr, double position, int y) {
        LocusScore score = getLocusScoreAt(chr, position);
        return ((score == null) ? "" : score.getValueString(position, getWindowFunction()));
    }

    /**
     * Method description
     *
     *
     * @param chr
     * @param position
     *
     * @return
     */
    public LocusScore getLocusScoreAt(String chr, double position) {
        int zoom = Math.max(0, IGVModel.getInstance().getViewContext().getZoom());
        List<LocusScore> scores = getSummaryScores(chr, (int) position - 10, (int) position + 10,
                zoom);

        // give a 2 pixel window, otherwise very narrow features will be missed.
        double bpPerPixel = IGVModel.getInstance().getViewContext().getScale();
        double minWidth = 2 * bpPerPixel;    /* * */

        if (scores == null) {
            return null;
        } else {
            return FeatureUtils.getFeatureAt(position, minWidth, scores);
        }
    }

    /**
     * Method description
     *
     *
     * @param chr
     * @param startLocation
     * @param endLocation
     * @param zoom
     *
     * @return
     */
    abstract public List<LocusScore> getSummaryScores(String chr, int startLocation,
            int endLocation, int zoom);

    /**
     * Method description
     *
     *
     * @param x
     * @param y
     *
     * @return
     */
    public boolean handleClick(int x, int y) {

        // Ignore
        return false;
    }

    /**
     * Method description
     *
     *
     * @param color
     */
    @Override
    public void setColor(Color color) {
        super.setColor(color);
        colorScale = null;
    }

    /**
     * Method description
     *
     *
     * @param color
     */
    @Override
    public void setAltColor(Color color) {
        super.setAltColor(color);
        colorScale = null;

    }

    /**
     * Method description
     *
     *
     * @param color
     */
    @Override
    public void setMidColor(Color color) {
        super.setMidColor(color);
        colorScale = null;

    }

    /**
     * Method description
     *
     *
     * @param type
     *
     * @return
     */
    public boolean isRegionScoreType(RegionScoreType type) {
        return ((getTrackType() == TrackType.GENE_EXPRESSION) && (type == RegionScoreType.EXPRESSION)) || (((getTrackType() == TrackType.COPY_NUMBER) || (getTrackType() == TrackType.ALLELE_SPECIFIC_COPY_NUMBER)) && ((type == RegionScoreType.AMPLIFICATION) || (type == RegionScoreType.DELETION))) || (type == RegionScoreType.SCORE);
    }
}
