/*
* The Broad Institute
* SOFTWARE COPYRIGHT NOTICE AGREEMENT
* This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
* of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
* (the "License"); you may not use this file except in compliance with
*  the License.  You may obtain a copy of the License at
*
*    http://www.opensource.org/licenses/gpl-2.0.php
*
* This software is supplied without any warranty or guaranteed support
* whatsoever. Neither the Broad Institute nor MIT can be responsible for its
* use, misuse, or functionality.
*/




package org.broad.igv.track;

//~--- non-JDK imports --------------------------------------------------------

import org.broad.igv.IGVConstants;

import org.broad.igv.data.DataSource;
import org.broad.igv.feature.LocusScore;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.h5.ObjectNotFoundException;
import org.broad.igv.renderer.DataRange;

//~--- JDK imports ------------------------------------------------------------

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;


/**
 *
 * @author jrobinso
 */
public class DataSourceTrack extends DataTrack {
    private DataSource dataSource;

    // private WindowFunction windowFunction = WindowFunction.median;

    /**
     * Constructs ...
     *
     *
     * @param locator
     * @param name
     * @param dataSource
     */
    public DataSourceTrack(ResourceLocator locator, String name, DataSource dataSource) {
        super(locator, name);

        this.dataSource = dataSource;
        setTrackType(dataSource.getTrackType());

        float min = (float) dataSource.getDataMin(IGVConstants.CHR_ALL);
        float max = (float) dataSource.getDataMax(IGVConstants.CHR_ALL);
        float baseline = 0;
        setDataRange(new DataRange(min, baseline, max));

    }

    /**
     * Method description
     *
     *
     * @param chr
     * @param startLocation
     * @param endLocation
     * @param zoom
     *
     * @return
     */
    public List<LocusScore> getSummaryScores(String chr, int startLocation, int endLocation,
            int zoom) {
        try
        {
            List<LocusScore> tmp = dataSource.getSummaryScoresForRange(chr, startLocation, endLocation, zoom);
            return tmp == null ? new ArrayList() : tmp;
        }
        catch (ObjectNotFoundException ex)
        {
            return new ArrayList();
        }
    }

    /**
     * Method description
     *
     *
     * @param statType
     */
    public void setStatType(WindowFunction statType) {

//      this.windowFunction = statType;
        dataSource.setWindowFunction(statType);

    }



    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isLogNormalized() {
        return dataSource.isLogNormalized();
    }

    /**
     * Method description
     *
     *
     * @param chr
     * @param start
     * @param end
     * @param zoom
     * @param type
     *
     * @return
     */
    public float getRegionScore(String chr, int start, int end, int zoom, RegionScoreType type) {

        if (end <= start)
        {
            return 0;
        }
        if (isRegionScoreType(type))
        {
            List<LocusScore> scores = getSummaryScores(chr, start, end, zoom);
            float regionScore = 0;
            int intervalSum = 0;
            for (LocusScore score : scores)
            {
                if ((score.getEnd() >= start) && (score.getStart() <= end))
                {
                    int interval = Math.min(end, score.getEnd())
                                   - Math.max(start, score.getStart());
                    float value = score.getScore();
                    regionScore += value * interval;
                    intervalSum += interval;
                }
            }
            if (intervalSum <= 0)
            {
                return -Float.MAX_VALUE;
            }
            else
            {
                regionScore /= intervalSum;
                return (type == RegionScoreType.DELETION) ? -regionScore : regionScore;
            }
        }
        else
        {
            return -Float.MAX_VALUE;
        }
    }

    /**
     * Method description
     *
     *
     * @param timestamp
     */
    @Override
    public void refreshData(long timestamp) {
        dataSource.refreshData(timestamp);
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public WindowFunction getWindowFunction() {
        return dataSource.getWindowFunction();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    @Override
    public Collection<WindowFunction> getAvailableWindowFunctions() {
        return dataSource.getAvailableWindowFunctions();
    }



}
