/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute 
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */

/*
 * AttributeManager.java
 *
 * Everything to do with attributes.
 */
package org.broad.igv.track;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import org.apache.log4j.Logger;
import org.broad.igv.feature.ParsingUtils;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.util.Utilities;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.util.AsciiLineReader;

/**
 *
 * @author jrobinso
 */
public class AttributeManager {

    private static Logger log = Logger.getLogger(AttributeManager.class);
    /** The set of currently loaded attribute resource files */
    Set<ResourceLocator> loadedResources = new HashSet();
    // Stores current file selections as String
    //private String currentAttributeFiles = null;
    final public static String ATTRIBUTES_LOADED_PROPERTY =
        "ATTRIBUTES_LOADED_PROPERTY";
    final public static String ATTRIBUTES_NARROWED_PROPERTY =
        "ATTRIBUTES_NARROWED_PROPERTY";
    // Property Change Support
    private PropertyChangeSupport propertyChangeSupport;
    /**
     * Map of data track identifiers (i.e. "array names") to its 
     * attributeMap.   The attributeMap for a track maps attribute name (for
     * example "Cell Type"  to value (for example "ES");
     */
    LinkedHashMap<String, Map<String, String>> attributeMap = new LinkedHashMap();
    /**
     * List of attribute names (keys).  The list
     * is kept so the keys may be fetched in the order they were added.
     */
    List<String> attributeKeys = new ArrayList();
    /**
     * The complete set of unique attribute values per attribute key.  This is useful in
     * assigning unique colors
     */
    Map<String, Set<String>> uniqueAttributeValues;
    static private AttributeManager singleton;

    private AttributeManager() {
        propertyChangeSupport = new PropertyChangeSupport(this);
        uniqueAttributeValues = new HashMap();

    }

    static synchronized public AttributeManager getInstance() {

        if (singleton == null)
        {
            singleton = new AttributeManager();
        }
        return singleton;
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        propertyChangeSupport.addPropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        propertyChangeSupport.removePropertyChangeListener(listener);
    }

    /**
     * Return the attribute value for the given track (trackIdentifier) and key.
     * In general this method should not be used,  use 
     *    Track.getAttributeValue(key) 
     * instead.
     */
    protected String getAttribute(String trackIdentifier, String attributeKey) {
        Map attributes = attributeMap.get(trackIdentifier);
        return (attributes == null ? null : (String) attributes.get(attributeKey));
    }

    /**
     * Return all attributes for a given track as a map
     * @param trackIdentifier
     * @return
     */
    protected Map<String, String> getAllAttributes(String trackIdentifier) {
        return attributeMap.get(trackIdentifier);
    }

    public void addAttributes(LinkedHashSet<Attribute> attributes) {

        try
        {
            for (Attribute attr : attributes)
            {
                setAttribute(attr.getTrackId(), attr.getKey(), attr.getValue());
            }
        } finally
        {

            IGVMainFrame.getInstance().packViews();

            // Repaint
            IGVMainFrame.getInstance().repaintDataAndHeaderPanels();
            IGVMainFrame.getInstance().repaintStatusAndZoomSlider();

            // Notify Listeners
            firePropertyChange(this, ATTRIBUTES_LOADED_PROPERTY, null, null);
        }
    }

    public void addAttributeKey(String key) {
        if (!attributeKeys.contains(key) && !key.startsWith("#"))
        {
            attributeKeys.add(key);
        }
    }

    /**
     * Set the attribute value for the given track and key.
     */
    private void setAttribute(String trackIdentifier, String attributeKey, String attributeValue) {

        addAttributeKey(attributeKey);

        Set<String> uniqueSet = uniqueAttributeValues.get(attributeKey);
        if (uniqueSet == null)
        {
            uniqueSet = new HashSet<String>();
            uniqueAttributeValues.put(attributeKey, uniqueSet);
        }
        uniqueSet.add(attributeValue);

        Map attributes = attributeMap.get(trackIdentifier);
        if (attributes == null)
        {
            attributes = new LinkedHashMap();
            attributeMap.put(trackIdentifier, attributes);
        }

        // attributeKey = column header, attributeValue = value for header
        // and track name (trackIdentifier) row intersection
        attributes.put(attributeKey, attributeValue);
    }

    public void clearAllAttributes() {
        attributeMap.clear();
        attributeKeys.clear();
        uniqueAttributeValues.clear();
        loadedResources = new HashSet();
    }

    /**
     *  Find track identifiers (arrayNames) that match the given attribute key/value
     *  combination
     */
    public Set<String> findTracksMatching(String key, String value) {
        HashSet<String> matchingTracks = new HashSet();
        for (Map.Entry<String, Map<String, String>> entry : attributeMap.entrySet())
        {
            String trackAttributeValue = entry.getValue().get(key);
            if (trackAttributeValue != null && trackAttributeValue.equals(value))
            {
                matchingTracks.add(entry.getKey());
            }
        }
        return matchingTracks;
    }

    /**
     * Return the list of attribute names (keys) in the order they should
     * be displayed.
     */
    public List<String> getAttributeKeys() {
        return attributeKeys;
    }

    /**
     * Load attributes from an ascii file in "SampleInfo" format.
     */
    public String loadAttributes(ResourceLocator locator) {
        AsciiLineReader reader = null;
        try
        {
            LinkedHashSet<Attribute> attributeList = new LinkedHashSet();
            reader = ParsingUtils.openAsciiReader(locator);
            String nextLine = reader.readLine();

            // Parse column neadings for attribute names.
            // Columns 1 and 2 are array and sample name (not attributes)
            String[] colHeadings = nextLine.split("\t");

            while ((nextLine = reader.readLine()) != null)
            {

                String[] values = nextLine.split("\t");

                if (values.length >= 2)
                {
                    String arrayName = values[0].trim();
                    // Loop through attribute columns
                    for (int i = 1; i < Math.min(values.length, colHeadings.length); i++)
                    {
                        String attributeName = colHeadings[i].trim();
                        String attributeValue = values[i].trim();
                        if (attributeValue.length() > 0)
                        {
                            attributeList.add(new Attribute(arrayName,
                                attributeName, attributeValue));
                        }
                    }

                }
            }

            if (attributeList.isEmpty())
            {
                StringBuffer invalidAttributeFileMessageText = new StringBuffer();
                invalidAttributeFileMessageText.append("The following files do not " +
                    "contains valid attribute data and could not be loaded:\n");
                invalidAttributeFileMessageText.append("\n\t");
                invalidAttributeFileMessageText.append(locator.getPath());
                return invalidAttributeFileMessageText.toString();

            }


            addAttributes(attributeList);
            reader.close();

            loadedResources.add(locator);

            //createCurrentAttributeFileString(files);
            TrackManager.getInstance().resetOverlayTracks();

            // If we get here everything went o.k.  This is signaled with a null
            // return value (this convention is a bit odd).
            return null;

        } catch (FileNotFoundException ex)
        {
            log.error("Error loading attribute file", ex);
            return ("Error:  Attribute file not found: " + locator.getPath());
        } catch (IOException ex)
        {
            log.error("IO Exception loading attribute file", ex);
            return ("Error reading attribute file: " + locator.getPath());
        } finally
        {
            if (reader != null)
            {
                reader.close();

            }
            firePropertyChange(this, ATTRIBUTES_LOADED_PROPERTY, null, null);
        }
    }

    public void firePropertyChange(Object source, String propertyName,
                                    Object oldValue, Object newValue) {

        PropertyChangeEvent event =
            new PropertyChangeEvent(
            source,
            propertyName,
            oldValue,
            newValue);
        propertyChangeSupport.firePropertyChange(event);
    }

    public Comparator getAttributeComparator() {
        return Utilities.getNumericStringComparator();
    }

    /**
     * 
     * @return set of curently loaded resources
     */
    public Set<ResourceLocator> getLoadedResources() {
        return loadedResources;
    }
}
