/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.track;

//~--- non-JDK imports --------------------------------------------------------
import org.apache.log4j.Logger;

import org.broad.igv.IGVConstants;


import org.broad.igv.PreferenceManager;
import org.broad.igv.feature.LocusScore;
import org.broad.igv.renderer.ColorScale;
import org.broad.igv.renderer.ContinuousColorScale;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.renderer.BarChartRenderer;
import org.broad.igv.renderer.DataRange;
import org.broad.igv.renderer.HeatmapRenderer;
import org.broad.igv.renderer.Renderer;
import org.broad.igv.renderer.RendererFactory;
import org.broad.igv.renderer.XYPlotRenderer;
import org.broad.igv.ui.IGVModel;
import org.broad.igv.ui.ViewContext;

import org.broad.igv.util.ColorUtilities;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.Rectangle;

import java.awt.event.MouseEvent;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.broad.igv.feature.FeatureUtils;
import org.broad.igv.renderer.GraphicUtils;
import org.broad.igv.ui.FontManager;

/**
 *
 * @author jrobinso
 */
public abstract class AbstractTrack implements Track {

    /** Field description */
    public static final String COLOR_ATTR_KEY = "#COLOR";
    /** Field description */
    public static final String ALT_COLOR_ATTR_KEY = "#ALTCOLOR";
    /** Field description */
    public static final String DATA_MID_ATTR_KEY = "#MID";
    /** Field description */
    public static final String DATA_MAX_ATTR_KEY = "#MAX";
    /** Field description */
    public static final String DATA_MIN_ATTR_KEY = "#MIN";
    /** Field description */
    public static final String HEIGHT_ATTR_KEY = "#HEIGHT";
    /** Field description */
    public static final String RENDERER_ATTR_KEY = "#GRAPHTYPE";
    private static Logger log = Logger.getLogger(AbstractTrack.class);
    /**
     * User supplied track name.  For data this is usually the sample name, or column
     * heading from an input file.
     */
    private String id;
    private String name;
    /**
     * Name for display.  Can be null, in which case "name" is used.
     */
    private String displayName;
    /**
     * The source file from which this track was created.  Defaults to an
     * empty string, which indicates that the track was not derived from a
     * data file.
     */
    private String sourceFile = "";
    /**
     * Top (Y) position
     */
    private int top;
    /**
     * Minimum allowable height for this track
     */
    private int minimumHeight = 1;
    /**
     * The file from which this track originated.
     */
    private ResourceLocator resourceLocator;
    private TrackType trackType = TrackType.OTHER;
    private static Class defaultRendererClass = BarChartRenderer.class;
    private static Map<TrackType, Class> defaultRendererMap = new HashMap();
    private Color posColor;
    private Color altColor;
    private Color midColor = Color.WHITE;
    /**
     * Continuous posColor scale for this track
     * @return
     */

    // private int defaultPreferredHeight =
    // PreferenceManager.getInstance().getDefaultTrackHeight();
    private boolean isSelected = false;
    private boolean visible = true;
    boolean overlayVisible;
    private int preferredHeight;
    private boolean isDraggable = true;
    /**
     * Map to store attributes specific to this track.  Attributes shared by multiple
     * tracks are stored in AttributeManager.
     */
    private Map<String, String> attributes = new HashMap();
    protected DataRange dataRange;

    /**
     * Set default renderer classes by track type.
     */
    static
    {
        defaultRendererMap.put(TrackType.RNAI, HeatmapRenderer.class);
        defaultRendererMap.put(TrackType.COPY_NUMBER, HeatmapRenderer.class);
        defaultRendererMap.put(TrackType.ALLELE_SPECIFIC_COPY_NUMBER, HeatmapRenderer.class);
        defaultRendererMap.put(TrackType.GENE_EXPRESSION, HeatmapRenderer.class);
        defaultRendererMap.put(TrackType.DNA_METHYLATION, HeatmapRenderer.class);
        defaultRendererMap.put(TrackType.LOH, HeatmapRenderer.class);
        defaultRendererMap.put(TrackType.OTHER, BarChartRenderer.class);
        defaultRendererMap.put(TrackType.CHIP_CHIP, HeatmapRenderer.class);
    }
    ContinuousColorScale colorScale;

    /**
     * Constructs ...
     *
     *
     * @param dataResourceLocator
     * @param name
     */
    public AbstractTrack(ResourceLocator dataResourceLocator, String name) {
        this.overlayVisible = PreferenceManager.getInstance().getDiplayOverlayTracks();
        this.resourceLocator = dataResourceLocator;
        this.name = name;
        this.id = String.valueOf(name == null ?  dataResourceLocator.getPath() : name);
    }

    /**
     * Ignored by default.  Subclases should override as required
     */
    public void preloadData(String chr, int start, int end, int zoom) {
    }

    /**
     * Update track properties from managed attributes.  This method should be called when
     * attributes are loaded.
     *
     */
    public void updateProperties() {
        Map<String, String> props = AttributeManager.getInstance().getAllAttributes(id);
        if (props != null)
        {
            try
            {
                String colorString = props.get(COLOR_ATTR_KEY);
                if (colorString != null)
                {
                    Color trackColor = this.getColorAttribute(COLOR_ATTR_KEY, null);
                    if (trackColor != null)
                    {
                        setColor(trackColor);
                    }
                }
                String altColorString = props.get(ALT_COLOR_ATTR_KEY);
                if (colorString != null)
                {
                    Color altColor = this.getColorAttribute(ALT_COLOR_ATTR_KEY, null);
                    if (altColor != null)
                    {
                        setAltColor(altColor);
                    }
                }

                String heightString = props.get(HEIGHT_ATTR_KEY);
                if (heightString != null)
                {
                    try
                    {
                        int h = Integer.parseInt(heightString);
                        if (h > 0)
                        {
                            setHeight(h);
                        }

                    } catch (NumberFormatException numberFormatException)
                    {
                        log.info("Unrecognized integer string: " + heightString);
                    }
                }
                String rendererString = props.get(RENDERER_ATTR_KEY);
                if (rendererString != null)
                {
                    Class rendererClass = RendererFactory.getRendererClass(rendererString.toUpperCase());
                    if (rendererClass != null)
                    {
                        setRendererClass(rendererClass);
                    }
                }
                String minString = props.get(DATA_MIN_ATTR_KEY);
                String baseString = props.get(DATA_MID_ATTR_KEY);
                String maxString = props.get(DATA_MAX_ATTR_KEY);
                if (maxString != null)
                {
                    try
                    {
                        float min = (minString == null) ? 0 : Float.parseFloat(minString);
                        float max = Float.parseFloat(maxString);
                        float base = min;
                        if (baseString != null)
                        {
                            base = Float.parseFloat(baseString);
                        }
                        setDataRange(new DataRange(min, base, max));
                    } catch (NumberFormatException e)
                    {
                        log.error("Error parsing float for axis definition. ", e);
                    }

                }

            } catch (Exception e)
            {
                log.error("Error updating track properties: ", e);
            }
        }
    }

    /**
     * For now the unique id for the track is its name.  This <may> not be unique,
     * but we have nothing better.
     * @return
     */
    public String getId() {
        return id;
    }

    /**
     * Method description
     *
     *
     * @param displayName
     */
    public void setDisplayName(String displayName) {
        this.displayName = displayName;
    }

    // TODO Provided for session saving.  Rename/refactor later
    public String getActualDisplayName() {
        return displayName;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getDisplayName() {

        if (displayName != null)
        {
            return displayName;
        }

        String sampleName = PreferenceManager.getInstance().getTrackAttributeName();
        if (sampleName != null)
        {
            sampleName = getAttributeValue(sampleName);
        }

        return ((sampleName == null) ? name : sampleName);
    }

    /**
     * Render a border.
     * @param context
     * @param rect
     */
    public void renderBorder(RenderContext context, Rectangle rect) {
        Renderer renderer = this.getRenderer();
        if (renderer != null)
        {
            this.getRenderer().renderBorder(this, context, rect);
        }
    }

    /**
     * Render a Y axis
     * @param context
     * @param rect
     */
    public void renderAxis(RenderContext context, Rectangle rect) {
        Renderer renderer = this.getRenderer();
        if (renderer != null)
        {
            this.getRenderer().renderAxis(this, context, rect);
        }
    }

    public void renderName(Graphics2D graphics, Rectangle rect, Rectangle visibleRect) {

        if (isSelected())
        {
            graphics.setBackground(Color.LIGHT_GRAY);
        } else
        {
            graphics.setBackground(Color.WHITE);
        }

        String trackName = getDisplayName();
        if ((trackName != null) && (rect.getHeight() > 1))
        {

            Graphics2D g2D = (Graphics2D) graphics.create();
            g2D.clearRect(rect.x, rect.y, rect.width, rect.height);

            // Calculate fontsize
            int fontSize = g2D.getFont().getSize();
            fontSize = (int) (fontSize * (rect.getHeight() / 20.0));
            fontSize = (fontSize > 12) ? 12 : fontSize;

            Font font = FontManager.getScalableFont(fontSize);
            g2D.setFont(font);
            FontManager.applyScalableTextHints(g2D);

            FontMetrics fontMetrics = g2D.getFontMetrics();

            trackName = addEllipseToString(trackName, rect.width, fontMetrics, g2D);
            GraphicUtils.drawVerticallyCenteredText(trackName, 5, rect, g2D, false);

            g2D.dispose();
        }

    }
    final private StringBuffer buffer = new StringBuffer();

    final private String addEllipseToString(String text, int panelWidth, FontMetrics fontMetrics,
                                              Graphics2D gc) {

        int textLength = text.length();
        Rectangle2D stringBounds = fontMetrics.getStringBounds(text, gc);
        double textWidth = stringBounds.getWidth() + 10;

        // If text is wider than the panel - add ellipses
        if ((textWidth > 0) && (textWidth > panelWidth))
        {

            // This is the percentage (of original text) to make new text
            double reductionPercentage = panelWidth / (textWidth);

            int expectedTextLength = (int) (textLength * reductionPercentage);

            int leftEnd = (expectedTextLength / 2) - 2;    // Remove 2 more characters
            int rightStart = (textLength - leftEnd);


            buffer.delete(0, buffer.length());
            buffer.append(text.substring(0, leftEnd));
            buffer.append("...");
            buffer.append(text.substring(rightStart, textLength));
            return buffer.toString();
        }
        return text;
    }

    /**
     * Called to overlay a track on another, presumably previously rendered,
     * track. The default behavior is to do nothing.
     *
     * @param context
     * @param rect
     */
    public void overlay(RenderContext context, Rectangle rect) {
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Color getColor() {
        if (posColor != null)
        {
            return posColor;
        }
        return getRenderer() == null ? null : getRenderer().getDefaultColor();

    }

    private Color getColorAttribute(String key, Color defaultValue) {
        String rgb = this.getAttributeValue(key);
        if (rgb != null)
        {
            try
            {
                return ColorUtilities.convertRGBStringToColor(rgb.replace("\"", ""));

            } catch (Exception exception)
            {
                log.info("Invalid color string " + rgb + " for track: " + getId());
            }
        }
        return defaultValue;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Color getAltColor() {
        return altColor;

    }

    /**
     * Method description
     *
     *
     * @return
     */
    public ResourceLocator getDataResourceLocator() {
        return resourceLocator;
    }

    /**
     * Add an attribute to this track and register the key with the attribute panel.
     *
     *   Note:  Attribute keys are case insensitive.  Currently this is implemented
     *          by forcing all keys to upper case
     * @param key
     * @param value
     */
    public void setAttributeValue(String key, String value) {
        String uppercaseKey = key.toUpperCase();
        attributes.put(uppercaseKey, value);
        AttributeManager.getInstance().addAttributeKey(uppercaseKey);
    }

    /**
     * Method description
     *
     *
     * @param attributeKey
     *
     * @return
     */
    public String getAttributeValue(String attributeKey) {
        String value = attributes.get(attributeKey);
        if (value == null)
        {
            value = AttributeManager.getInstance().getAttribute(getId(), attributeKey);
        }
        return value;
    }

    /**
     * Method description
     *
     *
     * @param y
     *
     * @return
     */
    public int getLevelNumber(int y) {
        return (preferredHeight == 0) ? 0 : y / preferredHeight;
    }

    /**
     * Returns the default height based on the default renderer for the data
     * type, as opposed to the actual renderer in use.  This is done to prevent
     * the track size from changing if renderer is changed.
     * @return
     */
    private int getDefaultHeight() {
        if (XYPlotRenderer.class.isAssignableFrom(getDefaultRendererClass()))
        {
            return PreferenceManager.getInstance().getDefaultChartTrackHeight();
        } else
        {
            return PreferenceManager.getInstance().getDefaultTrackHeight();
        }
    }

    /**
     * Method description
     *
     *
     * @param minimumHeight
     */
    public void setMinimumHeight(int minimumHeight) {
        this.minimumHeight = minimumHeight;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getMinimumHeight() {
        return minimumHeight;
    }

    /**
     * Method description
     *
     *
     * @param type
     */
    public void setTrackType(TrackType type) {
        this.trackType = type;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public TrackType getTrackType() {
        return trackType;
    }

    protected ViewContext getViewContext() {
        return IGVModel.getInstance().getViewContext();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isVisible() {
        return visible && ((getTrackType() != IGVConstants.overlayTrackType) || (this.overlayVisible == true));
    }

    /**
     * Method description
     *
     *
     *
     * @param color
     */
    public void setColor(Color color) {
        this.posColor = color;
    }

    /**
     * Method description
     *
     *
     *
     * @param color
     */
    public void setAltColor(Color color) {
        altColor = color;
    }

    /**
     * Method description
     *
     *
     *
     * @param isVisible
     */
    public void setVisible(boolean isVisible) {
        this.visible = isVisible;
    }

    /**
     * Method description
     *
     *
     * @param bool
     */
    public void setOverlayVisible(boolean bool) {
        this.overlayVisible = bool;
    }

    /**
     * Method description
     *
     * <<<<<<< .working
     * =======
     */
    public void updateState() {
    }

    /**
     * Method description
     *
     * >>>>>>> .merge-right.r3291
     *
     * @param selected
     */
    public void setSelected(boolean selected) {
        isSelected = selected;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isSelected() {
        return isSelected;
    }

    /**
     * Method description
     *
     *
     * @param value
     */
    public void setIsDraggable(boolean value) {
        isDraggable = value;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isDraggable() {
        return isDraggable;
    }
    int height = -1;

    /**
     * Method description
     *
     *
     * @param preferredHeight
     */
    public void setHeight(int preferredHeight) {
        this.height = preferredHeight;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getHeight() {
        return (height < 0) ? getDefaultHeight() : height;
    }

    public int getPreferredHeight() {
        return getHeight();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    /**
     * Method description
     *
     *
     * @return
     */
    public DataRange getAxisDefinition() {
        if (dataRange == null)
        {
            String chr = IGVModel.getInstance().getViewContext().getChrName();
            float min = (float) 0;
            float max = (float) 10;
            float baseline = 0;

            setDataRange(new DataRange(min, baseline, max));
        }
        return dataRange;
    }

    /**
     * Method description
     *
     *
     * @param axisDefinition
     */
    public void setDataRange(DataRange axisDefinition) {
        this.dataRange = axisDefinition;
    }

    protected Class getDefaultRendererClass() {
        Class def = defaultRendererMap.get(getTrackType());
        return (def == null) ? defaultRendererClass : def;
    }

    // Assumes features are sorted by start position
    protected LocusScore getFeatureAt(double position, double minWidth,
                                       List<? extends LocusScore> features) {

        return FeatureUtils.getFeatureAt(position, minWidth, features);
    }



    /**
     * Refresh the underlying data for the track.  Default implementation does nothing.
     * Subclasses can override
     * @param timestamp
     */
    public void refreshData(long timestamp) {
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public String getSourceFile() {
        return sourceFile;
    }

    /**
     * Method description
     *
     *
     * @param sourceFile
     */
    public void setSourceFile(String sourceFile) {
        this.sourceFile = sourceFile;
    }
    boolean expanded = false;

    /**
     * Method description
     *
     *
     * @param expanded
     */
    public void setExpanded(boolean expanded) {
        this.expanded = expanded;
    }

    /**
     * @return
     */
    public boolean isExpanded() {
        return expanded;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Collection<WindowFunction> getAvailableWindowFunctions() {
        return new ArrayList();

    }

    /**
     * Method description
     *
     *
     * @return
     */
    public Color getMidColor() {
        return midColor;
    }

    /**
     * Method description
     *
     *
     * @param midColor
     */
    public void setMidColor(Color midColor) {
        this.midColor = midColor;
    }

    public boolean handleClick(MouseEvent e) {
        return false;
    }

    /**
     * Method description
     *
     *
     * @param trackProperties
     */
    public void setTrackProperties(TrackProperties trackProperties) {

        if (!trackProperties.isAutoScale())
        {
            DataRange dr = new DataRange(trackProperties.getMinValue(),
                trackProperties.getMidValue(), trackProperties.getMaxValue());
            setDataRange(dr);
        }
        if (trackProperties.getColor() != null)
        {
            setColor(trackProperties.getColor());
        }
        if (trackProperties.getAltColor() != null)
        {
            setAltColor(trackProperties.getAltColor());
        }
        if (trackProperties.getMidColor() != null)
        {
            setMidColor(trackProperties.getMidColor());
        }
        if (trackProperties.getHeight() > 0)
        {
            setHeight(trackProperties.getHeight());
        }
        if (trackProperties.getMinHeight() > 0)
        {
            setMinimumHeight(trackProperties.getMinHeight());
        }
        if (trackProperties.getRendererClass() != null)
        {
            setRendererClass(trackProperties.getRendererClass());
        }
        if (trackProperties.getWindowingFunction() != null)
        {
            setStatType(trackProperties.getWindowingFunction());
        }
    }

    /**
     * @return the top
     */
    public int getTop() {
        return top;
    }

    /**
     * @param top the top to set
     */
    public void setTop(int top) {
        this.top = top;
    }

    /**
     * Return the color scale for this track.  If a specific scale exists for this data type
     * use that.  Otherwise create one using the track color and data range.
     *
     *
     * @return
     */
    public ContinuousColorScale getColorScale() {
        ColorScale typeColorScale = PreferenceManager.getInstance().getColorScale(getTrackType());
        if ((typeColorScale != null) && (typeColorScale instanceof ContinuousColorScale)) {
            return (ContinuousColorScale) typeColorScale;
        }
        if (colorScale == null) {
            DataRange ad = getAxisDefinition();
            if (getAltColor() == null) {
                colorScale = new ContinuousColorScale(ad.getMinimum(), ad.getMaximum(),
                        getMidColor(), getColor());
            } else {
                colorScale = new ContinuousColorScale(ad.getMinimum(), ad.getBaseline(),
                        ad.getMaximum(), getAltColor(), getMidColor(), getColor());
            }
            ((ContinuousColorScale) colorScale).setNoDataColor(IGVConstants.NO_DATA_COLOR);
        }
        return colorScale;
    }
}
