/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.track;

import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import org.apache.log4j.Logger;
import org.broad.igv.IGVConstants;
import org.broad.igv.feature.BasicFeature;
import org.broad.igv.feature.Feature;
import org.broad.igv.feature.Genome;
import org.broad.igv.renderer.BasicFeatureRenderer;
import org.broad.igv.renderer.FeatureRenderer;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.IGVModel;
import org.broad.igv.util.ResourceLocator;

/**
 *
 * @author jrobinso
 */
public abstract class AbstractFeatureTrack extends AbstractTrack {

    private static Logger log = Logger.getLogger(AbstractFeatureTrack.class);
    Rectangle expCollapse;
    /**
     * Allocate features to multiple "levels" for expanded mode.  In this mode features at each
     * level do not overlap.
     *
     * Assumption:  features are sorted by increasing start location.
     *
     * @param features
     * @return
     */
    // max # of levels
    static int maxLevels = 200;
    boolean expanded;
    List<Rectangle> featureRects = new ArrayList();
    private String lastChromosomeName;
    int levelHeight = 25;
    private List<List<Feature>> levelList;
    /**
     * A hack until tracks can be refactored.
     */
    private boolean mutationTrack = false;
    private FeatureRenderer renderer;

    public AbstractFeatureTrack(ResourceLocator locator, String name) {
        super(locator, name);
    }

    private void calculateLevels(List<Feature> features) {

        int currentLevelCount = this.getNumberOfFeatureLevels();
        levelList = new ArrayList<List<Feature>>();
        if (features.isEmpty())
        {
            return;
        }
        LinkedHashSet<Feature> featuresToAllocate = new LinkedHashSet();
        featuresToAllocate.addAll(features);
        while (!featuresToAllocate.isEmpty() && levelList.size() < maxLevels)
        {
            List<Feature> nextLevel = new ArrayList();
            levelList.add(nextLevel);
            int lastEnd = -1;
            for (Feature feature : featuresToAllocate)
            {
                if (feature.getStart() > lastEnd)
                {
                    nextLevel.add(feature);
                    lastEnd = feature.getEnd();
                }
            }
            if (nextLevel.isEmpty())
            {
                System.out.println("");
            }
            featuresToAllocate.removeAll(nextLevel);
        }

        // This dependency on IGVMainFrame here is unfortunate.  TODO -- refactor ths to
        // use event/listener ?
        if (levelList.size() != currentLevelCount)
        {
            IGVMainFrame.getInstance().doRefresh();
        }
    }

    /**
     * @deprecated
     * @return
     */
    public double getDataMax() {
        return 0;
    }

    public List<Feature> getFeatures(String chr, int startLocation, int endLocation) {
        return getFeatures(chr);
    }

    public synchronized List<List<Feature>> getFeaturesByLevels(String chr, int start, int end) {
        if ((levelList == null) || (lastChromosomeName == null) || !lastChromosomeName.equals(chr))
        {
            lastChromosomeName = chr;
            if (!isMultiLevel())
            {
                levelList = new ArrayList();
                List<Feature> features = getFeatures(chr, start, end);
                if (features != null)
                {
                    levelList.add(features);
                }
            } else
            {

                List<Feature> features = getFeatures(chr, start, end);
                if (features != null)
                {
                    calculateLevels(features);
                }
            }
        }
        return levelList;
    }

    @Override
    public int getHeight() {

        if (false == isVisible())
        {
            return 0;
        }

        int levelCount = 1;
        if ((levelList != null) && !levelList.isEmpty())
        {
            levelCount = levelList.size();
        }

        return super.getHeight() * levelCount;
    }

    public double getMedian(String stat) {
        return 1.0;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public int getNumberOfFeatureLevels() {

        if ((levelList == null) || levelList.isEmpty())
        {
            return 1;
        } else
        {
            return levelList.size();
        }
    }

    public float getRegionScore(String chr, int start, int end, int zoom, RegionScoreType type) {

        // Dummy implementation
        return -Float.MAX_VALUE;
    }

    public FeatureRenderer getRenderer() {
        if (renderer == null)
        {
            setRendererClass(BasicFeatureRenderer.class);
        }
        return renderer;
    }

    public String getValueStringAt(String chr, double position, int y) {

        // Determine the level number (for expanded tracks.
        int levelNumber = 0;
        if (isMultiLevel() && (this.featureRects != null))
        {
            for (int i = 0; i < featureRects.size(); i++)
            {
                Rectangle r = featureRects.get(i);
                if ((y >= r.y) && (y <= r.getMaxY()))
                {
                    levelNumber = i;
                    break;
                }
            }
        }

        int nLevels = this.getNumberOfFeatureLevels();
        List<Feature> features = null;
        if ((nLevels > 1) && (levelNumber < nLevels))
        {
            features = this.getFeaturesByLevels(chr, (int) position - 10, (int) position + 10).get(levelNumber);
        } else
        {
            features = getFeatures(chr, (int) position - 10, (int) position + 10);
        }
        if(features == null) {
            return null;
        }

        // give a 2 pixel window, otherwise very narrow features will be missed.
        double bpPerPixel = IGVModel.getInstance().getViewContext().getScale();
        double minWidth = 2 * bpPerPixel;
        Feature feature = (Feature) getFeatureAt(position, minWidth, features);
        return (feature == null) ? null : feature.getValueString(position, getWindowFunction());
    }

    public WindowFunction getWindowFunction() {
        return WindowFunction.count;
    }

    /**
     * Method description
     *
     *
     * @param x
     * @param y
     *
     * @return
     */
    public boolean handleClick(int x, int y) {

        /*
         * if (expCollapse.contains(x, y))
         * {
         *   setExpanded(!isExpanded());
         *   IGVMainFrame.getInstance().repaintDataPanels();
         *   return true;
         * }
         */
        return false;
    }

    @Override
    public boolean isExpanded() {
        return expanded;
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isLogNormalized() {
        return true;
    }

    /**
     * Return true if the track has multiple levels.  By default this is synonomous with an expanded
     * state.  Subclasses that have multiple levels in the collapsed state can override.
     *
     * @return
     */
    public boolean isMultiLevel() {
        return isExpanded();
    }

    /**
     * Method description
     *
     *
     * @return
     */
    public boolean isMutationTrack() {
        return mutationTrack;
    }

    public void overlay(RenderContext context, Rectangle rect) {
        getRenderer().setOverlayMode(true);
        renderFeatures(context, rect);
    }

    abstract public List<Feature> getFeatures(String chr);

    abstract public void setFeatures(String chr, List<Feature> features);

    private void recalculateLevels() {
        String chr = IGVModel.getInstance().getViewContext().getChrName();
        List<Feature> tmp = this.getFeatures(chr);
        if (tmp != null)
        {
            calculateLevels(tmp);
        }
    }

    public void render(RenderContext context, Rectangle rect) {
        getRenderer().setOverlayMode(false);
        Rectangle renderRect = new Rectangle(rect);
        int margin = Math.min(10, (int) (rect.height / 5.0));
        renderRect.y = renderRect.y + margin;
        renderFeatures(context, renderRect);
    }

    public void renderFeatures(RenderContext context, Rectangle inputRect) {
        try
        {
            List<List<Feature>> featureLists = getFeaturesByLevels(context.getChr(), (int) context.getStartLocation(), (int) context.getEndLocation());

            if (featureLists.isEmpty())
            {
                return;
            }

            int nLevels = featureLists.size();
            synchronized (featureRects)
            {

                featureRects.clear();


                // Divide rectangle into equal height levels
                double y = inputRect.getY();
                double h = inputRect.getHeight() / nLevels;
                int levelNumber = 0;
                for (List<Feature> features : featureLists)
                {
                    Rectangle rect = new Rectangle(inputRect.x, (int) y, inputRect.width, (int) h);
                    featureRects.add(rect);
                    getRenderer().renderFeatures(features, context, rect, this);
                    y += h;
                    levelNumber++;
                }
            }
        } catch (Exception ex)
        {
            log.error("Error rendering track", ex);
            throw new RuntimeException("Error rendering track ", ex);
        }
    }

    protected void sampleGenomeFeatures() {
        List<Feature> chrAllFeatures = new ArrayList(1000);
        Genome currentGenome = IGVModel.getInstance().getViewContext().getGenome();
        int sampleLength = (int) ((double) currentGenome.getLength() / (1000 * 700));
        int lastFeaturePosition = -1;
        for (String chr : currentGenome.getChromosomeNames())
        {
            List<Feature> features = getFeatures(chr);
            if (features != null)
            {
                long offset = currentGenome.getCumulativeOffset(chr);
                for (Feature f : features)
                {
                    int genStart = (int) ((offset + f.getStart()) / 1000);
                    int genEnd = (int) ((offset + f.getEnd()) / 1000);
                    if (genEnd > lastFeaturePosition + sampleLength)
                    {
                        BasicFeature f2 = new BasicFeature(IGVConstants.CHR_ALL, genStart, genEnd, f.getStrand());
                        f2.setName(f.getName());
                        chrAllFeatures.add(f2);

                        lastFeaturePosition = genEnd;
                    }
                }
            }
        }

        setFeatures(IGVConstants.CHR_ALL, chrAllFeatures);
    }

    public void setChromosome(String chr) {
    }

    protected void resetLevelList() {
        levelList = null;
    }

    /**
     * Method description
     *
     *
     * @param max
     */
    public void setDataMax(float max) {
    }

    @Override
    public void setExpanded(boolean value) {
        if (value != expanded)
        {
            expanded = value;
            if (expanded)
            {
                recalculateLevels();
            } else
            {
                levelList = null;
            }
        }
    }

    @Override
    public void setHeight(int newHeight) {

        int levelCount = 1;
        if ((levelList != null) && !levelList.isEmpty())
        {
            levelCount = levelList.size();
        }

        super.setHeight(Math.max(getMinimumHeight(), newHeight / levelCount));
    }

    /**
     * Method description
     *
     *
     * @param mutationTrack
     */
    public void setMutationTrack(boolean mutationTrack) {
        this.mutationTrack = mutationTrack;
    }

    public void setRendererClass(Class rc) {
        try
        {
            renderer = (FeatureRenderer) rc.newInstance();
        } catch (Exception ex)
        {
            log.error("Error instatiating renderer ", ex);
        }
    }

    public void setStatType(WindowFunction type) {
    }

    /**
     * Method description
     *
     *
     * @param zoom
     */
    public void setZoom(int zoom) {
    }
}

