package org.broad.igv.tools.sort;

import net.sf.samtools.util.SortingCollection;
import net.sf.samtools.util.AsciiLineReader;
import net.sf.samtools.util.CloseableIterator;

import java.io.*;
import java.util.Comparator;

import org.broad.igv.tools.DotAlignedSorter;

/**
 * Created by IntelliJ IDEA.
 * User: nazaire
 * Date: Jun 2, 2009
 */
public class Sorter
{
    static int MAX_RECORDS_IN_RAM = 1000000;
    static public File inputFile;
    static public File outputFile;
    static final String usageString = "Sort <inputFile> [outputFile]";

    /** Standard main method. */
    public static void main(String[] argv) {

        (new Sorter()).run(argv);
    }

    public int run(String[] argv) {

        if (argv.length == 0) {
            System.out.println(usageString);
            System.exit(-1);
        }

        inputFile = new File(argv[0]);
        if (!inputFile.exists()) {
            System.out.println("Error: " + inputFile.getAbsolutePath() + " does not exist.  Exiting");
            return -1;
        }

        if (argv.length > 1) {
            outputFile = new File(argv[1]);
        } else {
            outputFile = inputFile.isDirectory() ? inputFile : new File(inputFile.getParentFile(), inputFile.getName().replace(".aligned", ".sorted.aligned"));
        }


        if (inputFile.isDirectory()) {
            for (File file : inputFile.listFiles()) {
                String nm = file.getName();
                if (nm.endsWith(".aligned")) {
                    File of = new File(outputFile, nm.replace(".aligned", ".sorted.aligned"));
                    sort(file, of, 1);
                }
            }
        } else {
            sort(inputFile, outputFile, 1);
        }

        return 0;
    }

    public void sort(File inputFile, File outputFile)
    {
        sort(inputFile, outputFile, 0);
    }

    public void sort(File inputFile, File outputFile, int numHeaderLines) {

        FileInputStream fis = null;
        PrintWriter writer = null;

        try {
            fis = new FileInputStream(inputFile);
            writer = new PrintWriter(new BufferedWriter(new FileWriter(outputFile)));

            SortableRecordCodec codec = new SortableRecordCodec();

            Comparator<SortableRecord> comp = new Comparator<SortableRecord>() {

                public int compare(SortableRecord o1, SortableRecord o2) {

                    int t1 = o1.getChromosome().compareTo(o2.getChromosome());

                    if (t1 != 0) {
                        return t1;
                    }
                    return (int) (o1.getStart() - o2.getStart());
                }
            };


            SortingCollection cltn = SortingCollection.newInstance(SortableRecord.class,
                    codec, comp, MAX_RECORDS_IN_RAM);

            Parser parser = new Parser();
            AsciiLineReader reader = new AsciiLineReader(fis);
            SortableRecord next = null;

            //skip header
            String line = null;
            while(numHeaderLines != 0 && numHeaderLines != -1 && (line = reader.readLine()) != null)
            {
                writer.println(line);
                numHeaderLines--;
            }

            while ((next = parser.readNextRecord(reader)) != null) {
                cltn.add(next);
            }


            CloseableIterator<SortableRecord> iter = cltn.iterator();
            while (iter.hasNext()) {
                SortableRecord al = iter.next();
                writer.println(al.getText() + "\t");

            }
            iter.close();
        } catch (IOException ex) {
            ex.printStackTrace();
        } finally {
            try {
                fis.close();
                writer.close();
            } catch (IOException ex) {
            }
        }
    }
}
