/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.tools;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Comparator;
import net.sf.samtools.util.AsciiLineReader;
import net.sf.samtools.util.CloseableIterator;
import net.sf.samtools.util.SortingCollection;
import org.broad.igv.sam.DotAlignedAlignment;
import org.broad.igv.sam.reader.DotAlignedCodec;
import org.broad.igv.sam.reader.DotAlignedParser;

/**
 *
 * @author jrobinso
 */
public class DotAlignedSorter {

    static int MAX_RECORDS_IN_RAM = 1000000;
    static public File inputFile;
    static public File outputFile;
    static final String usageString = "SortDotAligned <inputFile> [outputFile]";

    /** Standard main method. */
    public static void main(String[] argv) {

        (new DotAlignedSorter()).run(argv);
    }

    public int run(String[] argv) {

        if (argv.length == 0) {
            System.out.println(usageString);
            System.exit(-1);
        }

        inputFile = new File(argv[0]);
        if (!inputFile.exists()) {
            System.out.println("Error: " + inputFile.getAbsolutePath() + " does not exist.  Exiting");
            return -1;
        }

        if (argv.length > 1) {
            outputFile = new File(argv[1]);
        } else {
            outputFile = inputFile.isDirectory() ? inputFile : new File(inputFile.getParentFile(), inputFile.getName().replace(".aligned", ".sorted.aligned"));
        }


        if (inputFile.isDirectory()) {
            for (File file : inputFile.listFiles()) {
                String nm = file.getName();
                if (nm.endsWith(".aligned")) {
                    File of = new File(outputFile, nm.replace(".aligned", ".sorted.aligned"));
                    sort(file, of);
                }
            }
        } else {
            sort(inputFile, outputFile);
        }

        return 0;
    }

    public void sort(File inputFile, File outputFile) {

        FileInputStream fis = null;
        PrintWriter writer = null;

        try {
            fis = new FileInputStream(inputFile);
            writer = new PrintWriter(new BufferedWriter(new FileWriter(outputFile)));

            DotAlignedCodec codec = new DotAlignedCodec();

            Comparator<DotAlignedAlignment> comp = new Comparator<DotAlignedAlignment>() {

                public int compare(DotAlignedAlignment o1, DotAlignedAlignment o2) {

                    int t1 = o1.getChromosome().compareTo(o2.getChromosome());
                    if (t1 != 0) {
                        return t1;
                    }
                    return (int) (o1.getStart() - o2.getStart());
                }
            };


            SortingCollection cltn = SortingCollection.newInstance(DotAlignedAlignment.class,
                    codec, comp, MAX_RECORDS_IN_RAM);

            DotAlignedParser parser = new DotAlignedParser();
            AsciiLineReader reader = new AsciiLineReader(fis);
            DotAlignedAlignment next = null;
            while ((next = parser.readNextRecord(reader)) != null) {
                cltn.add(next);
            }


            CloseableIterator<DotAlignedAlignment> iter = cltn.iterator();
            while (iter.hasNext()) {
                DotAlignedAlignment al = iter.next();
                writer.println(al.getChromosome() + "\t" +
                        al.getStart() + "\t" +
                        al.getEnd() + "\t" +
                        (al.isNegativeStrand() ? "-" : "+"));

            }
            iter.close();
        } catch (IOException ex) {
            ex.printStackTrace();
        } finally {
            try {
                fis.close();
                writer.close();
            } catch (IOException ex) {
            }
        }
    }
}
