/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this mappingFile except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */


package org.broad.igv.synteny;

//~--- JDK imports ------------------------------------------------------------

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

/**
 *
 * @author jrobinso
 */
public class SyntenyUtils {

    static String[] buffer = new String[50];
    static Map<String, List<Anchor>> anchors;

    /**
     * Method description
     *
     *
     * @param args
     */
    public static void main(String[] args) {
        String file = "/Users/jrobinso/mapWithUn_ph_MA4_ML100K.map";
        //String inputFile = "/Users/jrobinso/IGV/Synteny/ES.H3K4me3.aligned";
        //String outputFile = "/Users/jrobinso/IGV/Synteny/ES.H3K4me3.hg18_to_mm8.aligned";
        //loadAnchors(file);
        //convertAlignments(inputFile, outputFile);

         String inputFile = "/Users/jrobinso/hg18_to_mm8_regions.bed";
         String outputFile = "/Users/jrobinso/hg18_to_mm8_anchors.bed";
         convertRegionsToBed(file, inputFile, outputFile);

    }

    /**
     * Method description
     *
     *
     *
     * @param file
     */
    public static void loadAnchors(String file) {

        BufferedReader reader = null;
        anchors = new HashMap();

        try
        {
            reader = new BufferedReader(new FileReader(file));

            String nextLine;
            Pattern.compile("\t");
            while ((nextLine = reader.readLine()) != null)
            {

                // region R:chr1:chr2:D1 chr1 47243 59894 + chr2 111274749 111285190 +
                if (nextLine.startsWith("region"))
                {

                    // Ignore
                }
                else if (nextLine.startsWith("anchor"))
                {

                    // anchor A:chr1:3928:chr4:34422 chr1 17429161 17429302 + chr4 140098907 140099048 - 96.7
                    String[] tokens = nextLine.split(" ");

                    boolean forward = tokens[5].trim().equals(tokens[9].trim());
                    int toStart = Integer.parseInt(tokens[7]);
                    int toEnd = Integer.parseInt(tokens[8]);
                    int fromStart = Integer.parseInt(tokens[3]);
                    int fromEnd = Integer.parseInt(tokens[4]);

                    Anchor anchor = new Anchor(tokens[1] + "(" + tokens[9] + ")", tokens[2],
                                               fromStart, fromEnd, tokens[6], toStart, toEnd,
                                               tokens[9]);

                    List<Anchor> anchorList = anchors.get(anchor.getFromChr());
                    if (anchorList == null)
                    {
                        anchorList = new ArrayList(1000);
                        anchors.put(anchor.getFromChr(), anchorList);
                    }
                    anchorList.add(anchor);

                }

            }

            for (List<Anchor> anchorList : anchors.values())
            {
                sortAnchorList(anchorList);
            }

        }
        catch (IOException exception)
        {
            exception.printStackTrace();
        }
        finally
        {
            if (reader != null)
            {
                try
                {

                    reader.close();

                }
                catch (IOException iOException) {}
            }
        }
    }

    /**
     * Method description
     *
     *
     *
     * @param file
     * @param regionFile
     * @param anchorFile
     */
    public static void convertRegionsToBed(String file, String regionFile, String anchorFile) {

        BufferedReader reader = null;
        PrintWriter regionWriter = null;
        PrintWriter anchorWriter = null;

        try
        {
            regionWriter = new PrintWriter(regionFile);
            anchorWriter = new PrintWriter(anchorFile);
            reader = new BufferedReader(new FileReader(file));

            String nextLine;
            Pattern.compile("\t");
            while ((nextLine = reader.readLine()) != null)
            {

                // region R:chr1:chr2:D1 chr1 47243 59894 + chr2 111274749 111285190 +
                if (nextLine.startsWith("region"))
                {
                    String[] tokens = nextLine.split(" ");
                    regionWriter.println(tokens[6] + "\t" + tokens[7] + "\t" + tokens[8] + "\t"
                                         + tokens[1] + "\t\t" + tokens[9]);
                }
                else if (nextLine.startsWith("anchor"))
                {

                    // anchor A:chr1:3928:chr4:34422 chr1 17429161 17429302 + chr4 140098907 140099048 - 96.7
                    String[] tokens = nextLine.split(" ");

                    boolean forward = tokens[5].trim().equals(tokens[9].trim());
                    int toStart = Integer.parseInt(tokens[7]);
                    int toEnd = Integer.parseInt(tokens[8]);
                    int fromStart = Integer.parseInt(tokens[3]);
                    int fromEnd = Integer.parseInt(tokens[4]);

                    Anchor anchor = new Anchor(tokens[1] + "(" + tokens[9] + ")", tokens[2],
                                               fromStart, fromEnd, tokens[6], toStart, toEnd,
                                               tokens[9]);

                    // Center anchor
                    int start = anchor.mapPosition(fromStart);

                    // Align at end
                    // int start = (forward ? toStart : toEnd - fromSize);
                    int end = anchor.mapPosition(fromEnd);

                    anchorWriter.println(anchor.getToChr() + "\t" + start + "\t" + end + "\t"
                                         + anchor.getName() + "\t\t" + anchor.getDirection());
                }


            }


        }
        catch (IOException exception)
        {
            exception.printStackTrace();
        }
        finally
        {
            if (reader != null)
            {
                try
                {
                    regionWriter.close();
                    anchorWriter.close();
                    reader.close();

                }
                catch (IOException iOException) {}
            }
        }

    }

    /**
     * Method description
     *
     *
     * @param inputFile
     * @param outputFile
     */
    public static void convertAlignments(String inputFile, String outputFile) {

        BufferedReader reader = null;
        PrintWriter writer = null;

        try
        {
            writer = new PrintWriter(outputFile);
            reader = new BufferedReader(new FileReader(inputFile));

            String nextLine;
            while ((nextLine = reader.readLine()) != null)
            {

                // chr15     62533608        62533644        -       2041T.3.1       4
                String[] tokens = nextLine.split("\t");

                String chr = tokens[0].trim();
                int fromStart = Integer.parseInt(tokens[1]);
                int fromEnd = Integer.parseInt(tokens[2]);
                String fromStrand = tokens[3].trim();
                
                // Extend alignment -> fragment length
                if(fromStrand.equals("+")) {
                    fromEnd += 250;
                }
                else {
                    fromStart += 250;
                }

                Alignment alignment = new Alignment(chr, fromStart, fromEnd, fromStrand);

                Anchor anchor = getAnchorContaining(alignment);
                if (anchor != null)
                {
                    String mappedStrand = alignment.getStrand();
                    if (!anchor.getDirection())
                    {
                        mappedStrand = mappedStrand.equals("+") ? "-" : "+";

                    }
                    int mappedStart = anchor.mapPosition(fromStart);
                    int mappedEnd = anchor.mapPosition(fromEnd);
                    writer.println(
                       anchor.getToChr() + "\t" + mappedStart + "\t"
                                   + mappedEnd + "\t" + mappedStrand);
                }


            }


        }
        catch (Exception exception)
        {
            exception.printStackTrace();
        }
        finally
        {
            if (reader != null)
            {
                try
                {
                    writer.close();
                    reader.close();

                }
                catch (IOException iOException) {}
            }
        }

    }


    // chr15     62533608        62533644        -       2041T.3.1       4
    static class Alignment {

        String chr;
        int start;
        int end;
        String strand;

        /**
         * Constructs ...
         *
         *
         * @param chr
         * @param start
         * @param end
         * @param strand
         */
        public Alignment(String chr, int start, int end, String strand) {
            this.chr = chr;
            this.start = start;
            this.end = end;
            this.strand = strand;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public String getChr() {
            return chr;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public int getEnd() {
            return end;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public int getStart() {
            return start;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public String getStrand() {
            return strand;
        }


    }


    // anchor A:chr1:3928:chr4:34422 chr1 17429161 17429302 + chr4 140098907 140099048 - 96.7
    static class Anchor {

        private String name;
        private String fromChr;
        private int fromStart;
        private int fromEnd;
        private String toChr;
        private int toStart;
        private int toEnd;
        private boolean direction;

        int offset;

        /**
         * Constructs ...
         *
         *
         * @param name
         * @param fromChr
         * @param fromStart
         * @param fromEnd
         * @param toChr
         * @param toStart
         * @param toEnd
         * @param dir
         */
        public Anchor(String name, String fromChr, int fromStart, int fromEnd, String toChr,
                      int toStart, int toEnd, String dir) {
            this.name = name;
            this.fromChr = fromChr;
            this.fromStart = fromStart;
            this.fromEnd = fromEnd;
            this.toChr = toChr;
            this.toStart = toStart;
            this.toEnd = toEnd;
            this.direction = dir.equals("+");

            // Compute offset to center anchor
            int fromSize = Math.abs(fromEnd - fromStart);
            int toSize = Math.abs(toEnd - toStart);
            offset = (fromSize - toSize) / 2;
        }

        /**
         * Method description
         *
         *
         * @param fromPosition
         *
         * @return
         */
        public int mapPosition(int fromPosition) {

            return (direction == true)
                   ? toStart + offset + (fromPosition - fromStart)
                   : toEnd - (offset + (fromPosition - fromStart));
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public String getName() {
            return name;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public String getFromChr() {
            return fromChr;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public int getFromStart() {
            return fromStart;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public int getFromEnd() {
            return fromEnd;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public String getToChr() {
            return toChr;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public int getToStart() {
            return toStart;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public int getToEnd() {
            return toEnd;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public boolean getDirection() {
            return direction;
        }

        /**
         * Method description
         *
         *
         * @param alignment
         *
         * @return
         */
        public boolean overlaps(Alignment alignment) {
            return alignment.getChr().equals(getFromChr())
                   && ((alignment.getStart() >= getFromStart() && alignment.getStart() <= getFromEnd()) ||
                       (alignment.getEnd() >= getFromStart() && alignment.getEnd() <= getFromEnd()) ||
                       (alignment.getStart() < getFromStart() && alignment.getEnd() > getFromEnd()));
        }
    }


    /**
     * Sort the feature list by ascending start value
     *
     * @param features
     */
    public static void sortAnchorList(List<Anchor> features) {

        Collections.sort(features, new Comparator<Anchor>() {

            public int compare(Anchor o1, Anchor o2) {

                return (int) (o1.getFromStart() - o2.getFromStart());
            }
        });
    }

    /**
     * Method description
     *
     *
     * @param alignment
     *
     * @return
     */
    static Anchor getAnchorContaining(Alignment alignment) {
        List<Anchor> anchorList = anchors.get(alignment.getChr());
        if (anchorList != null)
        {
            int startIdx = getIndexBefore(anchorList, alignment.getStart());
            for (int i = startIdx; i < anchorList.size(); i++)
            {
                Anchor a = anchorList.get(i);
                if (a.getFromStart() > alignment.getEnd())
                {
                    break;
                }
                else if (a.overlaps(alignment))
                {
                    return a;
                }
            }
        }
        return null;

    }

    /**
     * Method description
     *
     *
     * @param values
     * @param x
     *
     * @return
     */
    static int getIndexBefore(List<Anchor> values, int x) {
        return getIndexBefore(values, x, 0, values.size());
    }

    /**
     * Method description
     *
     *
     * @param values
     * @param x
     * @param leftBound
     * @param rightBound
     *
     * @return
     */
    static int getIndexBefore(List<Anchor> values, int x, int leftBound, int rightBound) {

        int idx = (leftBound + rightBound) / 2;

        if ((idx == 0) || (idx == values.size() - 1))
        {
            return idx;
        }
        if (values.get(idx).getFromStart() == x)
        {
            return idx;
        }

        if (values.get(idx).getFromStart() < x)
        {
            if (values.get(idx + 1).getFromStart() >= x)
            {
                return idx;
            }
            else
            {
                leftBound = idx;
                return getIndexBefore(values, x, leftBound, rightBound);
            }
        }
        else
        {    // values[idx] > x
            if (values.get(idx - 1).getFromStart() <= x)
            {
                return idx - 1;
            }
            else
            {
                rightBound = idx;
                return getIndexBefore(values, x, leftBound, rightBound);
            }
        }
    }
}
