/*
* The Broad Institute
* SOFTWARE COPYRIGHT NOTICE AGREEMENT
* This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
* of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
* (the "License"); you may not use this file except in compliance with
*  the License.  You may obtain a copy of the License at
*
*    http://www.opensource.org/licenses/gpl-2.0.php
*
* This software is supplied without any warranty or guaranteed support
* whatsoever. Neither the Broad Institute nor MIT can be responsible for its
* use, misuse, or functionality.
*/




package org.broad.igv.synteny;

//~--- non-JDK imports --------------------------------------------------------

import org.broad.igv.data.DataUtils;
import org.broad.igv.data.WiggleDataset;
import org.broad.igv.data.WiggleParser;
import org.broad.igv.feature.FeatureUtils;
import org.broad.igv.util.ResourceLocator;

//~--- JDK imports ------------------------------------------------------------

import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.broad.igv.data.BasicScore;


/**
 *
 * @author jrobinso
 */
public class SyntenyManager {


    /**
     * Method description
     *
     *
     * @param dataset
     * @param mappings
     * @param outputFile
     */
    public static void MapWigFile(WiggleDataset dataset, List<Mapping> mappings, File outputFile) {

        PrintWriter pw = null;

        try
        {
            pw = new PrintWriter(new FileWriter(outputFile));

            // Segregate and sort ortholog
            Map<String, List<Mapping>> featureMap = new LinkedHashMap();
            for (Mapping f : mappings)
            {
                List<Mapping> fList = featureMap.get(f.getChromosome());
                if (fList == null)
                {
                    fList = new ArrayList();
                    featureMap.put(f.getChromosome(), fList);
                }
                fList.add(f);
            }

            for (List<Mapping> featureList : featureMap.values())
            {
                FeatureUtils.sortFeatureList(featureList);
            }

            pw.println("Chr\tStart\tEnd\t\tnucCount (Kwal)");
            // Loop through chromosomes
            for (String chr : featureMap.keySet())
            {
                List<Mapping> mappingList = featureMap.get(chr);
                List<BasicScore> scores = new ArrayList(mappingList.size());

                for (Mapping mapping : mappingList)
                {
                    Block queryBlock = mapping.getQueryBlock();
                    Block subjectBlock = mapping.getSubjectBlock();
                   
                    int[] subjectPosition = dataset.getStartLocations(subjectBlock.getContig());
                    float[] data = dataset.getData("ignore", subjectBlock.getContig());

                    int s0 = subjectBlock.getStart();
                    int s1 = subjectBlock.getEnd();
                    int q0 = queryBlock.getStart();
                    int q1 = queryBlock.getEnd();

                    int idx0 = DataUtils.getIndexBefore(subjectPosition, Math.min(s0, s1));
                    int idx1 = DataUtils.getIndexBefore(subjectPosition, Math.max(s0, s1)) + 1;
                    if (idx1 == subjectPosition.length)
                    {
                        idx1--;
                    }
                    
                    double beta = ((double) (q1 - q0)) / (s1 - s0);

                    for (int i = idx0; i <= idx1; i++)
                    {
                        int pos = (int) (q0 + beta * (subjectPosition[i] - s0));
                        float d = data[i];
                        scores.add(new BasicScore(chr, pos, pos+1, d));
                    }
                }
                
                FeatureUtils.sortFeatureList(scores);
                for(BasicScore s : scores) {
                        pw.println(s.getChromosome() + "\t" + s.getStart() + "\t" + 
                            (s.getEnd() + 1) + "\t\t" + s.getScore());
                    
                }
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
        finally
        {
            pw.close();
        }
    }
    
    

    /**
     * Method description
     *
     *
     * @param args
     */
    public static void main(String[] args) {

        String file = "/Users/jrobinso/Alex/Cgla_v_Kwal1.orthologs";
        String wiggleFile = "/Users/jrobinso/IGVTestData/alex/nucCount_Kwal.wig";
        File outputFile = new File("/Users/jrobinso/Alex/uncount_Kwal_to_Cgla.igv");

        List<Mapping> mappings = (new MappingParser()).parse(file);

        WiggleDataset ds = (new WiggleParser(new ResourceLocator(wiggleFile), "Kwal")).parse();

        MapWigFile(ds, mappings, outputFile);

    }




}
