/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.synteny;

//~--- JDK imports ------------------------------------------------------------
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;

import java.util.ArrayList;
import java.util.List;
import org.broad.igv.feature.Strand;

/**
 *
 * @author jrobinso
 */
public class MappingParser {

    /**
     * Method description
     *
     *queryID,subjectID,percentIden,alignmentLength,
     * mismatches,gapOpenings,queryStart,queryStop,subjectStart,subjectStop,eval,bitScore
     * 
     * 1	9	87.73	864	106	0	100842	101705	222166	221303	0	872
     * @param file
     *
     * @return
     */
    public List<Mapping> parse(String file) {

        List<Mapping> mappings = new ArrayList();

        BufferedReader reader = null;

        boolean tabblastn = file.contains(".tabblastn");

        int queryChrIdx = 0;
        int queryStartIdx = tabblastn ? 6 : 1;
        int queryEndIdx = tabblastn ? 7 : 2;
        int queryStrandIdx = tabblastn ? -1 : 3;

        int subjectChrIdx = tabblastn ? 1 : 4;
        int subjectStartIdx = tabblastn ? 8 : 5;
        int subjectEndIdx = tabblastn ? 9 : 6;
        int subjectStrandIdx = tabblastn ? -1 : 7;
        
        int nTokens = tabblastn ? 10 : 8;

        try
        {

            reader = new BufferedReader(new FileReader(file));

            String nextLine;
            while ((nextLine = reader.readLine()) != null)
            {
                String[] tokens = nextLine.split("\t");
                if (tokens.length >= nTokens)
                {
                    String queryCht = tokens[queryChrIdx];
                    int queryStart = Integer.parseInt(tokens[queryStartIdx]);
                    int queryEnd = Integer.parseInt(tokens[queryEndIdx]);
                    Strand queryStrand = Strand.NONE;
                    if(!tabblastn) {
                        int str = Integer.parseInt(tokens[queryStrandIdx]);
                        queryStrand = str == 1 ? Strand.POSITIVE : Strand.NEGATIVE;
                    }
                    
                    String subjectChr = tokens[subjectChrIdx];                    
                    int subjectStart = Integer.parseInt(tokens[subjectStartIdx]);
                    int subjectEnd = Integer.parseInt(tokens[subjectEndIdx]);
                    Strand subjectStrand = Strand.NONE;
                    if(!tabblastn) {
                        int str = Integer.parseInt(tokens[subjectStrandIdx]);
                        subjectStrand = str == 1 ? Strand.POSITIVE : Strand.NEGATIVE;
                    }
                    
                    float percentIden = tabblastn ? Float.parseFloat(tokens[2]) : 100.0f;

                    Block queryBlock = new Block(queryCht, queryStart, queryEnd, queryStrand);
                    Block subjectBlock = new Block(subjectChr, subjectStart, subjectEnd, subjectStrand);

                    mappings.add(new Mapping(queryBlock, subjectBlock, percentIden));
                }
            }


        } catch (IOException exception)
        {
        } finally
        {
            if (reader != null)
            {
                try
                {
                    reader.close();

                } catch (IOException iOException)
                {
                }
            }
        }


        return mappings;


    }
}
