/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.session;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.*;
import org.apache.log4j.Logger;

import org.broad.igv.util.ResourceLocator;
import org.broad.igv.renderer.ColorScale;
import org.broad.igv.renderer.DataRange;
import org.broad.igv.renderer.Renderer;
import org.broad.igv.renderer.RendererFactory;
import org.broad.igv.renderer.RendererFactory.RendererType;
import org.broad.igv.track.FeatureTrack;
import org.broad.igv.track.Track;
import org.broad.igv.track.TrackType;
import org.broad.igv.track.WindowFunction;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.RegionOfInterest;
import org.broad.igv.ui.TrackFilter;
import org.broad.igv.ui.TrackFilterElement;

import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;


import java.awt.Color;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringWriter;


import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.swing.JOptionPane;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.broad.igv.session.SessionManager.SessionAttribute;
import org.broad.igv.session.SessionManager.SessionElement;

/**
 *
 * @author jrobinso
 */
public class SessionWriter {

    static Logger log = Logger.getLogger(SessionWriter.class);

    /**
     * Method description
     *
     *
     * @param session
     * @param outputFile
     *
     * @throws IOException
     */
    public void saveSession(Session session, File outputFile) throws IOException {

        if (session == null)
        {
            RuntimeException e = new RuntimeException("No session found to save!");
            log.error("Session Management Error", e);
        }

        if (outputFile == null)
        {
            RuntimeException e = new RuntimeException("Can't save session file: " + outputFile);
            log.error("Session Management Error", e);
        }

        String xmlString = createXmlFromSession(session, outputFile);

        FileWriter fileWriter = null;
        try
        {
            fileWriter = new FileWriter(outputFile);
            fileWriter.write(xmlString);
        }
        finally
        {
            if (fileWriter != null)
            {
                fileWriter.close();
            }
        }
    }

    /**
     * Method description
     *
     *
     * @param session
     * @param outputFile
     *
     * @return
     *
     * @throws RuntimeException
     */
    public String createXmlFromSession(Session session, File outputFile) throws RuntimeException {

        StringBuffer stringBuffer = new StringBuffer();
        String xmlString = null;

        try
        {

            // Create a DOM document
            DocumentBuilder documentBuilder =
                DocumentBuilderFactory.newInstance().newDocumentBuilder();
            Document document = documentBuilder.newDocument();
            document.setStrictErrorChecking(true);

            // Global root element
            Element globalElement = document.createElement(SessionElement.GLOBAL.getText());

            globalElement.setAttribute(SessionAttribute.VERSION.getText(),
                session.getSessionVersion());

            String genome = session.getGenome();
            if (genome != null)
            {
                globalElement.setAttribute(SessionAttribute.GENOME.getText(), genome);
            }
            String locus = session.getLocus();
            if (locus != null)
            {
                globalElement.setAttribute(SessionAttribute.LOCUS.getText(), locus);
            }

            String groupBy = session.getGroupTracksBy();
            if (groupBy != null)
            {
                globalElement.setAttribute(SessionAttribute.GROUP_TRACKS_BY.getText(), groupBy);
            }


            // Resource Files
            Collection<ResourceLocator> resourceFiles = session.getResourceLocatorSet();
            if ((resourceFiles != null) && !resourceFiles.isEmpty())
            {

                Element filesElement = document.createElement(SessionElement.FILES.getText());

                String isRelativeDataFile = "false";
                String filepath = null;
                for (ResourceLocator resourceFile : resourceFiles)
                {

                    Element dataFileElement =
                        document.createElement(SessionElement.DATA_FILE.getText());

                    if (resourceFile.isLocal())
                    {

                        filepath = Utilities.getRelativePath(outputFile.getParentFile(),
                            resourceFile.getPath());
                        if (filepath.equals(resourceFile.getPath()))
                        {
                            isRelativeDataFile = "false";
                        } else
                        {
                            isRelativeDataFile = "true";
                        }
                        dataFileElement.setAttribute(SessionAttribute.RELATIVE_PATH.getText(),
                            isRelativeDataFile);
                    } else
                    {
                        filepath = resourceFile.getPath();
                    }

                    String platformIndependentPath = Utilities.getPlatformIndependentPath(filepath);
                    dataFileElement.setAttribute(SessionAttribute.NAME.getText(),
                        platformIndependentPath);

                    if (!resourceFile.isLocal())
                    {
                        dataFileElement.setAttribute(SessionAttribute.SERVER_URL.getText(),
                            resourceFile.getServerURL());
                    }

                    filesElement.appendChild(dataFileElement);
                }
                globalElement.appendChild(filesElement);
            }

            // Panels
            writePanels(session, stringBuffer, globalElement, document);


            // Regions of Interest
            Collection<RegionOfInterest> regions = session.getRegionsOfInterestSet();
            if ((regions != null) && !regions.isEmpty())
            {

                Element regionsElement = document.createElement(SessionElement.REGIONS.getText());
                for (RegionOfInterest region : regions)
                {
                    Element regionElement = document.createElement(SessionElement.REGION.getText());
                    regionElement.setAttribute(SessionAttribute.CHROMOSOME.getText(),
                        region.getChromosomeName());
                    regionElement.setAttribute(SessionAttribute.START_INDEX.getText(),
                        region.getStart().toString());
                    regionElement.setAttribute(SessionAttribute.END_INDEX.getText(),
                        region.getEnd().toString());
                    regionElement.setAttribute(SessionAttribute.DESCRIPTION.getText(),
                        region.getDescription());
                    regionsElement.appendChild(regionElement);
                }
                globalElement.appendChild(regionsElement);
            }

            // Filter
            TrackFilter trackFilter = session.getFilter();
            if (trackFilter != null)
            {

                Element filter = document.createElement(SessionElement.FILTER.getText());

                filter.setAttribute(SessionAttribute.NAME.getText(), trackFilter.getName());

                if (session.isFilterMatchAll())
                {
                    filter.setAttribute(SessionAttribute.FILTER_MATCH.getText(), "all");
                } else if (session.isFilterMatchAny())
                {
                    filter.setAttribute(SessionAttribute.FILTER_MATCH.getText(), "any");
                } else
                {    // Defaults to match all
                    filter.setAttribute(SessionAttribute.FILTER_MATCH.getText(), "all");
                }

                if (session.isFilterShowAllTracks())
                {
                    filter.setAttribute(SessionAttribute.FILTER_SHOW_ALL_TRACKS.getText(), "true");
                } else
                {    // Defaults
                    filter.setAttribute(SessionAttribute.FILTER_SHOW_ALL_TRACKS.getText(), "false");
                }
                globalElement.appendChild(filter);

                // Process FilterElement elements
                Iterator iterator = session.getFilter().getFilterElements();
                while (iterator.hasNext())
                {

                    TrackFilterElement trackFilterElement = (TrackFilterElement) iterator.next();

                    Element filterElementElement =
                        document.createElement(SessionElement.FILTER_ELEMENT.getText());
                    filterElementElement.setAttribute(SessionAttribute.ITEM.getText(),
                        trackFilterElement.getSelectedItem());
                    filterElementElement.setAttribute(
                        SessionAttribute.OPERATOR.getText(),
                        trackFilterElement.getComparisonOperator().getValue());
                    filterElementElement.setAttribute(SessionAttribute.VALUE.getText(),
                        trackFilterElement.getValue());
                    filterElementElement.setAttribute(
                        SessionAttribute.BOOLEAN_OPERATOR.getText(),
                        trackFilterElement.getBooleanOperator().getValue());
                    filter.appendChild(filterElementElement);
                }
            }

            // Visible Attributes
            Collection<String> attributes = session.getVisibleAttributes();
            if ((attributes != null) && !attributes.isEmpty())
            {

                Element visibleAttributesElement =
                    document.createElement(SessionElement.VISIBLE_ATTRIBUTES.getText());

                for (String visibleAttribute : attributes)
                {
                    Element visibleAttributeElement =
                        document.createElement(SessionElement.VISIBLE_ATTRIBUTE.getText());
                    visibleAttributeElement.setAttribute(SessionAttribute.NAME.getText(),
                        visibleAttribute);
                    visibleAttributesElement.appendChild(visibleAttributeElement);
                }
                globalElement.appendChild(visibleAttributesElement);
            }

            // Color Scales
            Map map = session.getColorScaleMap();
            if (map != null)
            {
                Set<Entry<TrackType, ColorScale>> entrySet = map.entrySet();
                if ((entrySet != null) && !entrySet.isEmpty())
                {

                    Element colorScalesElement =
                        document.createElement(SessionElement.COLOR_SCALES.getText());

                    for (Map.Entry<TrackType, ColorScale> entry : entrySet)
                    {
                        if (entry.getValue() == null)
                        {
                            continue;
                        }

                        Element colorScaleElement =
                            document.createElement(SessionElement.COLOR_SCALE.getText());
                        colorScaleElement.setAttribute(SessionAttribute.TYPE.getText(),
                            entry.getKey().name());
                        colorScaleElement.setAttribute(SessionAttribute.VALUE.getText(),
                            ((ColorScale) entry.getValue()).asString());
                        colorScalesElement.appendChild(colorScaleElement);
                    }
                    globalElement.appendChild(colorScalesElement);
                }
            }

            document.appendChild(globalElement);

            // Transform document into XML
            TransformerFactory factory = TransformerFactory.newInstance();
            Transformer transformer = factory.newTransformer();
            transformer.setOutputProperty(OutputKeys.INDENT, "yes");
            transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");

            StreamResult streamResult = new StreamResult(new StringWriter());
            DOMSource source = new DOMSource(document);
            transformer.transform(source, streamResult);

            xmlString = streamResult.getWriter().toString();
        } catch (Exception e)
        {
            String message = "An error has occurred while trying to create the session!";
            log.error(message, e);
            JOptionPane.showMessageDialog(IGVMainFrame.getInstance(), message);
            throw new RuntimeException(e);
        }

        return xmlString;
    }

    private void buildTrack(Track track, Element element, StringBuffer stringBuffer) {

        element.setAttribute(SessionAttribute.ID.getText(), track.getId());

        element.setAttribute(SessionAttribute.VISIBLE.getText(), String.valueOf(track.isVisible()));
        int height = track.getHeight();
        if ((track instanceof FeatureTrack) && ((FeatureTrack) track).isExpanded())
        {
            height = height / ((FeatureTrack) track).getNumberOfFeatureLevels();
        }
        String value = Integer.toString(height);
        element.setAttribute(SessionAttribute.HEIGHT.getText(), value);
        
        String displayName = track.getActualDisplayName();
        if(displayName != null) {
        element.setAttribute(SessionAttribute.DISPLAY_NAME.getText(), displayName);
            
        }
        

        Color color = track.getColor();
        if (color != null)
        {
            stringBuffer.delete(0, stringBuffer.length());
            stringBuffer.append(color.getRed());
            stringBuffer.append(",");
            stringBuffer.append(color.getGreen());
            stringBuffer.append(",");
            stringBuffer.append(color.getBlue());
            element.setAttribute(SessionAttribute.COLOR.getText(), stringBuffer.toString());
        }

        Renderer renderer = track.getRenderer();
        if (renderer != null)
        {
            RendererType type = RendererFactory.getRenderType(renderer);
            if (type != null)
            {
                element.setAttribute(SessionAttribute.RENDERER.getText(), type.name());
            }
        }

        WindowFunction wf = track.getWindowFunction();
        if (wf != null)
        {
            element.setAttribute(SessionAttribute.WINDOW_FUNCTION.getText(), wf.name());
        }

        DataRange axisDefinition = track.getAxisDefinition();
        if (axisDefinition != null)
        {
            stringBuffer.delete(0, stringBuffer.length());
            stringBuffer.append(axisDefinition.getMinimum());
            stringBuffer.append(",");
            stringBuffer.append(axisDefinition.getBaseline());
            stringBuffer.append(",");
            stringBuffer.append(axisDefinition.getMaximum());
            element.setAttribute(SessionAttribute.SCALE.getText(), stringBuffer.toString());
        }

        if (track instanceof FeatureTrack)
        {
            boolean expand = ((FeatureTrack) track).isExpanded();
            element.setAttribute(SessionAttribute.EXPAND.getText(), String.valueOf(expand));

        }
    }

    private void writePanels(Session session, StringBuffer stringBuffer, Element globalElement, Document document) throws DOMException {


        //  Panels
        for (String panel : session.getPanelNames())
        {
            List<Track> tracks = session.getTracks(panel);
            if ((tracks != null) && !tracks.isEmpty())
            {

                Element panelElement = document.createElement(SessionElement.PANEL.getText());
                panelElement.setAttribute("name", panel);

                for (Track track : tracks)
                {

                    Element dataTrackElement = document.createElement(SessionElement.TRACK.getText());
                    buildTrack(track, dataTrackElement, stringBuffer);
                    panelElement.appendChild(dataTrackElement);
                }
                globalElement.appendChild(panelElement);
            }
        }
    }
}

