/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.session;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.*;
import org.apache.log4j.Logger;

import org.broad.igv.util.FilterElement.BooleanOperator;
import org.broad.igv.util.FilterElement.Operator;
import org.broad.igv.util.ResourceLocator;
import org.broad.igv.track.TrackManager;
import org.broad.igv.ui.IGVMainFrame;
import org.broad.igv.ui.IGVModel;
import org.broad.igv.ui.RegionOfInterest;
import org.broad.igv.ui.TrackFilter;
import org.broad.igv.ui.TrackFilterElement;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.xml.sax.SAXException;

//~--- JDK imports ------------------------------------------------------------


import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import java.net.URL;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

import javax.swing.JOptionPane;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

/**
 *
 */
public class SessionManager {

    private static Logger log = Logger.getLogger(SessionManager.class);
    private static String INPUT_FILE_KEY = "INPUT_FILE_KEY";

    // Temporary values used in processing
    private Collection<String> visibleAttributes = null;
    private Collection<ResourceLocator> dataFiles;
    private Collection<ResourceLocator> missingDataFiles;
    private static Map<String, String> attributeSynonymMap = new HashMap();

    static
    {
        attributeSynonymMap.put("DATA FILE", "DATA SET");
        attributeSynonymMap.put("TRACK NAME", "NAME");
    }

    /**
     * Session Element types
     */
    public static enum SessionElement {

        PANEL("Panel"), TRACK("Track"),
        COLOR_SCALE("ColorScale"), COLOR_SCALES("ColorScales"), DATA_TRACK("DataTrack"),
        DATA_TRACKS("DataTracks"), FEATURE_TRACK("FeatureTrack"), FEATURE_TRACKS("FeatureTracks"),
        DATA_FILE("DataFile"), FILES("Files"), FILTER_ELEMENT("FilterElement"), FILTER("Filter"),
        GLOBAL("Global"), REGION("Region"), REGIONS("Regions"),
        VISIBLE_ATTRIBUTE("VisibleAttribute"), VISIBLE_ATTRIBUTES("VisibleAttributes");
        private String name;

        SessionElement(String name) {
            this.name = name;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public String getText() {
            return name;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        @Override
        public String toString() {
            return getText();
        }

        /**
         * Method description
         *
         *
         * @param value
         *
         * @return
         */
        static public SessionElement findEnum(String value) {

            if (value == null)
            {
                return null;
            } else
            {
                return SessionElement.valueOf(value);
            }
        }
    }

    /**
     * Session Attribute types
     */
    public static enum SessionAttribute {

        BOOLEAN_OPERATOR("booleanOperator"), COLOR("color"), CHROMOSOME("chromosome"), TYPE("type"),
        DESCRIPTION("description"), END_INDEX("end"), EXPAND("expand"), FILTER_MATCH("match"),
        FILTER_SHOW_ALL_TRACKS("showTracks"), GENOME("genome"), GROUP_TRACKS_BY("groupTracksBy"),
        HEIGHT("height"), ID("id"), ITEM("item"), LOCUS("locus"), NAME("name"),
        OPERATOR("operator"),
        RELATIVE_PATH("relativePath"), RENDERER("renderer"), SCALE("scale"),
        SERVER_URL("serverURL"),
        PATH("path"), SHOW_ATTRIBUTES("showAttributes"), START_INDEX("start"), VALUE("value"),
        VERSION("version"), VISIBLE("visible"), WINDOW_FUNCTION("windowFunction"),
        DISPLAY_NAME("displayName");
        private String name;

        SessionAttribute(String name) {
            this.name = name;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        public String getText() {
            return name;
        }

        /**
         * Method description
         *
         *
         * @return
         */
        @Override
        public String toString() {
            return getText();
        }

        /**
         * Method description
         *
         *
         * @param value
         *
         * @return
         */
        static public SessionAttribute findEnum(String value) {

            if (value == null)
            {
                return null;
            } else
            {
                return SessionAttribute.valueOf(value);
            }
        }
    }
    /**
     * Instance method for the Session Manager.
     */
    private static SessionManager instance;

    private SessionManager() {
    }

    /**
     * Method description
     *
     *
     * @return
     */
    synchronized public static SessionManager getInstance() {

        if (instance == null)
        {
            instance = new SessionManager();
        }
        return instance;
    }

    /**
     * Load a session from a url
     *
     * @param url
     * @param session
     *
     * @throws RuntimeException
     */
    public void loadSession(URL url, Session session) throws RuntimeException {

        InputStream inputStream = null;
        try
        {
            inputStream = new BufferedInputStream(url.openStream());
            loadSession(inputStream, session, url);
        } catch (Exception e)
        {
            log.error("Session Management Error", e);
            throw new RuntimeException(e);
        } finally
        {
            if (inputStream != null)
            {
                try
                {
                    inputStream.close();
                } catch (IOException iOException)
                {
                    log.error("Error closing session stream", iOException);
                }
            }
        }

    }

    /**
     * Load a session from a file
     *
     * @param inputStream
     * @param session
     * @param resourceName
     *
     * @throws RuntimeException
     */
    public void loadSession(InputStream inputStream, Session session, Object resourceName)
        throws RuntimeException {

        SessionManager manager = SessionManager.getInstance();
        Document document = null;
        try
        {
            document = manager.createDOMDocumentFromXmlFile(inputStream);
        } catch (Exception e)
        {
            log.error("Session Management Error", e);
            throw new RuntimeException(e);
        }

        HashMap additionalInformation = new HashMap();
        additionalInformation.put(INPUT_FILE_KEY, resourceName);

        processRootNode(session,
            document.getElementsByTagName(SessionElement.GLOBAL.getText()).item(0),
            additionalInformation);
        if (session.getFilter() != null)
        {
            IGVMainFrame.getInstance().setTrackFilter(session.getFilter());
        }
    }

    private void processRootNode(Session session, Node element, HashMap additionalInformation) {

        if ((element == null) || (session == null))
        {
            return;
        }

        String nodeName = element.getNodeName();
        if (!nodeName.equalsIgnoreCase(SessionElement.GLOBAL.getText()))
        {
            throw new RuntimeException(element + " is not the root of the xml!");
        }
        process(session, element, additionalInformation);
    }

    private void processGlobal(Session session, Element element, HashMap additionalInformation) {

        TrackManager.getInstance().clearTracks(true);
        session.setGenome(getAttribute(element, SessionAttribute.GENOME.getText()));
        session.setLocus(getAttribute(element, SessionAttribute.LOCUS.getText()));
        session.setGroupTracksBy(getAttribute(element, SessionAttribute.GROUP_TRACKS_BY.getText()));

        NodeList elements = element.getChildNodes();
        process(session, elements, additionalInformation);

    // IGVModel.getInstance().getViewContext().invalidateLocationScale();
    }

    /**
     * Process the Files element.
     *
     * The RELATIVE_PATH attribute specifies whether file paths are relative
     * or absolute.
     *
     * @param session
     * @param element
     */
    private void processFiles(Session session, Element element, HashMap additionalInformation) {

        dataFiles = new ArrayList();
        missingDataFiles = new ArrayList();
        NodeList elements = element.getChildNodes();
        process(session, elements, additionalInformation);

        if (missingDataFiles.size() > 0)
        {
            StringBuffer message = new StringBuffer();
            message.append("<html>The following data file(s) could not be located.<ul>");
            for (ResourceLocator file : missingDataFiles)
            {
                if (file.isLocal())
                {
                    message.append("<li>");
                    message.append(file.getPath());
                    message.append("</li>");
                } else
                {
                    message.append("<li>Server: ");
                    message.append(file.getServerURL());
                    message.append("  Path: ");
                    message.append(file.getPath());
                    message.append("</li>");
                }
            }
            message.append("</ul>");
            message.append("Common reasons for this include: ");
            message.append("<ul><li>The session or data files have been moved.</li> ");
            message.append(
                "<li>The data files are located on a drive that is not currently accessible.</li></ul>");
            message.append("</html>");

            JOptionPane.showMessageDialog(IGVMainFrame.getInstance(), message);
        }
        if (dataFiles.size() > 0)
        {
            // TODO -- load the tracks into a tmp collection, process elements to make updates,
            // then set permanently,
            TrackManager.getInstance().loadResources(dataFiles);
        }
        dataFiles = null;
    }

    /**
     * Process the data file element.  If relativePaths == true treat the
     * file path as relative to the session file path.  If false
     * @param session
     * @param element
     */
    private void processDataFile(Session session, Element element, HashMap additionalInformation) {

        ResourceLocator resourceLocator = null;
        String serverURL = getAttribute(element, SessionAttribute.SERVER_URL.getText());
        String filePath = getAttribute(element, SessionAttribute.NAME.getText());

        // If file is local
        if ((serverURL == null) || serverURL.trim().equals(""))
        {
            String relPathValue = getAttribute(element, SessionAttribute.RELATIVE_PATH.getText());
            boolean relativePaths = ((relPathValue != null) && relPathValue.equalsIgnoreCase("true"));
            File parent = (relativePaths ? new File(session.getPath()).getParentFile() : null);
            File file = new File(parent, filePath);
            resourceLocator = new ResourceLocator(file.getAbsolutePath());
        } else
        {    // ...else must be from Server
            resourceLocator = new ResourceLocator(serverURL, filePath);
        }

        if (resourceLocator.exists())
        {
            dataFiles.add(resourceLocator);
        } else
        {
            missingDataFiles.add(resourceLocator);
        }

        NodeList elements = element.getChildNodes();
        process(session, elements, additionalInformation);
    }

    private void processRegions(Session session, Element element, HashMap additionalInformation) {

        IGVModel.getInstance().clearRegionsOfInterest();
        NodeList elements = element.getChildNodes();
        process(session, elements, additionalInformation);
    }

    private void processRegion(Session session, Element element, HashMap additionalInformation) {

        String chromosome = getAttribute(element, SessionAttribute.CHROMOSOME.getText());
        String start = getAttribute(element, SessionAttribute.START_INDEX.getText());
        String end = getAttribute(element, SessionAttribute.END_INDEX.getText());
        String description = getAttribute(element, SessionAttribute.DESCRIPTION.getText());

        RegionOfInterest region = new RegionOfInterest(chromosome, new Integer(start),
            new Integer(end), description);
        session.addRegionOfInterest(region);

        NodeList elements = element.getChildNodes();
        process(session, elements, additionalInformation);
    }

    private void processFilter(Session session, Element element, HashMap additionalInformation) {

        String match = getAttribute(element, SessionAttribute.FILTER_MATCH.getText());
        String showAllTracks = getAttribute(element,
            SessionAttribute.FILTER_SHOW_ALL_TRACKS.getText());

        String filterName = getAttribute(element, SessionAttribute.NAME.getText());
        TrackFilter filter = new TrackFilter(filterName, null);
        additionalInformation.put(SessionElement.FILTER, filter);

        NodeList elements = element.getChildNodes();
        process(session, elements, additionalInformation);

        // Save the filter
        session.setFilter(filter);

        // Set filter properties
        if ("all".equalsIgnoreCase(match))
        {
            session.setFilterMatchAll(true);
            session.setFilterMatchAny(false);
        } else if ("any".equalsIgnoreCase(match))
        {
            session.setFilterMatchAll(false);
            session.setFilterMatchAny(true);
        }

        if ("true".equalsIgnoreCase(showAllTracks))
        {
            session.setFilterShowAllTracks(true);
        } else
        {
            session.setFilterShowAllTracks(false);
        }
    }

    private void processFilterElement(Session session, Element element,
                                       HashMap additionalInformation) {

        TrackFilter filter = (TrackFilter) additionalInformation.get(SessionElement.FILTER);
        String item = getAttribute(element, SessionAttribute.ITEM.getText());
        String operator = getAttribute(element, SessionAttribute.OPERATOR.getText());
        String value = getAttribute(element, SessionAttribute.VALUE.getText());
        String booleanOperator = getAttribute(element, SessionAttribute.BOOLEAN_OPERATOR.getText());

        TrackFilterElement trackFilterElement = new TrackFilterElement(filter, item,
            Operator.findEnum(operator), value,
            BooleanOperator.findEnum(booleanOperator));
        filter.add(trackFilterElement);

        NodeList elements = element.getChildNodes();
        process(session, elements, additionalInformation);
    }

    private void processVisibleAttributes(Session session, Element element,
                                           HashMap additionalInformation) {

        NodeList elements = element.getChildNodes();
        process(session, elements, additionalInformation);
    }

    private void processVisibleAttribute(Session session, Element element,
                                          Collection<String> visibleAttributes, HashMap additionalInformation) {

        String trackAtributeName = getAttribute(element, SessionAttribute.NAME.getText());
        visibleAttributes.add(trackAtributeName);

        // See if this attribute has synoyms.  This was added to handle the
        // situation of attribute header names changine between releases.  This
        // is not a perfect solution, and the use of this map should be minimized.
        if (attributeSynonymMap.containsKey(trackAtributeName))
        {
            visibleAttributes.add(attributeSynonymMap.get(trackAtributeName));
        }

        NodeList elements = element.getChildNodes();
        process(session, elements, additionalInformation);
    }

    private void processPanel(Session session, Element element,
                                HashMap additionalInformation) {
        String nodeName = element.getNodeName();
        String panelName = nodeName;
        if (nodeName.equalsIgnoreCase(SessionElement.PANEL.getText()))
        {
            panelName = element.getAttribute("name");
        }


        NodeList elements = element.getChildNodes();
        for (int i = 0; i < elements.getLength(); i++)
        {
            Node childNode = elements.item(i);
            if (childNode.getNodeName().equalsIgnoreCase(SessionElement.DATA_TRACK.getText()) ||
                childNode.getNodeName().equalsIgnoreCase(SessionElement.TRACK.getText()))
            {
                processTrack(session, (Element) childNode, additionalInformation, panelName);
            } else
            {
                process(session, childNode, additionalInformation);
            }
        }
    }

    private String getAttribute(Element element, String key) {

        String value = element.getAttribute(key);
        if (value != null)
        {
            if (value.trim().equals(""))
            {
                value = null;
            }
        }
        return value;

    }

    private void processTrack(Session session, Element element,
                               HashMap additionalInformation, String panelName) {

        String id = getAttribute(element, SessionAttribute.ID.getText());
        String displayName = getAttribute(element, SessionAttribute.DISPLAY_NAME.getText());
        String isVisible = getAttribute(element, SessionAttribute.VISIBLE.getText());
        String height = getAttribute(element, SessionAttribute.HEIGHT.getText());
        String color = getAttribute(element, SessionAttribute.COLOR.getText());
        String renderer = getAttribute(element, SessionAttribute.RENDERER.getText());
        String windowFunction = getAttribute(element, SessionAttribute.WINDOW_FUNCTION.getText());
        String scale = getAttribute(element, SessionAttribute.SCALE.getText());
        String expand = getAttribute(element, SessionAttribute.EXPAND.getText());
        session.updateTrack(id, displayName, isVisible, height, color, renderer, windowFunction, scale,
            expand);

        NodeList elements = element.getChildNodes();
        process(session, elements, additionalInformation);
    }


    private void processColorScales(Session session, Element element,
                                     HashMap additionalInformation) {

        NodeList elements = element.getChildNodes();
        process(session, elements, additionalInformation);
    }

    private void processColorScale(Session session, Element element,
                                    HashMap additionalInformation) {

        String trackType = getAttribute(element, SessionAttribute.TYPE.getText());
        String value = getAttribute(element, SessionAttribute.VALUE.getText());

        session.setColorScaleSet(trackType, value);

        NodeList elements = element.getChildNodes();
        process(session, elements, additionalInformation);
    }

    /**
     * Process a list of session element nodes.
     * @param session
     * @param elements
     */
    private void process(Session session, NodeList elements, HashMap additionalInformation) {
        for (int i = 0; i < elements.getLength(); i++)
        {
            Node childNode = elements.item(i);
            process(session, childNode, additionalInformation);
        }
    }

    /**
     * Process a single session element node.
     *
     * @param session
     * @param element
     */
    private void process(Session session, Node element, HashMap additionalInformation) {

        if ((element == null) || (session == null))
        {
            return;
        }

        String nodeName = element.getNodeName();
        if (true)
        {

            if (nodeName.equalsIgnoreCase(SessionElement.GLOBAL.getText()))
            {
                processGlobal(session, (Element) element, additionalInformation);
            } else if (nodeName.equalsIgnoreCase(SessionElement.FILES.getText()))
            {
                processFiles(session, (Element) element, additionalInformation);
            } else if (nodeName.equalsIgnoreCase(SessionElement.DATA_FILE.getText()))
            {
                processDataFile(session, (Element) element, additionalInformation);
            } else if (nodeName.equalsIgnoreCase(SessionElement.REGIONS.getText()))
            {
                processRegions(session, (Element) element, additionalInformation);
            } else if (nodeName.equalsIgnoreCase(SessionElement.REGION.getText()))
            {
                processRegion(session, (Element) element, additionalInformation);
            } else if (nodeName.equalsIgnoreCase(SessionElement.FILTER.getText()))
            {
                processFilter(session, (Element) element, additionalInformation);
            } else if (nodeName.equalsIgnoreCase(SessionElement.FILTER_ELEMENT.getText()))
            {
                processFilterElement(session, (Element) element, additionalInformation);
            } else if (nodeName.equalsIgnoreCase(SessionElement.VISIBLE_ATTRIBUTES.getText()))
            {
                visibleAttributes = new ArrayList();
                processVisibleAttributes(session, (Element) element, additionalInformation);
                session.setVisibleAttributes(visibleAttributes);
                visibleAttributes = null;
            } else if (nodeName.equalsIgnoreCase(SessionElement.VISIBLE_ATTRIBUTE.getText()))
            {
                processVisibleAttribute(session, (Element) element, visibleAttributes,
                    additionalInformation);
            } else if (nodeName.equalsIgnoreCase(SessionElement.COLOR_SCALES.getText()))
            {
                processColorScales(session, (Element) element, additionalInformation);
            } else if (nodeName.equalsIgnoreCase(SessionElement.COLOR_SCALE.getText()))
            {
                processColorScale(session, (Element) element, additionalInformation);
            } else if (nodeName.equalsIgnoreCase(SessionElement.DATA_TRACKS.getText()) || 
                nodeName.equalsIgnoreCase(SessionElement.FEATURE_TRACKS.getText()) ||
                nodeName.equalsIgnoreCase(SessionElement.PANEL.getText()))
            {
                processPanel(session, (Element) element, additionalInformation);

            }

        }
    }

    /**
     * Reads an xml from an input file and creates DOM document.
     *
     * @param file
     * @return
     * @throws ParserConfigurationException
     * @throws IOException
     * @throws SAXException
     */
    private Document createDOMDocumentFromXmlFile(InputStream inputStream)
        throws ParserConfigurationException, IOException, SAXException {
        DocumentBuilder documentBuilder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
        Document xmlDocument = documentBuilder.parse(inputStream);
        return xmlDocument;
    }
}
