/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.sam.reader;

import org.broad.igv.sam.*;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMFileReader;
import net.sf.samtools.SAMFileReader.ValidationStringency;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.util.CloseableIterator;
import org.apache.log4j.Logger;

/**
 * A wrapper for SamTextReader that supports query by interval.
 *
 * @author jrobinso
 */
public class SamQueryTextReader implements AlignmentQueryReader {

    static Logger log = Logger.getLogger(SamQueryTextReader.class);
    File samFile;
    SamIndex samIndex;
    FileInputStream is;
    SAMFileHeader header;

    public SamQueryTextReader(File samFile) {
        this.samFile = samFile;
        samIndex = SamUtils.getIndexFor(samFile);
        loadHeader();
    }

    public SamQueryTextReader(File samFile, File indexFile) {
        this.samFile = samFile;
        if (indexFile.exists()) {
            samIndex = new SamIndex(indexFile);
        }
    }

    public SAMFileHeader getHeader() {
        if (header != null) {
            loadHeader();
        }
        return header;
    }

    private void loadHeader() {
        SAMFileReader reader = new SAMFileReader(samFile);
        header = reader.getFileHeader();
        reader.close();
    }

    public CloseableIterator<Alignment> query(final String sequence, final int start, final int end, final boolean contained) {

        if (samIndex == null) {
            throw new java.lang.UnsupportedOperationException("SAM files must be indexed to support query methods");
        }
        if (!samIndex.containsChromosome(sequence))  {
            return EmptyAlignmentIterator.getInstance();
        }

        // If contained == false (include overlaps) we need to adjust the start to
        // ensure we get features that extend into this segment.  
        int startAdjustment = contained ? 0 : samIndex.getLongestFeature(sequence);
        int startTileNumber = Math.max(0, (start - startAdjustment)) / samIndex.getTileWidth();

        SamIndex.TileDef seekPos = samIndex.getTileDef(sequence, startTileNumber);

        if (seekPos != null) {

            try {
                // Skip to the start of the chromosome (approximate)
                is = new FileInputStream(samFile);
                is.getChannel().position(seekPos.getStartPosition());

                SAMFileReader reader = new SAMFileReader(is);
                reader.setValidationStringency(ValidationStringency.SILENT);

                CloseableIterator<SAMRecord> iter = reader.iterator();
                return new SAMQueryIterator(sequence, start, end, contained, iter);

            } catch (IOException ex) {
                log.error("Error opening sam file", ex);
            }
        }
        return  EmptyAlignmentIterator.getInstance();
    }

 
    public void close() throws IOException {
        if (is != null) {
            is.close();
        }
    }

    /**
     *  
     */
    class SAMQueryIterator implements CloseableIterator<Alignment> {

        String chr;
        int start;
        int end;
        boolean contained;
        SAMRecord currentRecord;
        CloseableIterator<SAMRecord> wrappedIterator;

        public SAMQueryIterator(String sequence, int start, int end, boolean contained,
                CloseableIterator<SAMRecord> wrappedIterator) {
            this.chr = sequence;
            this.start = start;
            this.end = end;
            this.contained = contained;
            this.wrappedIterator = wrappedIterator;
            advanceToFirstRecord();
        }

        private void advanceToFirstRecord() {
            while (wrappedIterator.hasNext()) {
                currentRecord = wrappedIterator.next();
                if (!currentRecord.getReferenceName().equals(chr)) {
                    break;
                } else if ((contained && currentRecord.getAlignmentStart() >= start) ||
                        (!contained && currentRecord.getAlignmentEnd() >= start)) {
                    break;
                }
            }
        }

        public void close() {
            wrappedIterator.close();
        }

        public boolean hasNext() {
            if (currentRecord == null || !chr.equals(currentRecord.getReferenceName())) {
                return false;
            } else {
                return contained ? currentRecord.getAlignmentEnd() <= end
                        : currentRecord.getAlignmentStart() <= end;
            }
        }

        public SamAlignment next() {
            SAMRecord ret = currentRecord;
            if (wrappedIterator.hasNext()) {
                currentRecord = wrappedIterator.next();
            } else {
                currentRecord = null;
            }
            return new SamAlignment(ret);

        }

        public void remove() {
            throw new UnsupportedOperationException("Not supported yet.");
        }
    }
}
