/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.sam.reader;

//~--- non-JDK imports --------------------------------------------------------


import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 *
 *
 * @author jrobinso
 */
public class SamIndex {

    private int tileWidth;
    private Map<String, ChromosomeIndex> chrIndeces;

    /**
     * Constructs ...
     *
     *
     * @param tileWidth
     */
    public SamIndex(int tileWidth) {
        this.tileWidth = tileWidth;
        chrIndeces = new LinkedHashMap();

    }

    /**
     */
    public SamIndex(File f) {
        try {
            chrIndeces = new LinkedHashMap();
            read(f);
        } catch (IOException ex) {
            // TODO log properly
            ex.printStackTrace();
        }

    }

    public boolean containsChromosome(String chr) {
        return chrIndeces.containsKey(chr);
    }

    public Set<String> getIndexedChromosomes() {
        return chrIndeces.keySet();
    }

    /**
     * 
     * @param chr
     * @param idx
     * @param count
     */
    public void add(String chr, long idx, int count, int longestFeature) {

        ChromosomeIndex chrIndex = chrIndeces.get(chr);
        if (chrIndex == null) {
            chrIndex = new ChromosomeIndex(longestFeature);
            chrIndeces.put(chr, chrIndex);
        }
        chrIndex.addTile(new TileDef(idx, count));

    }

    /**
     * 
     * @param chr
     * @param tile
     *
     * @return
     */
    public TileDef getTileDef(String chr, int tile) {

        ChromosomeIndex chrIdx = chrIndeces.get(chr);
        if (chrIdx == null) {

            // Todo -- throw an execption ?
            return null;
        } else {
            return chrIdx.getTileDefinition(tile);
        }
    }

    /**
     * Store a SamIndex to a stream.
     *
     * It is the responsibility of the caller  to close the stream.
     *
     *
     *
     *
     * @param f
     *
     * @throws IOException
     */
    public void store(File f) throws IOException {

        DataOutputStream dos = null;

        try {
            dos = new DataOutputStream(new BufferedOutputStream(new FileOutputStream(f)));

            dos.writeInt(getTileWidth());

            for (Map.Entry<String, ChromosomeIndex> entry : chrIndeces.entrySet()) {

                ChromosomeIndex chrIdx = entry.getValue();
                List<TileDef> tmp = chrIdx.getTileDefinitions();

                dos.writeUTF(entry.getKey());
                dos.writeInt(tmp.size());
                dos.writeInt(chrIdx.getLongestFeature());

                for (int i = 0; i < tmp.size(); i++) {
                    final SamIndex.TileDef tileDef = tmp.get(i);
                    dos.writeLong(tileDef.getStartPosition());
                    dos.writeInt(tileDef.getCount());
                }
            }
        } finally {
            dos.close();
        }

    }

    private void read(File f) throws IOException {

        DataInputStream dis = null;
        byte[] buffer = null;
        String chr = null;
        try {
            int length = (int) f.length();
            buffer = new byte[length];
            dis = new DataInputStream(new FileInputStream(f));
            dis.readFully(buffer);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            dis.close();
        }

        if (buffer != null && buffer.length > 0) {
            DataInputStream dis2 = new DataInputStream(new ByteArrayInputStream(buffer));

            tileWidth = dis2.readInt();
            while (dis2.available() != 0) {

                chr = dis2.readUTF();
                int nTiles = dis2.readInt();
                int longestFeature = dis2.readInt();

                List<TileDef> tileDefs = new ArrayList(nTiles);
                int tileNumber = 0;
                while (dis2.available() != 0 && tileNumber < nTiles) {
                    long pos = dis2.readLong();
                    int count = dis2.readInt();
                    tileDefs.add(new TileDef(pos, count));
                    tileNumber++;
                }

                chrIndeces.put(chr, new ChromosomeIndex(longestFeature, tileDefs));
            }
        }

    }
    public int getTileWidth() {
        return tileWidth;
    }

    public int getLongestFeature(String chr) {
        ChromosomeIndex tmp = this.chrIndeces.get(chr);
        if(tmp == null) {
            return 1000;
        }
        else {
            return tmp.getLongestFeature();
        }
    }

    static class ChromosomeIndex {

        private int longestFeature;
        private List<TileDef> tileDefinitions;

        ChromosomeIndex(int longestFeature) {
            this(longestFeature, new ArrayList<TileDef>());
        }

        public ChromosomeIndex(int longestFeature, List<TileDef> tileDefinitions) {
            this.longestFeature = longestFeature;
            this.tileDefinitions = tileDefinitions;
        }



        void addTile(TileDef tileDef) {
            getTileDefinitions().add(tileDef);
        }

        TileDef getTileDefinition(int i) {
            if (getTileDefinitions().isEmpty()) {
                // TODO -- throw exception ?
                return null;
            }
            int tileNumber = Math.min(i, getTileDefinitions().size() - 1);
            return getTileDefinitions().get(tileNumber);
        }

        /**
         * @return the longestFeature
         */
        public int getLongestFeature() {
            return longestFeature;
        }

        /**
         * @return the tileDefinitions
         */
        public List<TileDef> getTileDefinitions() {
            return tileDefinitions;
        }
    }

    public static class TileDef {

        private long startPosition;
        private int count;

        /**
         * Constructs ...
         *
         *
         * @param startPosition
         * @param count
         */
        public TileDef(long startPosition, int count) {
            this.startPosition = startPosition;
            this.count = count;
        }

        /**
         * @return the startPosition
         */
        public long getStartPosition() {
            return startPosition;
        }

        /**
         * @return the count
         */
        public int getCount() {
            return count;
        }
    }
}
