/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
*/

/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.broad.igv.sam.reader;

import org.broad.igv.sam.*;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMFileReader;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.util.CloseableIterator;

/**
 * Query reader for small sam files without indeces.  Reads the entire file
 * into memory.
 *
 * @author jrobinso
 */
public class SamInMemoryQueryReader implements AlignmentQueryReader {

    /** Map of chr (sequence) -> alignments */
    SAMFileHeader header;
    Map<String, List<Alignment>> alignments;

    public SamInMemoryQueryReader(File samFile) {
        init(samFile);
    }

    private void init(File samFile) {

        alignments = new HashMap();

        SAMFileReader reader = null;

        try {
            reader = new SAMFileReader(samFile);
            reader.setValidationStringency(SAMFileReader.ValidationStringency.SILENT);
            header = reader.getFileHeader();

            CloseableIterator<SAMRecord> iter = reader.iterator();
            while (iter.hasNext()) {
                SAMRecord record = iter.next();
                String chr = record.getReferenceName();
                List<Alignment> aList = alignments.get(chr);
                if (aList == null) {
                    aList = new ArrayList(10000);
                    alignments.put(chr, aList);
                }
                aList.add(new SamAlignment(record));
            }
            iter.close();

            sortAlignments();

        } finally {
            reader.close();
        }


    }

    private void sortAlignments() {
        // this sorter sorts "in place", saving memory
        //ArrayHeapObjectSorter sorter = new ArrayHeapObjectSorter();
        for (List<Alignment> aList : alignments.values()) {
            Collections.sort(aList, new Comparator<Alignment>() {

                public int compare(Alignment arg0, Alignment arg1) {
                    return arg0.getStart() - arg1.getStart();
                }
            });


        }
    }

    public void close() throws IOException {
        // No-op
    }

    public SAMFileHeader getHeader() {
        return header;
    }

    public CloseableIterator<Alignment> query(String sequence, int start, int end, boolean contained) {
        return new DumbIterator(sequence, start, end, contained);
    }

    // Really dumb and inefficient -- TODO implement index
    public class DumbIterator implements CloseableIterator<Alignment> {

        String seq;
        int start;
        int end;
        boolean contained;
        boolean finished = false;
        Iterator<Alignment> iter;
        Alignment nextRecord = null;

        public DumbIterator(String seq, int start, int end, boolean contained) {
            this.seq = seq;
            this.start = start;
            this.end = end;
            this.contained = contained;

            List<Alignment> aList = alignments.get(seq);
            if (aList == null) {
                finished = true;
            } else {
                iter = aList.iterator();
                advance();
            }

        }

        public void close() {
            finished = true;
            iter = null;
        // Ignored
        }

        public boolean hasNext() {
            return nextRecord != null;
        }

        public Alignment next() {
            Alignment ret = nextRecord;
            advance();
            return ret;
        }

        public void advance() {
            while (iter.hasNext()) {
                Alignment align = iter.next();
                if(align.getAlignmentStart() > end) {
                    nextRecord = null;
                    return;
                }
                if (contained) {
                    if (align.getAlignmentStart() >= start && align.getEnd() <= end) {
                        nextRecord = align;
                        return;
                    }
                } else {
                    if (align.getEnd() >= start && align.getAlignmentStart() <= end) {
                        nextRecord = align;
                        return;
                    }
                }
            }
            nextRecord = null;
        }

        public void remove() {
            throw new UnsupportedOperationException("Not supported yet.");
        }
    }
}
