/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.sam.reader;

import org.broad.igv.sam.*;
import java.io.IOException;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.LinkedHashMap;
import java.util.Map;

import java.util.logging.Level;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.util.CloseableIterator;
import org.apache.log4j.Logger;
import org.broad.igv.feature.ParsingUtils;
import org.broad.igv.util.AsciiLineReader;
import org.broad.igv.util.ResourceLocator;

/**
 *
 * @author jrobinso
 */
public class SamGroupQueryReader implements AlignmentQueryReader {

    private static Logger log = Logger.getLogger(SamGroupQueryReader.class);
    // Map of chromosome -> file
    Map<String, AlignmentQueryReader> readerCache;
    Map<String, ResourceLocator> locators;

    public SamGroupQueryReader(ResourceLocator locator) {
        readerCache = new Hashtable();
        init(locator);
    }

    public void close() throws IOException {
        for (AlignmentQueryReader reader : readerCache.values())
        {
            reader.close();
        }
    }

    public SAMFileHeader getHeader() {
        if (locators.isEmpty())
        {
            return null;
        }
        return getReader(locators.keySet().iterator().next()).getHeader();
    }

    private AlignmentQueryReader getReader(String chr) {
        if (readerCache.containsKey(chr))
        {
            return readerCache.get(chr);
        }

        AlignmentQueryReader reader = null;
        ResourceLocator loc = locators.get(chr);
        if (loc != null)
        {
            reader = SamQueryReaderFactory.getReader(loc);
            //reader = new SamCachingQueryReader(loc);
            readerCache.put(chr, reader);
        }
        return reader;

    }

    public CloseableIterator<Alignment> query(String sequence, int start, int end, boolean contained) {
        AlignmentQueryReader reader = getReader(sequence);
        if (reader == null)
        {
            return new EmptyIterator();
        }
        return reader.query(sequence, start, end, contained);
    }

    //  chr    path   server
    private void init(ResourceLocator locator) {
        locators = new LinkedHashMap();
        AsciiLineReader reader = null;
        try
        {
            reader = ParsingUtils.openAsciiReader(locator);
            String nextLine;
            while ((nextLine = reader.readLine()) != null)
            {
                String[] tokens = nextLine.split("\t");
                if (tokens.length < 2)
                {
                //    log.info("Unexpected # of tokens in file: " + locator.getPath() + " => " + nextLine);
                } else
                {
                    String chr = tokens[0].trim();
                    String path = tokens[1].trim();
                    String server = null;
                    if (tokens.length > 2)
                    {
                        server = tokens[2].trim();
                    }
                    locators.put(chr, new ResourceLocator(server, path));
                }
            }

        } catch (IOException ex)
        {
            log.error("Error parsing sam file group file", ex);
            throw new RuntimeException("Error parsing file: " + locator.getPath() + " (" +
                ex.getMessage());
        } finally
        {
            if (reader != null)
            {
                reader.close();
            }
        }

    }

    static public class EmptyIterator implements CloseableIterator<Alignment> {

        public void close() {

        }

        public boolean hasNext() {
            return false;
        }

        public Alignment next() {
            return null;
        }

        public void remove() {

        }
    }
}

