/*
 * The Broad Institute
 * SOFTWARE COPYRIGHT NOTICE AGREEMENT
 * This is copyright (2007-2009) by the Broad Institute/Massachusetts Institute
 * of Technology.  It is licensed to You under the Gnu Public License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *    http://www.opensource.org/licenses/gpl-2.0.php
 *
 * This software is supplied without any warranty or guaranteed support
 * whatsoever. Neither the Broad Institute nor MIT can be responsible for its
 * use, misuse, or functionality.
 */
package org.broad.igv.sam.reader;

//~--- non-JDK imports --------------------------------------------------------
import org.broad.igv.util.AsciiLineReader;

//~--- JDK imports ------------------------------------------------------------

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import net.sf.samtools.SAMFileHeader;
import net.sf.samtools.SAMFileReader;
import net.sf.samtools.SAMFileReader.ValidationStringency;
import net.sf.samtools.SAMFileWriter;
import net.sf.samtools.SAMFileWriterFactory;
import net.sf.samtools.SAMRecord;
import net.sf.samtools.SAMSequenceRecord;
import net.sf.samtools.util.CloseableIterator;

/**
 *
 * @author jrobinso
 */
public class SAMReaderTest {

    /**
     * Method description
     *
     *
     * @param args
     *
     * @throws IOException
     */
    public static void main(String[] args) throws IOException {

        //File f = new File("/Volumes/ifs/seq/dirseq/tcga_freeze3/all_tumors.no_dupes.sam");
        //File f = new File("/Volumes/igv/sam/1kg/NA12892.chrom1.SRP000032.2009_02.bam");
        File f = new File(args[0]);
        File f2 = new File(args[1]);
        boolean xonly = args.length > 2 && args[2].equals("-xOnly");
        int nLimit = args.length < 3 ? -1 : Integer.parseInt(args[3]);
        extractRearrangments(f, f2, xonly, nLimit);
    //File f = new File("/Users/jrobinso/IGV/Sam/NA11918.withdups.bam");
    //File f = new File("/Users/jrobinso/IGV/Sam/NA12003.100k.sam");

    //readIndex(new File("NA11918.withdups.bam.index"));
    //createBamIndex(f);
    //testReadSpeed(new File("/Users/jrobinso/IGV/Sam/NA12003.100k.sam"));
    //createIndex(f);
    //testReadBam(f);
    //testSeekBam(f, new File("NA11918.withdups.bam.index"));

    //testQueryBam(new File("/Users/jrobinso/IGV/Sam/303KY.8.paired.bam"));
    // jumpToChromosome(f, "chrX");
    //testTileManager(f);

    }

    public static void testQuerySamFile(File f) {
        //          public CloseableIterator<SAMRecord> query(final String sequence, final int start, final int end, final boolean contained) {
    }

    public static boolean isPutatitveXLocation(SAMRecord alignment) {
        return alignment.getMappingQuality() > 0 &&
                alignment.getReadPairedFlag() &&
                alignment.getReadPairedFlag() &&
                !alignment.getReadUnmappedFlag() &&
                !alignment.getReferenceName().equals(alignment.getMateReferenceName());
    }
    public static boolean isPutatitveRerrangement(SAMRecord alignment) {
        return alignment.getMappingQuality() > 0 &&
                alignment.getReadPairedFlag() &&
                alignment.getReadPairedFlag() &&
                !alignment.getReadUnmappedFlag() &&
                (!alignment.getReferenceName().equals(alignment.getMateReferenceName()) ||
                Math.abs(alignment.getInferredInsertSize()) > 10000);
    }

    public static void extractRearrangments(File f, File outputFile, boolean xOnly, int nLimit) {
        SAMFileReader.setDefaultValidationStringency(SAMFileReader.ValidationStringency.SILENT);
        SAMFileReader reader = new SAMFileReader(f);
        SAMFileHeader header = reader.getFileHeader();

        SAMFileWriter newWriter = (new SAMFileWriterFactory()).makeBAMWriter(header, true, outputFile);

        CloseableIterator<SAMRecord> iter = reader.iterator();

        int count = 0;
        while (iter.hasNext() && (nLimit < 0 || count < nLimit)) {
            SAMRecord record = iter.next();
            if(isPutatitveXLocation(record) || (!xOnly && isPutatitveRerrangement(record))) {
            //if (isPutatitveXLocation(record)) {
                newWriter.addAlignment(record);
            }
            count++;
        }

        newWriter.close();

    }

    public static void testReadRecords(File f, int nRecords) {
        SAMFileReader.setDefaultValidationStringency(SAMFileReader.ValidationStringency.SILENT);
        SAMFileReader reader = new SAMFileReader(f);
        SAMFileHeader header = reader.getFileHeader();
        for (SAMSequenceRecord rec : header.getSequenceDictionary().getSequences()) {
            System.out.println(rec.getSequenceName());
        }
        CloseableIterator<SAMRecord> iter = reader.iterator();
        int n = 0;
        long t0 = System.currentTimeMillis();
        while (iter.hasNext() && n <= nRecords) {
            SAMRecord record = iter.next();
            record.getAlignmentStart();
            record.getReadBases();
            n++;
        //System.out.println(record.getReferenceName());
        }
        System.out.println("Read " + n + " records in " + (System.currentTimeMillis() - t0) + " ms");

    }

    public static void readIndex(File f) {
        SamIndex idx = new SamIndex(f);
        System.out.println(idx.getTileDef("chr21", 0).getStartPosition());
    }

    private static void testQueryBam(File f) throws FileNotFoundException, IOException {
        SAMFileReader reader = new SAMFileReader(f);
        reader.setValidationStringency(ValidationStringency.SILENT);
        //reader.setValidationStringency(ValidationStringency.SILENT);
        //chr1:16,780,602-16,780,774
        //chr1:235675392
        CloseableIterator<SAMRecord> iter = reader.query("chr1", 235675390, 235675399, true);
        while (iter.hasNext()) {
            SAMRecord record = iter.next();
            System.out.println(record.getAlignmentStart());
        }
    }

    /*chr1 453
    chr2 9374
    chr3 35726
    chr4 1735
    chr5 69060
    chr6 88667
    chr7 69480
    chr8 10473
    chr9 2062
    chr10 67533
    chr11 132463
    chr12 24781
    chr13 17937496
    chr14 18072058
    chr15 18262913
    chr16 7011
    chr17 3697
    chr18 69
    chr19 40746
    chr20 8555
    chr21 9719951
    chr22 14433767
    chrX 2709596
    chrY 3141261
    chr4 191262143
    chr5 69060
    chr5 71188
    chr5 71251
    chr5 77042
    chr5 77042
    chr5 77079*/
    @SuppressWarnings("empty-statement")
    private static void jumpToChromosome(File f, String chr, int tile)
            throws FileNotFoundException, IOException {

        File idxFile = new File(f.getAbsolutePath() + ".index");
        SamIndex samIndex = null;

        if (idxFile.exists()) {
            samIndex = new SamIndex(idxFile);
        } else {
            System.out.println("IDX file does not exist");
            return;
        }

        SamIndex.TileDef seekPos = samIndex.getTileDef(chr, tile);

        // Skip to the start of the chromosome (approximate)
        FileInputStream is = new FileInputStream(f);
        is.getChannel().position(seekPos.getStartPosition());

        SAMFileReader reader = new SAMFileReader(is);
        CloseableIterator<SAMRecord> iter = reader.iterator();

        try {
            long t0 = System.currentTimeMillis();
            while (iter.hasNext()) {
                SAMRecord record = iter.next();
                System.out.println(record.getAlignmentStart());
            }
        } catch (Exception ex) {
            System.out.println(ex);
        }
        is.close();

    }

    private static void testReadSpeed(File f) throws FileNotFoundException, IOException {

        BufferedReader br = new BufferedReader(new FileReader(f), 512000);

        long t0 = System.currentTimeMillis();
        while (br.readLine() != null) {
        }
        long dt = System.currentTimeMillis() - t0;
        System.out.println("BR = " + dt);
        br.close();

        DataInputStream dis = new DataInputStream(new BufferedInputStream(new FileInputStream(f)));
        t0 = System.currentTimeMillis();
        while (dis.readLine() != null) {
        }
        dt = System.currentTimeMillis() - t0;
        System.out.println("DIS = " + dt);
        dis.close();

        FileInputStream is = new FileInputStream(f);
        InputStream bis = new BufferedInputStream(is);
        t0 = System.currentTimeMillis();
        AsciiLineReader utils = new AsciiLineReader(bis);

        String nextLine = "";
        while ((nextLine = utils.readLine()) != null) {
            //    System.out.println(nextLine);
        }
        dt = System.currentTimeMillis() - t0;
        System.out.println("AsciiLineReader = " + dt);
        bis.close();


    }
}
